#!/usr/bin/scotty

# copyright 2001 Derek Chen-Becker <dbecker@cpicorp.com>
#
# permission is hereby granted to modify/use this software
# as long as the original author(s) are credited.
#
# There is no warranty with this software, either written or
# implied.
#

set portIfIndex 1.3.6.1.4.1.9.5.1.4.1.1.11
set portName 1.3.6.1.4.1.9.5.1.4.1.1.4
set portAdminSpeed 1.3.6.1.4.1.9.5.1.4.1.1.9
set portDuplex 1.3.6.1.4.1.9.5.1.4.1.1.10
set portType 1.3.6.1.4.1.9.5.1.4.1.1.5
set portOperStatus 1.3.6.1.4.1.9.5.1.4.1.1.6

set ifIn [mib oid ifInOctets]
set ifOut [mib oid ifOutOctets]

proc OIDOf { var } { return [lindex $var 0] }
proc valOf { var } { return [lindex $var 2] }
proc DSvalOf { var } { return [mib format sysDescr [lindex $var 2]] }

if {$argc < 2 || $argc > 3} {
        puts ""
	puts "Generate MRTG config and html pages for interfaces on an"
	puts "SNMP device with ifTable and ifMIB capabilities. Known to"
	puts "work on Cisco IOS, Foundry switches. Running this script"
	puts "automatically generates:"
	puts ""
	puts "mrtg.cfg	Config file for all interfaces"
	puts "index.html	Index file for all interfaces"
	puts "daily.html	page of all of the daily interface graphs"
	puts "weekly.html	page of all of the weekly interface graphs"
	puts "monthly.html	page of all of the monthly interface graphs"
	puts ""
	puts "On Cisco IOS and foundry switches this includes the interface"
	puts "description in all pages if it is set."
	puts ""
	puts "Usage: $argv0 <ip> <community> \[<work directory>\]"
        exit
}

set addr [lindex $argv 0]
set comm [lindex $argv 1]
if { [lindex $argv 2] != "" } {
	set workDir [lindex $argv 2]
	} else {
	set workDir "."
}

set sess [snmp session -address $addr -community $comm]

# Check to see if the mib is loaded. The Octet String has a Display Hint
# that turns the Hex into Ascii in the mib. If the mib isn't present, we need
# to format the returned portName value. We'll use a "virtual function" to
# handle it.
if [catch "mib oid portName"] {
	# no mib.
	set nameParse DSvalOf
	} else {
	set nameParse valOf
}

$sess walk entry "ifIndex ifDescr ifAlias ifSpeed ifAdminStatus ifOperStatus" {
	set portID [string tolower [join [split [valOf [lindex $entry 1]] "/" ] "-"]]
        lappend portList $portID
	#regexp "$portIfIndex\.(\[0-9\]+\.\[0-9\]+)" [OIDOf [lindex $entry 0]] whole portID
        #lappend portList $portID
	# parse out the name, using the correct format (see above)
        set portAttr($portID,name) [valOf [lindex $entry 2]]
        #set portAttr($portID,name) [string trim [$nameParse [lindex $entry 1]] " <>"]
        set portAttr($portID,index) [valOf [lindex $entry 0]]
	#puts [valOf [lindex $entry 2]]
        #if { [valOf [lindex $entry 2]] == "autoDetect" || [valOf [lindex $entry 2]] == "1" } {
        #	set portAttr($portID,speed) "auto"
        #        } else {
        #set portAttr($portID,speed) [expr [string range [valOf [lindex $entry 2]] 1 end] / 1000000]
        #}
        set portAttr($portID,speed) "-"
        #set portAttr($portID,duplex) [valOf [lindex $entry 3]]
        set portAttr($portID,duplex) "-"
	#set portAttr($portID,type) [valOf [lindex $entry 4]]
        set portAttr($portID,type) "-"
	#set otherInfo [$sess get "ifSpeed.$portAttr($portID,index) ifAdminStatus.$portAttr($portID,index) ifOperStatus.$portAttr($portID,index)"]
	#set operspeed [expr [valOf [lindex $otherInfo 0]] / 1000000]
	set portAttr($portID,operspeed) [expr [valOf [lindex $entry 3]] / 1000000]
	set portAttr($portID,adminstatus) [valOf [lindex $entry 4]]
	set portAttr($portID,operstatus) [valOf [lindex $entry 5]]
}

$sess destroy

#sort the port list...
set sortedList [lsort -dictionary $portList]

set output [open "$workDir/mrtg.cfg" w]

#print the mrtg.cfg header
puts $output "# Generated [clock format [clock seconds]] by $tnm(user) using script \"$argv0\""
puts $output "WorkDir: $workDir"
#puts $output "Refresh: 600"
#puts $output "Interval: 10"
#100Base-X max speed = 12,500,000 Bytes/sec
puts $output "MaxBytes\[_\]: 12500000"
#We don't need the yearly data yet...
puts $output "Suppress\[_\]: y"
puts $output "Options\[_\]: growright, bits"

#for each entry, output a target...
foreach target $sortedList {
	puts $output "Target\[$target\]: $ifIn.$portAttr($target,index)&$ifOut.$portAttr($target,index):$comm@$addr"
        puts $output "Title\[$target\]: PortMon for $target ($portAttr($target,name))"
        puts $output "PageTop\[$target\]: <h3>Analysis of $target ($portAttr($target,name))<\h3>"
}

close $output

#now make the html index...
set output [open "$workDir/index.html" w]

puts $output "<html><title>Port monitoring on $addr</title><body>"
puts $output "<h2>Port monitoring on $addr</h2>"
puts $output "Please select from the following list of port names to"
puts $output "view the associated graph."
puts $output "<ul>"
puts $output "<li><a href=\"topN.html\">Top talkers</a>"
puts $output "<li><a href=\"daily.html\">Daily graphs</a>"
puts $output "<li><a href=\"weekly.html\">Weekly graphs</a>"
puts $output "<li><a href=\"monthly.html\">Monthly graphs</a>"
puts $output "</ul>"
puts $output ""
puts $output "<table width=\"100%\">"
puts $output "<tr align=\"left\"><th>Port<th>Name<th>Admin Status<th>Oper Status<th>Admin Speed<th>Oper Speed<th>Duplex<th>Type</tr>"
foreach target $sortedList {
	puts $output "<tr><td><a href=\"$target.html\">($target)<td>$portAttr($target,name)</a><td>$portAttr($target,adminstatus)<td>$portAttr($target,operstatus)<td>$portAttr($target,speed)<td>$portAttr($target,operspeed)<td>$portAttr($target,duplex)<td>$portAttr($target,type)</tr>"
}
puts $output "</table>"
puts $output "<hr>"
puts $output "Unauthorized use prohibited. Questions, comments, contact <a href=\"mailto:dbecker@cpicorp.com\"><address>dbecker@cpicorp.com</address></a>"

close $output

#and the daily index
set output [open "$workDir/daily.html" w]

puts $output "<html><title>Daily graphs for $addr</title><body>"
puts $output "<h2>Daily port traffic on $addr</h2>"
foreach target $sortedList {
	puts $output "<h4>($target) $portAttr($target,name) ($portAttr($target,speed)/$portAttr($target,duplex))</h4>"
	puts $output "<a href=\"$target.html\"><img src=\"$target-day.png\" alt=\"$target\"></a><br>"
}
puts $output "<hr>"
puts $output "Unauthorized use prohibited. Questions, comments, contact <a href=\"mailto:dbecker@cpicorp.com\"><address>dbecker@cpicorp.com</address></a>"

close $output

#and the monthly index
set output [open "$workDir/monthly.html" w]

puts $output "<html><title>Monthly graphs for $addr</title><body>"
puts $output "<h2>Monthly port traffic on $addr</h2>"
foreach target $sortedList {
        puts $output "<h4>($target) $portAttr($target,name) ($portAttr($target,speed)/$portAttr($target,duplex))</h4>"
        puts $output "<a href=\"$target.html\"><img src=\"$target-month.png\" alt=\"$target\"></a><br>"
}
puts $output "<hr>"
puts $output "Unauthorized use prohibited. Questions, comments, contact <a href=\"mailto:dbecker@cpicorp.com\"><address>dbecker@cpicorp.com</address></a>"

close $output

#and the Weekly index
set output [open "$workDir/weekly.html" w]

puts $output "<html><title>Weekly graphs for $addr</title><body>"
puts $output "<h2>Weekly port traffic on $addr</h2>"
foreach target $sortedList {
        puts $output "<h4>($target) $portAttr($target,name) ($portAttr($target,speed)/$portAttr($target,duplex))</h4>"
        puts $output "<a href=\"$target.html\"><img src=\"$target-week.png\" alt=\"$target\"></a><br>"
}
puts $output "<hr>"
puts $output "Unauthorized use prohibited. Questions, comments, contact <a href=\"mailto:dbecker@cpicorp.com\"><address>dbecker@cpicorp.com</address></a>"

close $output
exit
