/*
 * Copyright 2018 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the Apache License 2.0 (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */

/*
 * Small subset of DRBG test vectors from:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/
 * The index in the names given below (e.g- kat1680)- refers to the CAVS index.
 */

#include <openssl/obj_mac.h>
#include "internal/nelem.h"
#include "drbg_cavs_data.h"


static const unsigned char kat1_nor_entropyin[] = {
    0x13, 0x6c, 0xf1, 0xc1, 0x74, 0xe5, 0xa0, 0x9f, 0x66, 0xb9, 0x62, 0xd9,
    0x94, 0x39, 0x65, 0x25,
};
static const unsigned char kat1_nor_nonce[] = {
    0xff, 0xf1, 0xc6, 0x64, 0x5f, 0x19, 0x23, 0x1f,
};
static const unsigned char kat1_nor_persstr[] = {0};
static const unsigned char kat1_nor_addin0[] = {0};
static const unsigned char kat1_nor_addin1[] = {0};
static const unsigned char kat1_nor_retbytes[] = {
    0x0e, 0x28, 0x13, 0x0f, 0xa5, 0xca, 0x11, 0xed, 0xd3, 0x29, 0x3c, 0xa2,
    0x6f, 0xdb, 0x8a, 0xe1, 0x81, 0x06, 0x11, 0xf7, 0x87, 0x15, 0x08, 0x2e,
    0xd3, 0x84, 0x1e, 0x74, 0x86, 0xf1, 0x66, 0x77, 0xb2, 0x8e, 0x33, 0xff,
    0xe0, 0xb9, 0x3d, 0x98, 0xba, 0x57, 0xba, 0x35, 0x8c, 0x13, 0x43, 0xab,
    0x2a, 0x26, 0xb4, 0xeb, 0x79, 0x40, 0xf5, 0xbc, 0x63, 0x93, 0x84, 0x64,
    0x1e, 0xe8, 0x0a, 0x25, 0x14, 0x03, 0x31, 0x07, 0x62, 0x68, 0xbd, 0x1c,
    0xe7, 0x02, 0xad, 0x53, 0x4d, 0xda, 0x0e, 0xd8,
};
static const struct drbg_kat_no_reseed kat1_nor_t = {
    0, kat1_nor_entropyin, kat1_nor_nonce, kat1_nor_persstr,
    kat1_nor_addin0, kat1_nor_addin1, kat1_nor_retbytes
};
static const struct drbg_kat kat1_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat1_nor_t
};

static const unsigned char kat2_nor_entropyin[] = {
    0xdf, 0xed, 0x69, 0xb0, 0x89, 0x02, 0xfc, 0xfb, 0x79, 0x5d, 0x5d, 0x35,
    0xdb, 0xe2, 0x3f, 0x6b,
};
static const unsigned char kat2_nor_nonce[] = {
    0x37, 0x25, 0x8e, 0x82, 0x04, 0x32, 0xe3, 0x92,
};
static const unsigned char kat2_nor_persstr[] = {0};
static const unsigned char kat2_nor_addin0[] = {0};
static const unsigned char kat2_nor_addin1[] = {0};
static const unsigned char kat2_nor_retbytes[] = {
    0xad, 0xcb, 0x8e, 0x2c, 0xbb, 0xc5, 0x95, 0x7d, 0x53, 0x8a, 0x20, 0xdb,
    0x18, 0xb5, 0xe7, 0xfe, 0x35, 0x0a, 0x90, 0xa2, 0x01, 0x35, 0x9f, 0xab,
    0x9e, 0x0f, 0x15, 0x4c, 0x53, 0xaa, 0x14, 0x6b, 0xc6, 0xaf, 0x1f, 0xcc,
    0x7f, 0xf8, 0xf3, 0x30, 0xb8, 0xd9, 0xf3, 0xd7, 0xb0, 0x38, 0x48, 0x8b,
    0xa6, 0x27, 0xe6, 0xfa, 0x21, 0xd0, 0x14, 0x73, 0x77, 0xb1, 0x34, 0x04,
    0x22, 0xb2, 0x26, 0x34, 0xb4, 0x12, 0xda, 0xc6, 0x9a, 0xc8, 0x2c, 0x35,
    0xb5, 0xfb, 0x41, 0x1a, 0x4e, 0x42, 0xa1, 0x33,
};
static const struct drbg_kat_no_reseed kat2_nor_t = {
    1, kat2_nor_entropyin, kat2_nor_nonce, kat2_nor_persstr,
    kat2_nor_addin0, kat2_nor_addin1, kat2_nor_retbytes
};
static const struct drbg_kat kat2_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat2_nor_t
};

static const unsigned char kat3_nor_entropyin[] = {
    0xf3, 0x2e, 0x36, 0xff, 0x80, 0x98, 0xf9, 0x32, 0x03, 0x5c, 0xae, 0x42,
    0x9c, 0x0e, 0xce, 0x72,
};
static const unsigned char kat3_nor_nonce[] = {
    0x55, 0x52, 0xce, 0xb1, 0x82, 0x32, 0x4b, 0x49,
};
static const unsigned char kat3_nor_persstr[] = {0};
static const unsigned char kat3_nor_addin0[] = {0};
static const unsigned char kat3_nor_addin1[] = {0};
static const unsigned char kat3_nor_retbytes[] = {
    0xba, 0xc9, 0x4b, 0xc1, 0x79, 0x7b, 0x20, 0x9b, 0x8e, 0x57, 0x67, 0x09,
    0xd6, 0x8d, 0x92, 0xa9, 0x4a, 0xa8, 0xa5, 0xf4, 0x14, 0xfe, 0xee, 0x3e,
    0x87, 0xd2, 0xf2, 0x1f, 0xcb, 0x59, 0x39, 0x31, 0x4b, 0x1b, 0x47, 0x11,
    0x27, 0x40, 0x77, 0xe7, 0x94, 0xe8, 0xe3, 0x85, 0xa2, 0x8f, 0xab, 0xb5,
    0xb4, 0x90, 0x81, 0xef, 0x14, 0xcb, 0x13, 0x96, 0xcb, 0x12, 0x54, 0x2f,
    0xd1, 0xb4, 0x8c, 0x2a, 0xd4, 0xdf, 0x98, 0xf7, 0x4b, 0xf0, 0x83, 0x46,
    0xf0, 0xa2, 0x9f, 0xf7, 0x08, 0x8d, 0xcb, 0xfe,
};
static const struct drbg_kat_no_reseed kat3_nor_t = {
    2, kat3_nor_entropyin, kat3_nor_nonce, kat3_nor_persstr,
    kat3_nor_addin0, kat3_nor_addin1, kat3_nor_retbytes
};
static const struct drbg_kat kat3_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat3_nor_t
};

static const unsigned char kat4_nor_entropyin[] = {
    0x21, 0x3e, 0x90, 0x22, 0x89, 0x8a, 0xa0, 0x5c, 0x7a, 0x6a, 0x01, 0xa4,
    0xb8, 0xd3, 0x43, 0xcc,
};
static const unsigned char kat4_nor_nonce[] = {
    0x7b, 0x69, 0x81, 0xd7, 0xe1, 0x6c, 0x98, 0x80,
};
static const unsigned char kat4_nor_persstr[] = {0};
static const unsigned char kat4_nor_addin0[] = {0};
static const unsigned char kat4_nor_addin1[] = {0};
static const unsigned char kat4_nor_retbytes[] = {
    0xa7, 0x49, 0x06, 0x34, 0x0d, 0x75, 0x14, 0xdb, 0x5b, 0x44, 0x86, 0x01,
    0x70, 0xa5, 0xe0, 0x73, 0x3c, 0x67, 0xd3, 0xce, 0x7e, 0x51, 0xbd, 0x32,
    0xb0, 0x49, 0x80, 0x0d, 0x8e, 0x3c, 0x36, 0xc8, 0x45, 0x56, 0x00, 0x84,
    0xde, 0xcc, 0x79, 0xe6, 0xf7, 0xb4, 0xd6, 0xf2, 0xf1, 0x27, 0x59, 0x2e,
    0x64, 0x58, 0x91, 0x79, 0x2d, 0x0f, 0x56, 0x2b, 0x67, 0xbc, 0x8a, 0x1f,
    0x63, 0xaa, 0x14, 0x9d, 0x84, 0x9d, 0xe1, 0xfe, 0x03, 0x28, 0x25, 0xf3,
    0xa5, 0xd4, 0x6f, 0xe3, 0x70, 0x50, 0x24, 0xeb,
};
static const struct drbg_kat_no_reseed kat4_nor_t = {
    3, kat4_nor_entropyin, kat4_nor_nonce, kat4_nor_persstr,
    kat4_nor_addin0, kat4_nor_addin1, kat4_nor_retbytes
};
static const struct drbg_kat kat4_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat4_nor_t
};

static const unsigned char kat5_nor_entropyin[] = {
    0x09, 0x1b, 0xa7, 0x85, 0xfa, 0xc4, 0x66, 0xb4, 0x85, 0x1e, 0xa2, 0xdc,
    0xa8, 0xb3, 0x8b, 0xcd,
};
static const unsigned char kat5_nor_nonce[] = {
    0x90, 0x12, 0xcb, 0xb0, 0xf9, 0xb5, 0x9c, 0xac,
};
static const unsigned char kat5_nor_persstr[] = {0};
static const unsigned char kat5_nor_addin0[] = {0};
static const unsigned char kat5_nor_addin1[] = {0};
static const unsigned char kat5_nor_retbytes[] = {
    0x39, 0x9a, 0xe6, 0x1f, 0xe5, 0x31, 0x44, 0x8c, 0x82, 0x06, 0xd7, 0xac,
    0x12, 0x7b, 0x6a, 0x44, 0x86, 0xad, 0xae, 0xbe, 0xff, 0x35, 0x67, 0xb6,
    0xbc, 0xd0, 0xf5, 0x54, 0xb8, 0x2a, 0x6a, 0xa6, 0x33, 0x7f, 0x73, 0x11,
    0x46, 0x59, 0x46, 0x91, 0x22, 0x83, 0xdc, 0x3f, 0xb0, 0x5c, 0xf8, 0x9a,
    0x74, 0xad, 0x0a, 0x18, 0x51, 0xc5, 0xa2, 0x00, 0x26, 0xca, 0x99, 0x0f,
    0x11, 0x84, 0xa8, 0xa9, 0x97, 0x8a, 0xad, 0x2e, 0x0c, 0x40, 0xb6, 0xc4,
    0xe3, 0x57, 0x1b, 0xfa, 0x0f, 0x14, 0x43, 0x29,
};
static const struct drbg_kat_no_reseed kat5_nor_t = {
    4, kat5_nor_entropyin, kat5_nor_nonce, kat5_nor_persstr,
    kat5_nor_addin0, kat5_nor_addin1, kat5_nor_retbytes
};
static const struct drbg_kat kat5_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat5_nor_t
};

static const unsigned char kat6_nor_entropyin[] = {
    0xff, 0x07, 0x6a, 0x63, 0x0f, 0x39, 0xbd, 0x94, 0xd0, 0x7e, 0x88, 0xd0,
    0x23, 0x2d, 0xd5, 0x6e,
};
static const unsigned char kat6_nor_nonce[] = {
    0xf5, 0x1a, 0x8c, 0x5d, 0x4c, 0xd4, 0x13, 0x1f,
};
static const unsigned char kat6_nor_persstr[] = {0};
static const unsigned char kat6_nor_addin0[] = {0};
static const unsigned char kat6_nor_addin1[] = {0};
static const unsigned char kat6_nor_retbytes[] = {
    0x99, 0x1d, 0x77, 0xb5, 0x15, 0xbb, 0x73, 0xde, 0xc2, 0x22, 0x6c, 0x24,
    0x86, 0xe6, 0x34, 0xd6, 0x53, 0x1a, 0xaa, 0x6c, 0xef, 0x9f, 0x64, 0x6f,
    0xba, 0x5f, 0xdd, 0x02, 0x10, 0x83, 0x4c, 0x73, 0x0b, 0x7f, 0x2f, 0xa1,
    0x10, 0xab, 0x06, 0x1c, 0xfb, 0xb0, 0x8a, 0x95, 0xb3, 0x69, 0x4b, 0x62,
    0xe8, 0x1d, 0x0c, 0x6b, 0x82, 0xcf, 0x29, 0x4e, 0xa6, 0x2f, 0xde, 0xc0,
    0x94, 0xc3, 0xed, 0x78, 0x34, 0xfb, 0xf9, 0xb8, 0x76, 0xf1, 0xa6, 0x3a,
    0x58, 0x3e, 0x2e, 0xa4, 0x2a, 0x58, 0xff, 0x74,
};
static const struct drbg_kat_no_reseed kat6_nor_t = {
    5, kat6_nor_entropyin, kat6_nor_nonce, kat6_nor_persstr,
    kat6_nor_addin0, kat6_nor_addin1, kat6_nor_retbytes
};
static const struct drbg_kat kat6_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat6_nor_t
};

static const unsigned char kat7_nor_entropyin[] = {
    0x0f, 0x67, 0x9e, 0x66, 0x2f, 0x08, 0x19, 0x7e, 0x76, 0x45, 0x76, 0x39,
    0x19, 0xba, 0x08, 0xe8,
};
static const unsigned char kat7_nor_nonce[] = {
    0x18, 0x7e, 0x82, 0xeb, 0x84, 0x4b, 0x32, 0xc2,
};
static const unsigned char kat7_nor_persstr[] = {0};
static const unsigned char kat7_nor_addin0[] = {0};
static const unsigned char kat7_nor_addin1[] = {0};
static const unsigned char kat7_nor_retbytes[] = {
    0xff, 0x63, 0x9c, 0x15, 0xf8, 0x73, 0xd3, 0x50, 0xf1, 0x77, 0x8f, 0x1e,
    0x9e, 0xa0, 0xa5, 0x61, 0x4e, 0xb2, 0x60, 0xac, 0x45, 0x3f, 0x5c, 0x57,
    0x30, 0xbb, 0x0b, 0xe9, 0x07, 0x2a, 0x01, 0x3f, 0x15, 0x96, 0xae, 0x26,
    0x6c, 0xc0, 0x9f, 0xf1, 0xc8, 0xa7, 0xfe, 0x52, 0x0e, 0xbd, 0x40, 0x69,
    0x24, 0x8d, 0x86, 0xea, 0x36, 0xde, 0x43, 0x58, 0x0c, 0xd0, 0x7c, 0xce,
    0xf1, 0xec, 0x8b, 0x6f, 0xe1, 0x6c, 0x2a, 0x50, 0x04, 0x24, 0xab, 0xe6,
    0x57, 0x65, 0xc4, 0x18, 0x91, 0x97, 0x2d, 0xc9,
};
static const struct drbg_kat_no_reseed kat7_nor_t = {
    6, kat7_nor_entropyin, kat7_nor_nonce, kat7_nor_persstr,
    kat7_nor_addin0, kat7_nor_addin1, kat7_nor_retbytes
};
static const struct drbg_kat kat7_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat7_nor_t
};

static const unsigned char kat8_nor_entropyin[] = {
    0xc4, 0x48, 0x62, 0x40, 0x35, 0xdf, 0x3b, 0x16, 0x8a, 0xf3, 0x97, 0x55,
    0xb9, 0x52, 0xf0, 0xef,
};
static const unsigned char kat8_nor_nonce[] = {
    0x6c, 0x32, 0xbd, 0x9d, 0xd5, 0x34, 0x7f, 0x3c,
};
static const unsigned char kat8_nor_persstr[] = {0};
static const unsigned char kat8_nor_addin0[] = {0};
static const unsigned char kat8_nor_addin1[] = {0};
static const unsigned char kat8_nor_retbytes[] = {
    0x3b, 0xb7, 0x25, 0x63, 0x37, 0x45, 0xc3, 0x0a, 0x3c, 0xfe, 0x9f, 0xeb,
    0x05, 0x29, 0x53, 0x2a, 0x65, 0x36, 0x8b, 0x65, 0xd2, 0xe7, 0xfb, 0x99,
    0x0d, 0x6b, 0xa2, 0xf9, 0xbb, 0xbb, 0xb8, 0xcd, 0x66, 0xec, 0x1a, 0xf5,
    0x30, 0x61, 0x4f, 0xe3, 0x9b, 0x7a, 0xd5, 0x5b, 0x2a, 0x62, 0x91, 0x79,
    0x17, 0x41, 0xf7, 0xe7, 0x61, 0x16, 0xda, 0x1e, 0xbf, 0xed, 0x65, 0x8e,
    0xc5, 0x5a, 0x86, 0x29, 0x47, 0x67, 0x16, 0x82, 0xd2, 0x87, 0xc0, 0xf6,
    0xe8, 0xce, 0xab, 0xd5, 0x0c, 0x4c, 0x37, 0x59,
};
static const struct drbg_kat_no_reseed kat8_nor_t = {
    7, kat8_nor_entropyin, kat8_nor_nonce, kat8_nor_persstr,
    kat8_nor_addin0, kat8_nor_addin1, kat8_nor_retbytes
};
static const struct drbg_kat kat8_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat8_nor_t
};

static const unsigned char kat9_nor_entropyin[] = {
    0xef, 0xd2, 0x56, 0x11, 0x48, 0xbf, 0xad, 0x41, 0x59, 0x5b, 0xe9, 0x69,
    0xc3, 0xf0, 0x21, 0xfe,
};
static const unsigned char kat9_nor_nonce[] = {
    0x7c, 0x72, 0x33, 0xe9, 0xe0, 0x6f, 0xa1, 0x89,
};
static const unsigned char kat9_nor_persstr[] = {0};
static const unsigned char kat9_nor_addin0[] = {0};
static const unsigned char kat9_nor_addin1[] = {0};
static const unsigned char kat9_nor_retbytes[] = {
    0xd9, 0xb6, 0xb5, 0xb0, 0xaa, 0x26, 0xa0, 0x1d, 0xe1, 0x52, 0xce, 0xd9,
    0x4a, 0x83, 0xbb, 0xb6, 0xaf, 0xfd, 0xb9, 0x15, 0xe4, 0x88, 0xa0, 0x4b,
    0x64, 0x22, 0x11, 0xf8, 0xe3, 0xb8, 0x8d, 0x89, 0xf5, 0x9d, 0x92, 0xf9,
    0xf4, 0xfc, 0xd8, 0x99, 0x89, 0x07, 0xbf, 0x14, 0x8b, 0xc5, 0x8f, 0x30,
    0x11, 0x5c, 0x8f, 0xff, 0x85, 0xc5, 0x57, 0x13, 0x54, 0x5f, 0xe7, 0x75,
    0x82, 0xd9, 0x06, 0x75, 0x8d, 0xb2, 0x0a, 0xe7, 0x53, 0x7f, 0x3d, 0xa6,
    0xf9, 0xa8, 0xc9, 0x94, 0xfa, 0x2d, 0x01, 0xb5,
};
static const struct drbg_kat_no_reseed kat9_nor_t = {
    8, kat9_nor_entropyin, kat9_nor_nonce, kat9_nor_persstr,
    kat9_nor_addin0, kat9_nor_addin1, kat9_nor_retbytes
};
static const struct drbg_kat kat9_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat9_nor_t
};

static const unsigned char kat10_nor_entropyin[] = {
    0xa1, 0x3e, 0xd8, 0xf6, 0x04, 0xac, 0x03, 0x46, 0xc2, 0xac, 0x27, 0x01,
    0x29, 0xbb, 0x35, 0x78,
};
static const unsigned char kat10_nor_nonce[] = {
    0x9c, 0xa5, 0xf6, 0x2c, 0x4d, 0x09, 0xde, 0x87,
};
static const unsigned char kat10_nor_persstr[] = {0};
static const unsigned char kat10_nor_addin0[] = {0};
static const unsigned char kat10_nor_addin1[] = {0};
static const unsigned char kat10_nor_retbytes[] = {
    0x6c, 0xf7, 0x0e, 0x27, 0xa9, 0x79, 0x3c, 0x14, 0xeb, 0xae, 0x73, 0xb6,
    0x53, 0x79, 0x8c, 0x15, 0xe1, 0x4e, 0x6c, 0xb9, 0xb8, 0x54, 0x34, 0x9e,
    0x42, 0xd8, 0x80, 0x1d, 0x49, 0x36, 0x4d, 0x77, 0x88, 0x68, 0x6f, 0x89,
    0xc6, 0xea, 0x2d, 0xb7, 0xf0, 0x38, 0xdd, 0x42, 0x43, 0x2d, 0xd7, 0x44,
    0xbb, 0xc9, 0x1e, 0x1f, 0xcd, 0xaf, 0x6e, 0xee, 0x2e, 0x76, 0x32, 0x5d,
    0xb7, 0x29, 0xfd, 0x3a, 0x99, 0xb9, 0x6c, 0xd6, 0x38, 0xc4, 0xa8, 0x92,
    0x35, 0x16, 0xfa, 0x04, 0xfd, 0x90, 0xea, 0xc5,
};
static const struct drbg_kat_no_reseed kat10_nor_t = {
    9, kat10_nor_entropyin, kat10_nor_nonce, kat10_nor_persstr,
    kat10_nor_addin0, kat10_nor_addin1, kat10_nor_retbytes
};
static const struct drbg_kat kat10_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat10_nor_t
};

static const unsigned char kat11_nor_entropyin[] = {
    0xdf, 0xe6, 0xeb, 0xca, 0x43, 0xec, 0x46, 0xb8, 0xaa, 0xc3, 0x53, 0x33,
    0xdd, 0xc4, 0x7d, 0x44,
};
static const unsigned char kat11_nor_nonce[] = {
    0x5d, 0x03, 0xcb, 0xac, 0xd4, 0x4d, 0xb1, 0x69,
};
static const unsigned char kat11_nor_persstr[] = {0};
static const unsigned char kat11_nor_addin0[] = {0};
static const unsigned char kat11_nor_addin1[] = {0};
static const unsigned char kat11_nor_retbytes[] = {
    0x1c, 0x1d, 0x75, 0x6b, 0x6d, 0x5d, 0x47, 0x5e, 0xe3, 0x93, 0x19, 0x39,
    0x92, 0x00, 0xb7, 0xf8, 0x64, 0x11, 0x24, 0x54, 0xbf, 0x03, 0xcc, 0x61,
    0xf6, 0x85, 0x6c, 0x5d, 0x42, 0x07, 0xb1, 0xcd, 0x97, 0x92, 0xc0, 0x84,
    0x1a, 0xb3, 0xb5, 0x67, 0x30, 0xab, 0x26, 0xd9, 0x5c, 0x36, 0x0c, 0xc0,
    0x2a, 0xc4, 0x61, 0x1e, 0xf4, 0x7f, 0x80, 0xd6, 0x65, 0xff, 0xb8, 0x32,
    0x7e, 0x10, 0x3f, 0xef, 0x24, 0x6b, 0xf9, 0x3a, 0x2a, 0x10, 0x38, 0xb2,
    0xbe, 0xda, 0x9a, 0x8a, 0x6d, 0x21, 0xc4, 0x15,
};
static const struct drbg_kat_no_reseed kat11_nor_t = {
    10, kat11_nor_entropyin, kat11_nor_nonce, kat11_nor_persstr,
    kat11_nor_addin0, kat11_nor_addin1, kat11_nor_retbytes
};
static const struct drbg_kat kat11_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat11_nor_t
};

static const unsigned char kat12_nor_entropyin[] = {
    0x48, 0xe5, 0xfb, 0xd1, 0xac, 0xc6, 0x01, 0xe0, 0x2d, 0x81, 0xce, 0x03,
    0x0f, 0x10, 0x01, 0x55,
};
static const unsigned char kat12_nor_nonce[] = {
    0xa0, 0xfc, 0x82, 0xd8, 0xe6, 0x92, 0xe6, 0x9b,
};
static const unsigned char kat12_nor_persstr[] = {0};
static const unsigned char kat12_nor_addin0[] = {0};
static const unsigned char kat12_nor_addin1[] = {0};
static const unsigned char kat12_nor_retbytes[] = {
    0x9e, 0xea, 0x52, 0x5c, 0xa5, 0x72, 0x09, 0x5b, 0x4d, 0x0b, 0x96, 0x7c,
    0xa8, 0xfc, 0xc5, 0xda, 0xe6, 0xec, 0x32, 0x6a, 0x0b, 0x9b, 0xb6, 0x3c,
    0x4a, 0x7f, 0x08, 0xd1, 0xda, 0xd4, 0xa5, 0xd2, 0x1c, 0x15, 0xcf, 0x4b,
    0x8d, 0xf9, 0xeb, 0x8a, 0xd2, 0x60, 0x37, 0xd9, 0xdc, 0xe9, 0xa7, 0x84,
    0x8a, 0xf5, 0xee, 0x8e, 0x30, 0x49, 0x3d, 0xea, 0x7b, 0xe2, 0xb1, 0xf1,
    0xff, 0xe8, 0xc4, 0x17, 0x1c, 0x13, 0x11, 0xac, 0xf9, 0x82, 0x1c, 0x06,
    0x61, 0x48, 0x38, 0xae, 0x2b, 0x8a, 0xb7, 0xb2,
};
static const struct drbg_kat_no_reseed kat12_nor_t = {
    11, kat12_nor_entropyin, kat12_nor_nonce, kat12_nor_persstr,
    kat12_nor_addin0, kat12_nor_addin1, kat12_nor_retbytes
};
static const struct drbg_kat kat12_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat12_nor_t
};

static const unsigned char kat13_nor_entropyin[] = {
    0xdc, 0x8c, 0x36, 0x9f, 0x6f, 0x65, 0x9c, 0xf7, 0x60, 0x71, 0x0b, 0x1a,
    0x69, 0xa5, 0x47, 0xcf,
};
static const unsigned char kat13_nor_nonce[] = {
    0x45, 0x84, 0x7c, 0xf4, 0x8b, 0x89, 0xb3, 0xc9,
};
static const unsigned char kat13_nor_persstr[] = {0};
static const unsigned char kat13_nor_addin0[] = {0};
static const unsigned char kat13_nor_addin1[] = {0};
static const unsigned char kat13_nor_retbytes[] = {
    0xf8, 0x45, 0x62, 0xc1, 0x5d, 0x3a, 0x35, 0x74, 0xa2, 0x7d, 0x3e, 0xd8,
    0x99, 0x1f, 0x17, 0xb8, 0xe7, 0xd3, 0xb7, 0x4a, 0x06, 0x50, 0xd6, 0xdc,
    0x32, 0x8d, 0x94, 0x7a, 0x98, 0x42, 0x57, 0xcf, 0x22, 0xf9, 0xf6, 0xef,
    0x5f, 0x5f, 0xb2, 0x08, 0x0e, 0x59, 0xd6, 0x5f, 0x6f, 0x2c, 0xa9, 0x96,
    0xb7, 0xb1, 0x51, 0xd7, 0xb7, 0x9b, 0xc9, 0x8c, 0xc7, 0x1e, 0x50, 0x42,
    0xa1, 0x0b, 0xc4, 0xb9, 0x33, 0xfa, 0x5c, 0xa2, 0x73, 0x1b, 0xfc, 0x5c,
    0xc6, 0xb0, 0xb7, 0xca, 0x25, 0xe6, 0x41, 0x6b,
};
static const struct drbg_kat_no_reseed kat13_nor_t = {
    12, kat13_nor_entropyin, kat13_nor_nonce, kat13_nor_persstr,
    kat13_nor_addin0, kat13_nor_addin1, kat13_nor_retbytes
};
static const struct drbg_kat kat13_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat13_nor_t
};

static const unsigned char kat14_nor_entropyin[] = {
    0x4e, 0x8b, 0x4b, 0x1e, 0x2f, 0x6a, 0x9f, 0xe8, 0xab, 0xb5, 0x6f, 0x71,
    0x6b, 0xc8, 0x94, 0xb8,
};
static const unsigned char kat14_nor_nonce[] = {
    0x38, 0x56, 0x5e, 0x90, 0xcd, 0x13, 0x1e, 0x3f,
};
static const unsigned char kat14_nor_persstr[] = {0};
static const unsigned char kat14_nor_addin0[] = {0};
static const unsigned char kat14_nor_addin1[] = {0};
static const unsigned char kat14_nor_retbytes[] = {
    0xe0, 0x46, 0x05, 0x0e, 0xd2, 0xa1, 0x4a, 0x56, 0x4d, 0xc6, 0x9b, 0x7e,
    0x4a, 0xa8, 0x9a, 0x99, 0x8a, 0x0d, 0xc7, 0x32, 0x5a, 0x05, 0xe4, 0x66,
    0x6b, 0x0c, 0xf6, 0x11, 0x88, 0x5f, 0x63, 0x91, 0xd9, 0x8c, 0x31, 0xd8,
    0xe2, 0x52, 0x7e, 0xa2, 0x96, 0x79, 0x1d, 0x0c, 0x8f, 0x34, 0x52, 0x2c,
    0xed, 0x84, 0x21, 0x7b, 0xae, 0x3e, 0xac, 0x5e, 0xb8, 0x73, 0x2c, 0xbc,
    0x4a, 0xf0, 0x71, 0x69, 0xe2, 0x5f, 0x9f, 0xb6, 0xda, 0x45, 0x88, 0xe0,
    0x19, 0x96, 0x9d, 0x79, 0xfc, 0xcc, 0xd5, 0xf7,
};
static const struct drbg_kat_no_reseed kat14_nor_t = {
    13, kat14_nor_entropyin, kat14_nor_nonce, kat14_nor_persstr,
    kat14_nor_addin0, kat14_nor_addin1, kat14_nor_retbytes
};
static const struct drbg_kat kat14_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat14_nor_t
};

static const unsigned char kat15_nor_entropyin[] = {
    0x6f, 0xe9, 0x59, 0x7b, 0x59, 0x90, 0x3b, 0x1a, 0xf4, 0x01, 0x2a, 0x15,
    0x36, 0x8a, 0xf7, 0xb1,
};
static const unsigned char kat15_nor_nonce[] = {
    0xfd, 0x3e, 0x84, 0xb3, 0xa9, 0x6c, 0xaa, 0xff,
};
static const unsigned char kat15_nor_persstr[] = {0};
static const unsigned char kat15_nor_addin0[] = {0};
static const unsigned char kat15_nor_addin1[] = {0};
static const unsigned char kat15_nor_retbytes[] = {
    0x1e, 0xee, 0x4c, 0x78, 0x64, 0x76, 0xd4, 0x88, 0xe5, 0x8d, 0x0e, 0x06,
    0x5b, 0xb0, 0x25, 0xdb, 0x54, 0x87, 0x87, 0xfa, 0xfb, 0xe7, 0x57, 0xf2,
    0x9e, 0xe2, 0xbd, 0x47, 0x81, 0xcf, 0x69, 0x21, 0x60, 0x91, 0xba, 0x2b,
    0x68, 0x91, 0x9b, 0x54, 0xad, 0x30, 0x70, 0xac, 0x72, 0xa2, 0x34, 0x23,
    0x20, 0xeb, 0x1e, 0x69, 0x7b, 0x91, 0x15, 0xac, 0xbe, 0x07, 0xe1, 0x94,
    0xd0, 0x60, 0x56, 0x2e, 0x4d, 0x0f, 0xd9, 0x66, 0xab, 0x29, 0xe2, 0xc5,
    0xe5, 0x60, 0x57, 0x4b, 0x2d, 0xac, 0x04, 0xce,
};
static const struct drbg_kat_no_reseed kat15_nor_t = {
    14, kat15_nor_entropyin, kat15_nor_nonce, kat15_nor_persstr,
    kat15_nor_addin0, kat15_nor_addin1, kat15_nor_retbytes
};
static const struct drbg_kat kat15_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat15_nor_t
};

static const unsigned char kat16_nor_entropyin[] = {
    0xc3, 0xef, 0x82, 0xce, 0x24, 0x1f, 0x02, 0xe4, 0x29, 0x8b, 0x11, 0x8c,
    0xa4, 0xf1, 0x62, 0x25,
};
static const unsigned char kat16_nor_nonce[] = {
    0x15, 0xe3, 0x2a, 0xbb, 0xae, 0x6b, 0x74, 0x33,
};
static const unsigned char kat16_nor_persstr[] = {0};
static const unsigned char kat16_nor_addin0[] = {
    0x2b, 0x79, 0x00, 0x52, 0xf0, 0x9b, 0x36, 0x4d, 0x4a, 0x82, 0x67, 0xa0,
    0xa7, 0xde, 0x63, 0xb8,
};
static const unsigned char kat16_nor_addin1[] = {
    0x2e, 0xe0, 0x81, 0x9a, 0x67, 0x1d, 0x07, 0xb5, 0x08, 0x5c, 0xc4, 0x6a,
    0xa0, 0xe6, 0x1b, 0x56,
};
static const unsigned char kat16_nor_retbytes[] = {
    0x58, 0x25, 0xfa, 0x1d, 0x1d, 0xc3, 0x3c, 0x64, 0xcd, 0xc8, 0x69, 0x06,
    0x82, 0xef, 0xf0, 0x60, 0x39, 0xe7, 0x95, 0x08, 0xc3, 0xaf, 0x48, 0xe8,
    0x80, 0xf8, 0x22, 0x7d, 0x5f, 0x9a, 0xaa, 0x14, 0xb3, 0xbc, 0x76, 0xba,
    0xee, 0x47, 0x7e, 0xbb, 0xb5, 0xc4, 0x55, 0x47, 0x13, 0x41, 0x79, 0x22,
    0x32, 0x57, 0x52, 0x5e, 0x8f, 0x3a, 0xfe, 0xfb, 0x78, 0xb5, 0x9d, 0xa0,
    0x32, 0xf1, 0x00, 0x6d, 0x74, 0xc9, 0x83, 0x13, 0x75, 0xa6, 0x77, 0xea,
    0xb3, 0x23, 0x9c, 0x94, 0xeb, 0xe3, 0xf7, 0xfa,
};
static const struct drbg_kat_no_reseed kat16_nor_t = {
    0, kat16_nor_entropyin, kat16_nor_nonce, kat16_nor_persstr,
    kat16_nor_addin0, kat16_nor_addin1, kat16_nor_retbytes
};
static const struct drbg_kat kat16_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat16_nor_t
};

static const unsigned char kat17_nor_entropyin[] = {
    0x4b, 0xa7, 0x07, 0x43, 0xac, 0xcd, 0x43, 0x19, 0xe1, 0x1d, 0xe0, 0x46,
    0x1d, 0x9a, 0x37, 0xaa,
};
static const unsigned char kat17_nor_nonce[] = {
    0x84, 0x5e, 0x4a, 0x96, 0x46, 0x2f, 0xb1, 0x6a,
};
static const unsigned char kat17_nor_persstr[] = {0};
static const unsigned char kat17_nor_addin0[] = {
    0xc7, 0xde, 0xec, 0x6c, 0xb5, 0x04, 0x9c, 0x34, 0xc8, 0xbe, 0x42, 0x37,
    0x5a, 0xfe, 0xc6, 0xf0,
};
static const unsigned char kat17_nor_addin1[] = {
    0x37, 0xaf, 0x84, 0x11, 0x08, 0x5b, 0x41, 0xd8, 0x8e, 0xcf, 0x89, 0x9f,
    0xf2, 0xbb, 0x13, 0x25,
};
static const unsigned char kat17_nor_retbytes[] = {
    0x7b, 0x6d, 0xac, 0xba, 0x0f, 0x66, 0xf2, 0xa1, 0x20, 0x71, 0xf9, 0xda,
    0x43, 0xf7, 0x5a, 0x61, 0x24, 0x0d, 0xf7, 0x5a, 0x47, 0xdd, 0xa1, 0x6f,
    0x47, 0x4e, 0xd2, 0xcf, 0x03, 0x08, 0xf5, 0x07, 0x94, 0xbf, 0x01, 0x7c,
    0xc6, 0xe6, 0xf0, 0x18, 0xfa, 0x9f, 0xa1, 0x50, 0x07, 0x48, 0xf7, 0xe0,
    0xb3, 0x6e, 0x62, 0xc6, 0x2a, 0x8d, 0xc4, 0x27, 0xf5, 0x8c, 0x1f, 0x2c,
    0x5a, 0xed, 0x10, 0xa4, 0x72, 0x6d, 0x88, 0x6c, 0xa1, 0x23, 0x25, 0xef,
    0x0c, 0x56, 0x6d, 0x5b, 0xc9, 0xa0, 0x4e, 0xcf,
};
static const struct drbg_kat_no_reseed kat17_nor_t = {
    1, kat17_nor_entropyin, kat17_nor_nonce, kat17_nor_persstr,
    kat17_nor_addin0, kat17_nor_addin1, kat17_nor_retbytes
};
static const struct drbg_kat kat17_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat17_nor_t
};

static const unsigned char kat18_nor_entropyin[] = {
    0xac, 0x3c, 0xd0, 0xea, 0x3d, 0x26, 0x19, 0x7c, 0xc9, 0x37, 0x9b, 0x82,
    0xbd, 0x82, 0x89, 0xff,
};
static const unsigned char kat18_nor_nonce[] = {
    0xe3, 0x63, 0xff, 0x61, 0x74, 0x1a, 0xab, 0x5e,
};
static const unsigned char kat18_nor_persstr[] = {0};
static const unsigned char kat18_nor_addin0[] = {
    0xc7, 0xc3, 0x0f, 0x2a, 0x6e, 0x50, 0x0b, 0x0b, 0xf9, 0x46, 0xd0, 0x21,
    0x93, 0x2b, 0xb1, 0x24,
};
static const unsigned char kat18_nor_addin1[] = {
    0xbc, 0xb4, 0x0c, 0x37, 0x3a, 0x4a, 0x9b, 0xe6, 0x06, 0x79, 0x0e, 0x55,
    0x17, 0x37, 0xf0, 0x6c,
};
static const unsigned char kat18_nor_retbytes[] = {
    0x15, 0xf4, 0x9e, 0x79, 0x14, 0xfe, 0x0b, 0xce, 0x03, 0x9b, 0x99, 0x06,
    0x0c, 0xc3, 0x72, 0xce, 0xb4, 0x53, 0x28, 0x29, 0x46, 0x57, 0x9a, 0x47,
    0xd9, 0xbf, 0xb6, 0x1c, 0x5a, 0xed, 0x7a, 0x4d, 0xa0, 0xdc, 0x77, 0xbe,
    0x7e, 0x7b, 0xc3, 0x3b, 0x63, 0xd6, 0x50, 0xf2, 0x84, 0x90, 0xd9, 0x2f,
    0xb5, 0x87, 0x25, 0xd5, 0x19, 0xfa, 0xd9, 0x44, 0x92, 0xf7, 0x12, 0xbd,
    0x81, 0x93, 0xd0, 0x94, 0xc8, 0x65, 0xcc, 0xbd, 0x41, 0x96, 0x25, 0xd5,
    0xd2, 0x7e, 0x24, 0x9b, 0x1c, 0xc5, 0x55, 0xcc,
};
static const struct drbg_kat_no_reseed kat18_nor_t = {
    2, kat18_nor_entropyin, kat18_nor_nonce, kat18_nor_persstr,
    kat18_nor_addin0, kat18_nor_addin1, kat18_nor_retbytes
};
static const struct drbg_kat kat18_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat18_nor_t
};

static const unsigned char kat19_nor_entropyin[] = {
    0xeb, 0x51, 0xfe, 0x7b, 0xd7, 0x24, 0x74, 0x65, 0x99, 0x47, 0xb0, 0xed,
    0x51, 0x09, 0xbe, 0x38,
};
static const unsigned char kat19_nor_nonce[] = {
    0x0a, 0x2a, 0xff, 0x93, 0xbf, 0xcc, 0x07, 0x22,
};
static const unsigned char kat19_nor_persstr[] = {0};
static const unsigned char kat19_nor_addin0[] = {
    0x94, 0xf6, 0x17, 0x3e, 0xd7, 0xc3, 0x6a, 0x63, 0xc4, 0xa9, 0x5c, 0x91,
    0x4d, 0x07, 0x42, 0xb7,
};
static const unsigned char kat19_nor_addin1[] = {
    0x6b, 0x11, 0x98, 0x91, 0x48, 0xcb, 0x86, 0x04, 0x07, 0xa3, 0x67, 0x5a,
    0x33, 0x79, 0x89, 0x69,
};
static const unsigned char kat19_nor_retbytes[] = {
    0xa6, 0x79, 0xa0, 0x8a, 0x10, 0x0e, 0xcc, 0x52, 0x50, 0xc3, 0x3b, 0x9b,
    0x1d, 0xcd, 0x9e, 0x20, 0x46, 0x9b, 0x30, 0x31, 0x6d, 0x0e, 0x0e, 0x47,
    0x80, 0xff, 0xe1, 0x5c, 0x06, 0xc9, 0x9c, 0x04, 0x40, 0xf7, 0xdb, 0xf0,
    0x46, 0x21, 0x21, 0x5d, 0x3d, 0x50, 0x81, 0x20, 0x1d, 0x4a, 0x10, 0x41,
    0x6a, 0x91, 0x38, 0x78, 0x2c, 0x56, 0x92, 0xed, 0x9e, 0xae, 0xdc, 0x99,
    0xc7, 0xcc, 0xfb, 0x09, 0xab, 0x08, 0x49, 0x38, 0x11, 0x50, 0x14, 0x5d,
    0xc0, 0xe5, 0x5a, 0x8f, 0xa8, 0x89, 0x80, 0x68,
};
static const struct drbg_kat_no_reseed kat19_nor_t = {
    3, kat19_nor_entropyin, kat19_nor_nonce, kat19_nor_persstr,
    kat19_nor_addin0, kat19_nor_addin1, kat19_nor_retbytes
};
static const struct drbg_kat kat19_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat19_nor_t
};

static const unsigned char kat20_nor_entropyin[] = {
    0x9c, 0xdb, 0x91, 0xa2, 0x1b, 0x4f, 0xdc, 0xc4, 0x10, 0xf9, 0x60, 0xce,
    0xb7, 0xac, 0x8e, 0xc8,
};
static const unsigned char kat20_nor_nonce[] = {
    0x9d, 0x01, 0xf7, 0x3a, 0xe1, 0x68, 0xed, 0x6e,
};
static const unsigned char kat20_nor_persstr[] = {0};
static const unsigned char kat20_nor_addin0[] = {
    0xe1, 0xeb, 0x4b, 0xb2, 0xd0, 0x80, 0xb1, 0xd3, 0xe9, 0x5b, 0xc0, 0x42,
    0xac, 0xe9, 0x4c, 0x68,
};
static const unsigned char kat20_nor_addin1[] = {
    0x06, 0x2d, 0xc8, 0x66, 0xf4, 0x14, 0xb9, 0x51, 0xe3, 0xf6, 0x1a, 0xa7,
    0xfe, 0x6d, 0x1f, 0xfe,
};
static const unsigned char kat20_nor_retbytes[] = {
    0xbf, 0xb0, 0x07, 0x7e, 0x99, 0x5f, 0x49, 0xc4, 0x03, 0xdc, 0x38, 0x29,
    0x7e, 0xe4, 0x7b, 0x41, 0x93, 0x88, 0x64, 0xd7, 0x34, 0xb2, 0xe7, 0xb7,
    0x94, 0xe9, 0xbb, 0xf4, 0x61, 0xa2, 0xa4, 0xa4, 0x5f, 0x36, 0x4d, 0x82,
    0xa6, 0x6b, 0x37, 0x30, 0xbc, 0x5c, 0x99, 0x02, 0xd7, 0xb5, 0x99, 0x9f,
    0x74, 0x55, 0x4e, 0x59, 0xe4, 0x54, 0xd6, 0x58, 0xa8, 0x6d, 0x48, 0x13,
    0xb5, 0xe7, 0x60, 0x9e, 0x03, 0xbc, 0x75, 0x0c, 0x38, 0xcd, 0x52, 0xb3,
    0x66, 0x76, 0x05, 0x7e, 0xd8, 0x0b, 0xfd, 0x41,
};
static const struct drbg_kat_no_reseed kat20_nor_t = {
    4, kat20_nor_entropyin, kat20_nor_nonce, kat20_nor_persstr,
    kat20_nor_addin0, kat20_nor_addin1, kat20_nor_retbytes
};
static const struct drbg_kat kat20_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat20_nor_t
};

static const unsigned char kat21_nor_entropyin[] = {
    0x7d, 0xd2, 0xff, 0xfc, 0x51, 0x45, 0x6d, 0xd4, 0x72, 0x60, 0x39, 0x5d,
    0xdc, 0xa6, 0x0c, 0x8f,
};
static const unsigned char kat21_nor_nonce[] = {
    0x63, 0x1e, 0x5b, 0x0c, 0xe8, 0x4e, 0x36, 0x52,
};
static const unsigned char kat21_nor_persstr[] = {0};
static const unsigned char kat21_nor_addin0[] = {
    0xcf, 0xc3, 0xab, 0x0a, 0x51, 0xc3, 0xb0, 0xad, 0xb3, 0x20, 0xfb, 0xa6,
    0x11, 0x2a, 0xc8, 0x56,
};
static const unsigned char kat21_nor_addin1[] = {
    0xf5, 0xaa, 0xae, 0x47, 0xc8, 0x23, 0xc0, 0x4e, 0x69, 0x60, 0xdc, 0xc2,
    0x05, 0xfd, 0x77, 0xa3,
};
static const unsigned char kat21_nor_retbytes[] = {
    0xc6, 0x11, 0x8f, 0x46, 0x89, 0xc7, 0xfc, 0xcd, 0x5a, 0xeb, 0x46, 0x87,
    0x75, 0x80, 0x7d, 0x39, 0x63, 0x1f, 0x5b, 0x31, 0x5f, 0x0c, 0x83, 0xb5,
    0xd9, 0x6f, 0xa6, 0xc2, 0xeb, 0xc0, 0xf9, 0x5e, 0xf4, 0x41, 0x4e, 0xfe,
    0x1c, 0x99, 0xce, 0x71, 0xe6, 0x2c, 0x46, 0x17, 0xdd, 0x1c, 0xc9, 0x57,
    0x77, 0xdd, 0xda, 0xa8, 0x25, 0x8d, 0x06, 0x5b, 0xf3, 0xaf, 0x2a, 0xdc,
    0x67, 0x9d, 0xd3, 0xd1, 0x9d, 0x23, 0xce, 0xd6, 0xe2, 0xb4, 0xb4, 0x7c,
    0x85, 0x65, 0x76, 0xf9, 0x69, 0x7a, 0x5f, 0x42,
};
static const struct drbg_kat_no_reseed kat21_nor_t = {
    5, kat21_nor_entropyin, kat21_nor_nonce, kat21_nor_persstr,
    kat21_nor_addin0, kat21_nor_addin1, kat21_nor_retbytes
};
static const struct drbg_kat kat21_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat21_nor_t
};

static const unsigned char kat22_nor_entropyin[] = {
    0x74, 0x91, 0xcd, 0x16, 0x5d, 0xe8, 0x4b, 0xe1, 0xce, 0x94, 0xf0, 0xbc,
    0x28, 0xb7, 0xa0, 0x1c,
};
static const unsigned char kat22_nor_nonce[] = {
    0x7d, 0xe5, 0xf3, 0x7f, 0xbd, 0x77, 0x75, 0x26,
};
static const unsigned char kat22_nor_persstr[] = {0};
static const unsigned char kat22_nor_addin0[] = {
    0x2c, 0x2e, 0xb1, 0x93, 0x19, 0x7b, 0xe1, 0x1f, 0xd9, 0x6c, 0x6b, 0x4f,
    0x8a, 0x56, 0x74, 0x8e,
};
static const unsigned char kat22_nor_addin1[] = {
    0x41, 0x1d, 0x9f, 0x71, 0xe0, 0xf9, 0x7e, 0x94, 0x06, 0x50, 0xf4, 0xca,
    0x99, 0x9f, 0xde, 0x5c,
};
static const unsigned char kat22_nor_retbytes[] = {
    0xdf, 0xd1, 0x5d, 0x00, 0xe3, 0x69, 0x3e, 0x89, 0xef, 0xff, 0xed, 0xd2,
    0x87, 0xaf, 0x2f, 0x6f, 0x33, 0x00, 0x4c, 0xb4, 0x2b, 0x16, 0xb3, 0x8b,
    0x41, 0xa3, 0x93, 0x54, 0x81, 0x90, 0xf8, 0x41, 0xdb, 0xca, 0xdd, 0xf6,
    0x7c, 0x41, 0x0b, 0x79, 0xa9, 0xcf, 0x10, 0x4a, 0xd1, 0x80, 0x5e, 0x89,
    0xe2, 0xc1, 0xa4, 0x1a, 0x24, 0x05, 0x36, 0x14, 0xd6, 0x7e, 0x95, 0x67,
    0xc1, 0xac, 0x98, 0x9d, 0xb3, 0x3d, 0x79, 0x2d, 0x80, 0x24, 0x55, 0xb1,
    0x7b, 0x17, 0x4a, 0xdb, 0xc9, 0xde, 0xd7, 0xb4,
};
static const struct drbg_kat_no_reseed kat22_nor_t = {
    6, kat22_nor_entropyin, kat22_nor_nonce, kat22_nor_persstr,
    kat22_nor_addin0, kat22_nor_addin1, kat22_nor_retbytes
};
static const struct drbg_kat kat22_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat22_nor_t
};

static const unsigned char kat23_nor_entropyin[] = {
    0x14, 0x98, 0x67, 0x34, 0xe4, 0x19, 0xb6, 0x47, 0xda, 0x59, 0xba, 0xd3,
    0x86, 0xa1, 0x6c, 0xf9,
};
static const unsigned char kat23_nor_nonce[] = {
    0xc3, 0x7c, 0x9b, 0xf5, 0xa4, 0x36, 0x51, 0x23,
};
static const unsigned char kat23_nor_persstr[] = {0};
static const unsigned char kat23_nor_addin0[] = {
    0x58, 0xbb, 0x48, 0x58, 0x52, 0xa8, 0x67, 0xeb, 0x55, 0x6c, 0x53, 0xe9,
    0x04, 0xc5, 0x59, 0x39,
};
static const unsigned char kat23_nor_addin1[] = {
    0x92, 0xfd, 0x0b, 0x39, 0xdb, 0x4f, 0x98, 0x26, 0x03, 0x99, 0xd5, 0xc1,
    0x2d, 0x74, 0x4a, 0xd3,
};
static const unsigned char kat23_nor_retbytes[] = {
    0x81, 0xc7, 0x50, 0x9f, 0xd3, 0xaa, 0x29, 0xf7, 0x9e, 0x6c, 0xe6, 0x03,
    0x7e, 0xd4, 0x80, 0x03, 0x17, 0xb7, 0x4a, 0x6e, 0xab, 0x5c, 0x8e, 0xe1,
    0x52, 0x05, 0xd2, 0x92, 0xbd, 0xab, 0x76, 0xd7, 0xba, 0xe6, 0x2c, 0xb2,
    0xc2, 0x6c, 0x6b, 0xb9, 0x5d, 0x66, 0xbc, 0x8d, 0x3b, 0x27, 0x0e, 0x89,
    0xf9, 0xe6, 0x1b, 0x62, 0x81, 0x8e, 0xbe, 0x27, 0xff, 0xe4, 0x9f, 0x3e,
    0xb8, 0x6e, 0xe5, 0xfc, 0xd4, 0x92, 0xb7, 0xee, 0x03, 0xdd, 0xca, 0x6d,
    0xd3, 0x33, 0xa3, 0xd9, 0x33, 0x4f, 0xe2, 0xda,
};
static const struct drbg_kat_no_reseed kat23_nor_t = {
    7, kat23_nor_entropyin, kat23_nor_nonce, kat23_nor_persstr,
    kat23_nor_addin0, kat23_nor_addin1, kat23_nor_retbytes
};
static const struct drbg_kat kat23_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat23_nor_t
};

static const unsigned char kat24_nor_entropyin[] = {
    0x29, 0x87, 0x24, 0x60, 0xa7, 0xf1, 0x3c, 0x18, 0xe6, 0x80, 0x67, 0xa0,
    0xfb, 0x8d, 0xd6, 0xc2,
};
static const unsigned char kat24_nor_nonce[] = {
    0x7d, 0x62, 0xe2, 0x0a, 0xa7, 0x1a, 0x94, 0x5b,
};
static const unsigned char kat24_nor_persstr[] = {0};
static const unsigned char kat24_nor_addin0[] = {
    0xeb, 0xd9, 0x87, 0xbd, 0x55, 0x1b, 0x35, 0x4b, 0xdd, 0xc1, 0x74, 0x14,
    0xa2, 0x6e, 0x1f, 0x08,
};
static const unsigned char kat24_nor_addin1[] = {
    0xc2, 0xc0, 0xe5, 0x3e, 0x61, 0x45, 0x1a, 0xe1, 0x7f, 0xf6, 0xa9, 0x0c,
    0x49, 0xb4, 0x66, 0xd7,
};
static const unsigned char kat24_nor_retbytes[] = {
    0x95, 0x60, 0x98, 0x22, 0x20, 0x32, 0x63, 0xf9, 0x2a, 0x40, 0x95, 0x5a,
    0x56, 0x19, 0x24, 0x0e, 0x74, 0x09, 0x13, 0x63, 0x34, 0xe3, 0x1b, 0x95,
    0x63, 0x68, 0xb4, 0x9b, 0xb8, 0x55, 0x06, 0x51, 0x72, 0xde, 0x37, 0x18,
    0xc4, 0x04, 0x83, 0x87, 0x1b, 0x45, 0x5d, 0x8e, 0x25, 0xf4, 0x34, 0x23,
    0xb9, 0x91, 0xb3, 0xaa, 0x9b, 0xcf, 0x0e, 0x27, 0xb4, 0x50, 0x9b, 0x45,
    0xd8, 0xf7, 0x98, 0x3b, 0xc0, 0x54, 0xed, 0x5f, 0xf1, 0x99, 0x9f, 0xbb,
    0x75, 0xf9, 0xe9, 0x1b, 0xfe, 0xab, 0x8a, 0x6e,
};
static const struct drbg_kat_no_reseed kat24_nor_t = {
    8, kat24_nor_entropyin, kat24_nor_nonce, kat24_nor_persstr,
    kat24_nor_addin0, kat24_nor_addin1, kat24_nor_retbytes
};
static const struct drbg_kat kat24_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat24_nor_t
};

static const unsigned char kat25_nor_entropyin[] = {
    0xbb, 0x0a, 0x1f, 0x9b, 0x84, 0x74, 0x1b, 0x55, 0x38, 0x4b, 0x89, 0xc0,
    0x55, 0xd5, 0x02, 0x72,
};
static const unsigned char kat25_nor_nonce[] = {
    0x25, 0x86, 0xdf, 0xe6, 0x30, 0x6d, 0xb7, 0x3d,
};
static const unsigned char kat25_nor_persstr[] = {0};
static const unsigned char kat25_nor_addin0[] = {
    0xbc, 0xf6, 0xee, 0x1e, 0xac, 0x1d, 0x2d, 0x3a, 0x7b, 0x96, 0x05, 0x00,
    0xf8, 0x0c, 0x53, 0x28,
};
static const unsigned char kat25_nor_addin1[] = {
    0x54, 0x3f, 0xea, 0x55, 0x15, 0xec, 0x0e, 0xa5, 0xd2, 0x5d, 0x01, 0x9b,
    0xce, 0x97, 0x39, 0x09,
};
static const unsigned char kat25_nor_retbytes[] = {
    0x9d, 0x88, 0x97, 0x0a, 0x80, 0x76, 0x40, 0x1f, 0xc1, 0x32, 0xec, 0xff,
    0x78, 0x2a, 0x2f, 0x68, 0x17, 0xe4, 0x82, 0x9f, 0x6e, 0x87, 0x41, 0x72,
    0xab, 0x3b, 0x99, 0x1f, 0x46, 0xff, 0x53, 0x19, 0x98, 0x7d, 0xd6, 0xae,
    0x53, 0x9f, 0xc2, 0xa3, 0x65, 0x7f, 0x00, 0x48, 0x0c, 0x7b, 0xb5, 0xe7,
    0xdd, 0x6d, 0x1a, 0x12, 0x96, 0x9e, 0x81, 0xa3, 0x87, 0xa9, 0xae, 0x43,
    0xab, 0x9c, 0x0d, 0x28, 0x26, 0x44, 0x8c, 0x6c, 0xa9, 0xe9, 0xbf, 0x9a,
    0xe0, 0x82, 0x1a, 0x35, 0xc4, 0xe2, 0xf0, 0xbd,
};
static const struct drbg_kat_no_reseed kat25_nor_t = {
    9, kat25_nor_entropyin, kat25_nor_nonce, kat25_nor_persstr,
    kat25_nor_addin0, kat25_nor_addin1, kat25_nor_retbytes
};
static const struct drbg_kat kat25_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat25_nor_t
};

static const unsigned char kat26_nor_entropyin[] = {
    0xb9, 0xde, 0x65, 0x31, 0x31, 0x2e, 0x4c, 0x7c, 0x6f, 0x49, 0x2f, 0x94,
    0x3d, 0x74, 0xc2, 0xc3,
};
static const unsigned char kat26_nor_nonce[] = {
    0xae, 0x15, 0xbd, 0xf3, 0xcd, 0x20, 0x86, 0xdc,
};
static const unsigned char kat26_nor_persstr[] = {0};
static const unsigned char kat26_nor_addin0[] = {
    0xf4, 0x03, 0x54, 0x91, 0x8f, 0x58, 0x78, 0xcc, 0xb0, 0x2e, 0x38, 0x29,
    0x89, 0x30, 0x35, 0xf5,
};
static const unsigned char kat26_nor_addin1[] = {
    0xdf, 0x58, 0x7b, 0x08, 0x22, 0xe6, 0x49, 0x18, 0xc2, 0xf4, 0x90, 0x3a,
    0x0f, 0x3e, 0x2e, 0x3b,
};
static const unsigned char kat26_nor_retbytes[] = {
    0xa7, 0xab, 0x93, 0xcd, 0x9f, 0xfe, 0x24, 0x6d, 0xd4, 0xe3, 0xdc, 0x55,
    0xba, 0x94, 0x6e, 0xb2, 0x15, 0x1d, 0x03, 0xd0, 0xdd, 0x2b, 0x6a, 0x81,
    0x2f, 0x8e, 0x3a, 0x2d, 0xb1, 0x9c, 0x78, 0x10, 0xfb, 0xf2, 0xb9, 0x51,
    0x34, 0x13, 0x14, 0x15, 0x62, 0xaa, 0xd7, 0xe5, 0x7f, 0x8d, 0xdf, 0x74,
    0x25, 0x70, 0x2f, 0x6c, 0x19, 0x76, 0xdf, 0xcd, 0x82, 0x1a, 0x71, 0x58,
    0x68, 0xcb, 0x6d, 0x43, 0x6a, 0xcf, 0x97, 0x08, 0x48, 0x1b, 0xfa, 0x09,
    0x54, 0x54, 0xc8, 0xc6, 0xb8, 0x1d, 0xe6, 0x2b,
};
static const struct drbg_kat_no_reseed kat26_nor_t = {
    10, kat26_nor_entropyin, kat26_nor_nonce, kat26_nor_persstr,
    kat26_nor_addin0, kat26_nor_addin1, kat26_nor_retbytes
};
static const struct drbg_kat kat26_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat26_nor_t
};

static const unsigned char kat27_nor_entropyin[] = {
    0x70, 0xfd, 0xd3, 0x7d, 0x38, 0x64, 0xbe, 0x69, 0x92, 0xfb, 0xdb, 0x78,
    0x55, 0x64, 0x9f, 0xcc,
};
static const unsigned char kat27_nor_nonce[] = {
    0x9f, 0x6d, 0x99, 0x4d, 0x1c, 0xdd, 0x27, 0x0b,
};
static const unsigned char kat27_nor_persstr[] = {0};
static const unsigned char kat27_nor_addin0[] = {
    0x9e, 0x5a, 0x48, 0xc9, 0xb3, 0x42, 0x89, 0xdc, 0xce, 0x94, 0x7c, 0x37,
    0xcf, 0x1a, 0x8d, 0xb7,
};
static const unsigned char kat27_nor_addin1[] = {
    0xa2, 0x3d, 0xd5, 0x8b, 0xf5, 0xfd, 0x18, 0x7f, 0xc2, 0xa0, 0x63, 0x62,
    0x24, 0x38, 0x8a, 0x3f,
};
static const unsigned char kat27_nor_retbytes[] = {
    0xbb, 0x89, 0xab, 0x58, 0x54, 0xed, 0xe9, 0x2a, 0xb6, 0x76, 0x6d, 0xb2,
    0xdb, 0x04, 0xa7, 0x8c, 0xd0, 0x31, 0x92, 0xba, 0x64, 0xf9, 0x57, 0x77,
    0x9d, 0x6a, 0x23, 0x22, 0x1e, 0xdb, 0xe8, 0x6c, 0x1a, 0xb7, 0x6d, 0xdd,
    0x1a, 0xf7, 0x40, 0x66, 0x88, 0xfb, 0xeb, 0xb8, 0xad, 0x37, 0x7b, 0xb4,
    0x99, 0xf6, 0x6e, 0xc9, 0x67, 0x27, 0x05, 0xd2, 0xfe, 0x30, 0xe8, 0xa2,
    0xd5, 0xee, 0xd9, 0xb5, 0xda, 0x5c, 0x2c, 0x2f, 0xd1, 0xe0, 0x59, 0xd0,
    0x6c, 0x84, 0xe3, 0xa4, 0xd0, 0x46, 0xfc, 0x66,
};
static const struct drbg_kat_no_reseed kat27_nor_t = {
    11, kat27_nor_entropyin, kat27_nor_nonce, kat27_nor_persstr,
    kat27_nor_addin0, kat27_nor_addin1, kat27_nor_retbytes
};
static const struct drbg_kat kat27_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat27_nor_t
};

static const unsigned char kat28_nor_entropyin[] = {
    0x0e, 0x18, 0x34, 0xca, 0x71, 0x8e, 0x2e, 0xc8, 0x9d, 0xf6, 0x6f, 0x76,
    0x7e, 0x31, 0x01, 0x8b,
};
static const unsigned char kat28_nor_nonce[] = {
    0xb9, 0x77, 0x57, 0xad, 0xf5, 0x15, 0x8e, 0x82,
};
static const unsigned char kat28_nor_persstr[] = {0};
static const unsigned char kat28_nor_addin0[] = {
    0x1b, 0x8f, 0x11, 0x5d, 0xc9, 0x37, 0x54, 0x00, 0x21, 0x2e, 0x50, 0x33,
    0x76, 0xa9, 0x0f, 0x6b,
};
static const unsigned char kat28_nor_addin1[] = {
    0xb7, 0xb8, 0xa7, 0x4d, 0x1d, 0xb1, 0xec, 0xb6, 0x6c, 0x8a, 0x7f, 0x82,
    0xec, 0x55, 0xec, 0x13,
};
static const unsigned char kat28_nor_retbytes[] = {
    0xa5, 0x5d, 0x54, 0xfa, 0x57, 0xaa, 0x36, 0x45, 0x6c, 0x64, 0xed, 0xf2,
    0xa4, 0xbf, 0xc3, 0xa0, 0x45, 0x8d, 0x20, 0xc0, 0x0b, 0xd1, 0x9b, 0x5d,
    0xc9, 0x71, 0x64, 0x7e, 0xb4, 0x6d, 0x28, 0x24, 0x66, 0xc5, 0xad, 0xe2,
    0x64, 0x03, 0xf3, 0x7a, 0xf1, 0x90, 0x6e, 0x57, 0x55, 0x0b, 0xd4, 0x1e,
    0xd4, 0xe8, 0x84, 0x8b, 0x48, 0x5a, 0x12, 0x20, 0x50, 0xc7, 0x1e, 0xc1,
    0xc5, 0x9a, 0x28, 0x39, 0x9f, 0xf0, 0x70, 0x97, 0xab, 0x28, 0x49, 0xcb,
    0x01, 0x65, 0xfe, 0x6f, 0xa5, 0x77, 0xf0, 0x32,
};
static const struct drbg_kat_no_reseed kat28_nor_t = {
    12, kat28_nor_entropyin, kat28_nor_nonce, kat28_nor_persstr,
    kat28_nor_addin0, kat28_nor_addin1, kat28_nor_retbytes
};
static const struct drbg_kat kat28_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat28_nor_t
};

static const unsigned char kat29_nor_entropyin[] = {
    0x49, 0x66, 0x6d, 0x8b, 0x26, 0xcf, 0x63, 0x0f, 0x37, 0xea, 0xb9, 0x08,
    0x9e, 0xab, 0xba, 0xe2,
};
static const unsigned char kat29_nor_nonce[] = {
    0x85, 0xea, 0xd7, 0x58, 0xfa, 0x67, 0xa2, 0x9a,
};
static const unsigned char kat29_nor_persstr[] = {0};
static const unsigned char kat29_nor_addin0[] = {
    0x77, 0x02, 0xe3, 0x97, 0xa4, 0x47, 0x47, 0x2d, 0xf6, 0xab, 0xa9, 0xb3,
    0x22, 0x36, 0x8a, 0xee,
};
static const unsigned char kat29_nor_addin1[] = {
    0xd1, 0xaf, 0xea, 0x4c, 0x22, 0x36, 0x93, 0x33, 0xa7, 0xb0, 0x82, 0x99,
    0xae, 0x3d, 0x44, 0xb1,
};
static const unsigned char kat29_nor_retbytes[] = {
    0x63, 0x85, 0x4c, 0x32, 0xa5, 0x80, 0x54, 0xe7, 0xdd, 0xb0, 0xc0, 0xc3,
    0x9b, 0x8e, 0x54, 0xa9, 0xde, 0x6d, 0xf4, 0xa9, 0xd8, 0xb6, 0x32, 0x44,
    0x52, 0xcf, 0x9d, 0x02, 0x23, 0x19, 0xf6, 0xb7, 0xe3, 0x0c, 0x7f, 0x6c,
    0x75, 0x01, 0x9a, 0x0f, 0x37, 0xb7, 0x4b, 0x8d, 0x1c, 0xb1, 0x9b, 0x0d,
    0xd2, 0x81, 0x4d, 0xe6, 0xb3, 0x66, 0x8f, 0xc7, 0x9c, 0x34, 0x22, 0x2f,
    0xff, 0xaf, 0xba, 0xca, 0x94, 0x89, 0x41, 0x3d, 0x0e, 0x29, 0xf8, 0xb6,
    0x2d, 0x75, 0x94, 0x27, 0x43, 0x71, 0xdd, 0x90,
};
static const struct drbg_kat_no_reseed kat29_nor_t = {
    13, kat29_nor_entropyin, kat29_nor_nonce, kat29_nor_persstr,
    kat29_nor_addin0, kat29_nor_addin1, kat29_nor_retbytes
};
static const struct drbg_kat kat29_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat29_nor_t
};

static const unsigned char kat30_nor_entropyin[] = {
    0xa3, 0xa0, 0xcf, 0xee, 0x0a, 0x1f, 0xcc, 0x88, 0x55, 0x83, 0x61, 0x5d,
    0x2d, 0x4b, 0x59, 0x52,
};
static const unsigned char kat30_nor_nonce[] = {
    0xe7, 0xde, 0x2c, 0x6f, 0xfb, 0xf1, 0xfc, 0xa0,
};
static const unsigned char kat30_nor_persstr[] = {0};
static const unsigned char kat30_nor_addin0[] = {
    0x93, 0xdc, 0x42, 0x4b, 0xd0, 0xd2, 0x66, 0x87, 0x96, 0x01, 0x74, 0x5a,
    0x23, 0x31, 0x71, 0x41,
};
static const unsigned char kat30_nor_addin1[] = {
    0xa1, 0x73, 0x21, 0x01, 0x5d, 0x32, 0x7c, 0x5d, 0xc0, 0xbc, 0x1e, 0x13,
    0x0a, 0xad, 0x81, 0xee,
};
static const unsigned char kat30_nor_retbytes[] = {
    0xf6, 0x82, 0x83, 0x4b, 0x5b, 0x49, 0x2e, 0x09, 0xff, 0x8e, 0x0f, 0x2c,
    0x80, 0x68, 0x3b, 0x03, 0x2a, 0x3b, 0x26, 0x2d, 0x16, 0xbc, 0x60, 0x9c,
    0x55, 0x0d, 0xc0, 0xe7, 0x4a, 0x4b, 0x7d, 0x8e, 0xbc, 0x0e, 0x3b, 0x8f,
    0x2c, 0x99, 0x70, 0xd9, 0x0a, 0xec, 0x9a, 0x82, 0x49, 0x7d, 0xde, 0xd2,
    0x04, 0x22, 0xb1, 0x7b, 0x9e, 0x3c, 0xc3, 0xbc, 0xa7, 0x71, 0xcb, 0xe7,
    0x17, 0xdd, 0xae, 0xd5, 0xa7, 0xa6, 0xae, 0x26, 0x01, 0xc7, 0xf7, 0x65,
    0xea, 0xa7, 0x19, 0xb7, 0x16, 0x24, 0xe8, 0x3b,
};
static const struct drbg_kat_no_reseed kat30_nor_t = {
    14, kat30_nor_entropyin, kat30_nor_nonce, kat30_nor_persstr,
    kat30_nor_addin0, kat30_nor_addin1, kat30_nor_retbytes
};
static const struct drbg_kat kat30_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat30_nor_t
};

static const unsigned char kat31_nor_entropyin[] = {
    0xf7, 0xe3, 0x16, 0xb1, 0x31, 0x17, 0xdc, 0xc1, 0x8c, 0x44, 0x07, 0xb6,
    0xa5, 0xcd, 0xc5, 0xd8,
};
static const unsigned char kat31_nor_nonce[] = {
    0xb8, 0x0d, 0xde, 0xe7, 0x5c, 0xf3, 0x9a, 0x48,
};
static const unsigned char kat31_nor_persstr[] = {
    0x81, 0x6c, 0xb1, 0x37, 0xef, 0x64, 0xf9, 0xdf, 0x71, 0xa3, 0xb3, 0xa0,
    0xb3, 0xaa, 0xf9, 0xb1,
};
static const unsigned char kat31_nor_addin0[] = {0};
static const unsigned char kat31_nor_addin1[] = {0};
static const unsigned char kat31_nor_retbytes[] = {
    0xbe, 0x88, 0x85, 0x85, 0xd9, 0x5d, 0x95, 0x26, 0x9f, 0x00, 0x2a, 0xbd,
    0x8b, 0x1e, 0x33, 0xb2, 0xcb, 0x66, 0x7a, 0x96, 0xd3, 0xbe, 0x6d, 0x20,
    0xd7, 0x84, 0xb1, 0xbb, 0xc6, 0x63, 0x93, 0x47, 0x83, 0x7d, 0x01, 0xd4,
    0xb9, 0x5e, 0xed, 0x81, 0x37, 0xcf, 0x29, 0xfe, 0x72, 0x4c, 0xfe, 0xdf,
    0x8b, 0x23, 0xf9, 0x25, 0x84, 0x80, 0xbe, 0x35, 0x0c, 0x34, 0x07, 0x97,
    0x3c, 0x59, 0xa9, 0xd7, 0xf3, 0xa6, 0x58, 0x5b, 0x3c, 0x0e, 0x36, 0xa3,
    0x6c, 0x62, 0x34, 0xd6, 0x88, 0x52, 0xac, 0xaf,
};
static const struct drbg_kat_no_reseed kat31_nor_t = {
    0, kat31_nor_entropyin, kat31_nor_nonce, kat31_nor_persstr,
    kat31_nor_addin0, kat31_nor_addin1, kat31_nor_retbytes
};
static const struct drbg_kat kat31_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat31_nor_t
};

static const unsigned char kat32_nor_entropyin[] = {
    0x03, 0x3f, 0x1e, 0x32, 0x61, 0x2f, 0xd1, 0x19, 0x74, 0xc6, 0x99, 0x2b,
    0xe2, 0x8b, 0x00, 0x9d,
};
static const unsigned char kat32_nor_nonce[] = {
    0x6b, 0x52, 0x8c, 0x9d, 0x2a, 0xcc, 0xd0, 0xaf,
};
static const unsigned char kat32_nor_persstr[] = {
    0x1b, 0x1d, 0xef, 0xb5, 0x46, 0x20, 0x6d, 0x55, 0x5c, 0x55, 0xbf, 0x23,
    0xa7, 0xfe, 0xfd, 0x10,
};
static const unsigned char kat32_nor_addin0[] = {0};
static const unsigned char kat32_nor_addin1[] = {0};
static const unsigned char kat32_nor_retbytes[] = {
    0xf1, 0xe0, 0x12, 0xc3, 0x3a, 0x6f, 0x2a, 0x5f, 0x06, 0xf0, 0x0e, 0xca,
    0xe9, 0x97, 0xe4, 0xd6, 0xc4, 0x8b, 0x3e, 0x5b, 0x76, 0x9c, 0x05, 0x89,
    0x51, 0xe3, 0xcb, 0x3c, 0xd1, 0xae, 0x07, 0x94, 0x16, 0x5b, 0x54, 0xd9,
    0x0f, 0x0d, 0xf6, 0x6a, 0x40, 0x02, 0x59, 0x56, 0xeb, 0x76, 0xa6, 0x15,
    0x0a, 0x85, 0x0c, 0xec, 0x48, 0xc2, 0x49, 0x7d, 0x2d, 0x58, 0x61, 0x29,
    0x68, 0x77, 0xe6, 0x27, 0xdf, 0x2f, 0x06, 0x4d, 0x3d, 0x7c, 0x12, 0xf9,
    0x69, 0x2d, 0x65, 0x7f, 0x4e, 0xea, 0x70, 0x86,
};
static const struct drbg_kat_no_reseed kat32_nor_t = {
    1, kat32_nor_entropyin, kat32_nor_nonce, kat32_nor_persstr,
    kat32_nor_addin0, kat32_nor_addin1, kat32_nor_retbytes
};
static const struct drbg_kat kat32_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat32_nor_t
};

static const unsigned char kat33_nor_entropyin[] = {
    0xb2, 0x07, 0xab, 0x84, 0x0b, 0x35, 0x7a, 0xe0, 0xf8, 0x4d, 0x84, 0xfd,
    0x91, 0xfe, 0x9d, 0xb2,
};
static const unsigned char kat33_nor_nonce[] = {
    0x6a, 0xd6, 0xf6, 0x7b, 0x65, 0x4f, 0x8c, 0x1d,
};
static const unsigned char kat33_nor_persstr[] = {
    0x80, 0xe0, 0x29, 0x0e, 0x51, 0xb6, 0x7e, 0x7a, 0x5f, 0x26, 0xb1, 0x51,
    0x71, 0x7c, 0x11, 0xc4,
};
static const unsigned char kat33_nor_addin0[] = {0};
static const unsigned char kat33_nor_addin1[] = {0};
static const unsigned char kat33_nor_retbytes[] = {
    0xb1, 0x50, 0x05, 0x33, 0x48, 0x3e, 0x00, 0x7f, 0x92, 0x21, 0x66, 0x3f,
    0x00, 0xe3, 0x74, 0x5b, 0x15, 0x1b, 0x80, 0x85, 0x22, 0x9d, 0xf0, 0x5e,
    0x90, 0xad, 0x6e, 0x1b, 0x4f, 0x28, 0x6c, 0x11, 0x65, 0x6b, 0xcd, 0x13,
    0xec, 0x28, 0x0c, 0xb9, 0x20, 0x75, 0xb5, 0x0c, 0x17, 0xf4, 0xcc, 0x9b,
    0xf4, 0xa0, 0x7d, 0x66, 0x1c, 0xfe, 0x17, 0x51, 0x8e, 0xc5, 0xd2, 0x64,
    0x93, 0x18, 0xb9, 0x58, 0x7d, 0xb8, 0x8e, 0x09, 0x3e, 0x98, 0x1f, 0xc5,
    0x16, 0x74, 0xc9, 0x89, 0x30, 0x44, 0xac, 0x7d,
};
static const struct drbg_kat_no_reseed kat33_nor_t = {
    2, kat33_nor_entropyin, kat33_nor_nonce, kat33_nor_persstr,
    kat33_nor_addin0, kat33_nor_addin1, kat33_nor_retbytes
};
static const struct drbg_kat kat33_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat33_nor_t
};

static const unsigned char kat34_nor_entropyin[] = {
    0xd8, 0x89, 0x57, 0xfb, 0xca, 0x97, 0xba, 0x8a, 0x2e, 0x73, 0xeb, 0x64,
    0xa3, 0x8e, 0xf3, 0x1c,
};
static const unsigned char kat34_nor_nonce[] = {
    0x3e, 0x07, 0xa9, 0x17, 0xc1, 0x11, 0xc5, 0xf4,
};
static const unsigned char kat34_nor_persstr[] = {
    0x70, 0x86, 0x0b, 0xe0, 0x05, 0xc1, 0xdd, 0x2b, 0xf3, 0x22, 0x74, 0x40,
    0x40, 0xe2, 0xb7, 0x02,
};
static const unsigned char kat34_nor_addin0[] = {0};
static const unsigned char kat34_nor_addin1[] = {0};
static const unsigned char kat34_nor_retbytes[] = {
    0xd7, 0xa9, 0xf9, 0xc6, 0xac, 0x7b, 0x5a, 0xb8, 0xe0, 0x6e, 0xf7, 0x07,
    0x62, 0x68, 0x88, 0x62, 0x8a, 0x3f, 0xa9, 0xcc, 0x86, 0x38, 0x5a, 0xb5,
    0x6e, 0x39, 0x90, 0x1e, 0x8b, 0x72, 0x0b, 0x93, 0x2c, 0x5c, 0x77, 0x98,
    0x0a, 0x96, 0x7f, 0x38, 0xed, 0xb9, 0x5f, 0x19, 0xb3, 0xc0, 0xf8, 0xe9,
    0x5e, 0x0b, 0xfd, 0xc8, 0xfb, 0x40, 0x47, 0x71, 0xf0, 0x53, 0xda, 0x5f,
    0x96, 0xe7, 0x9d, 0x52, 0x94, 0x5c, 0xb4, 0x27, 0x79, 0x81, 0xa9, 0x78,
    0xce, 0xe8, 0x0d, 0x84, 0xb2, 0x86, 0xe1, 0xb6,
};
static const struct drbg_kat_no_reseed kat34_nor_t = {
    3, kat34_nor_entropyin, kat34_nor_nonce, kat34_nor_persstr,
    kat34_nor_addin0, kat34_nor_addin1, kat34_nor_retbytes
};
static const struct drbg_kat kat34_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat34_nor_t
};

static const unsigned char kat35_nor_entropyin[] = {
    0x0a, 0x4c, 0x59, 0x0e, 0xbc, 0xef, 0x05, 0x56, 0xaf, 0x98, 0x11, 0xb6,
    0x46, 0x2b, 0x92, 0xd6,
};
static const unsigned char kat35_nor_nonce[] = {
    0x23, 0x8d, 0x39, 0x5f, 0x8d, 0x29, 0x52, 0x05,
};
static const unsigned char kat35_nor_persstr[] = {
    0x3d, 0xf3, 0xdd, 0xb4, 0xcb, 0x22, 0xe2, 0xa6, 0xcc, 0xd6, 0x85, 0xf0,
    0x94, 0x07, 0x8a, 0x12,
};
static const unsigned char kat35_nor_addin0[] = {0};
static const unsigned char kat35_nor_addin1[] = {0};
static const unsigned char kat35_nor_retbytes[] = {
    0x22, 0xdf, 0xbc, 0x94, 0x62, 0xcf, 0xf9, 0x4c, 0x26, 0xd6, 0x07, 0x3c,
    0xe1, 0x09, 0x49, 0xfb, 0x5c, 0xd5, 0x3c, 0x60, 0x55, 0xd7, 0x27, 0xb1,
    0xbe, 0x79, 0x8c, 0x5b, 0x1d, 0x01, 0x8f, 0x73, 0x21, 0x39, 0xe7, 0x4f,
    0xd7, 0xdf, 0x60, 0x42, 0x24, 0x42, 0x3a, 0x2c, 0xba, 0x88, 0xdf, 0xe9,
    0x8a, 0xf8, 0x8f, 0x99, 0x77, 0xb7, 0xd7, 0x9f, 0xff, 0x08, 0xfc, 0x6c,
    0xde, 0xfc, 0xad, 0x3a, 0xc7, 0x40, 0x84, 0xba, 0xb8, 0xaf, 0x04, 0xc5,
    0xfd, 0xab, 0xc9, 0x43, 0xd5, 0x1e, 0xda, 0x0a,
};
static const struct drbg_kat_no_reseed kat35_nor_t = {
    4, kat35_nor_entropyin, kat35_nor_nonce, kat35_nor_persstr,
    kat35_nor_addin0, kat35_nor_addin1, kat35_nor_retbytes
};
static const struct drbg_kat kat35_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat35_nor_t
};

static const unsigned char kat36_nor_entropyin[] = {
    0xf8, 0xd9, 0x70, 0x58, 0x29, 0xd4, 0x90, 0x8e, 0x43, 0xd0, 0xfb, 0x62,
    0xf7, 0x2b, 0x12, 0x0a,
};
static const unsigned char kat36_nor_nonce[] = {
    0x34, 0x96, 0x18, 0xdc, 0xe5, 0x51, 0xe7, 0xc3,
};
static const unsigned char kat36_nor_persstr[] = {
    0x05, 0x4d, 0x18, 0xec, 0x3b, 0x0f, 0x5a, 0xfc, 0x46, 0xf9, 0x7b, 0xbc,
    0x0c, 0x58, 0x21, 0xe1,
};
static const unsigned char kat36_nor_addin0[] = {0};
static const unsigned char kat36_nor_addin1[] = {0};
static const unsigned char kat36_nor_retbytes[] = {
    0xa4, 0x3c, 0x01, 0xfe, 0x30, 0xb0, 0x1d, 0xd1, 0x46, 0x8e, 0x4c, 0x81,
    0xf9, 0xa6, 0xb0, 0x4f, 0x8c, 0x34, 0xf9, 0x04, 0xb2, 0x1e, 0x5d, 0xbe,
    0x18, 0x90, 0x77, 0x8c, 0x13, 0xf5, 0xb3, 0x6a, 0x66, 0x48, 0x37, 0xb9,
    0x8a, 0x1b, 0xb0, 0xf5, 0x33, 0x65, 0x3d, 0x6a, 0xc7, 0xe5, 0x37, 0xa6,
    0xa0, 0xc5, 0xc8, 0x79, 0xba, 0xf4, 0x58, 0x15, 0xec, 0xae, 0xe6, 0xb0,
    0x3a, 0x67, 0x7b, 0x2d, 0xc5, 0xcc, 0x12, 0x04, 0x2f, 0x7c, 0xa1, 0x61,
    0xbf, 0x90, 0xb2, 0xa7, 0x3c, 0xf4, 0x1d, 0x23,
};
static const struct drbg_kat_no_reseed kat36_nor_t = {
    5, kat36_nor_entropyin, kat36_nor_nonce, kat36_nor_persstr,
    kat36_nor_addin0, kat36_nor_addin1, kat36_nor_retbytes
};
static const struct drbg_kat kat36_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat36_nor_t
};

static const unsigned char kat37_nor_entropyin[] = {
    0x53, 0xe0, 0x93, 0x05, 0x5d, 0x91, 0x2f, 0x7f, 0xe9, 0x6f, 0x0b, 0xcc,
    0xc4, 0x61, 0x65, 0xc8,
};
static const unsigned char kat37_nor_nonce[] = {
    0x80, 0xf5, 0x97, 0x7f, 0x22, 0xa0, 0x96, 0xb1,
};
static const unsigned char kat37_nor_persstr[] = {
    0x92, 0xad, 0x83, 0x18, 0x7b, 0x24, 0x74, 0x8e, 0xd7, 0x7b, 0x36, 0x31,
    0x39, 0x17, 0xe9, 0x80,
};
static const unsigned char kat37_nor_addin0[] = {0};
static const unsigned char kat37_nor_addin1[] = {0};
static const unsigned char kat37_nor_retbytes[] = {
    0xe9, 0x59, 0x37, 0xe3, 0x4e, 0x5e, 0x0c, 0x54, 0x7a, 0xeb, 0x0e, 0x8d,
    0xbe, 0x3c, 0xb9, 0xba, 0x5f, 0x82, 0x39, 0x78, 0xe3, 0xa4, 0x22, 0x55,
    0xa0, 0x80, 0x3d, 0x09, 0x6b, 0xea, 0x9c, 0xd2, 0x2f, 0x38, 0x7b, 0x4d,
    0x96, 0x7e, 0xc6, 0xc6, 0xd5, 0xa6, 0x6d, 0x62, 0xf4, 0x5a, 0x83, 0x76,
    0x73, 0xc5, 0x86, 0x35, 0x1d, 0x75, 0x33, 0x47, 0x2d, 0x3b, 0xd4, 0xfa,
    0x14, 0x9d, 0x26, 0xe1, 0x5e, 0xba, 0xe9, 0x10, 0x71, 0x46, 0xb9, 0x92,
    0x19, 0xd4, 0xfa, 0xed, 0x13, 0x80, 0xa8, 0x1b,
};
static const struct drbg_kat_no_reseed kat37_nor_t = {
    6, kat37_nor_entropyin, kat37_nor_nonce, kat37_nor_persstr,
    kat37_nor_addin0, kat37_nor_addin1, kat37_nor_retbytes
};
static const struct drbg_kat kat37_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat37_nor_t
};

static const unsigned char kat38_nor_entropyin[] = {
    0xca, 0xcb, 0x0b, 0xf9, 0x34, 0xe4, 0x27, 0xbc, 0xcd, 0x07, 0xa6, 0xed,
    0x85, 0xd5, 0x51, 0xe1,
};
static const unsigned char kat38_nor_nonce[] = {
    0xbc, 0x16, 0x67, 0x6c, 0x43, 0x86, 0xff, 0xa4,
};
static const unsigned char kat38_nor_persstr[] = {
    0x5f, 0xd3, 0xb0, 0x5b, 0xea, 0xab, 0x79, 0x95, 0x37, 0x09, 0xed, 0x3c,
    0x60, 0xe7, 0x12, 0x30,
};
static const unsigned char kat38_nor_addin0[] = {0};
static const unsigned char kat38_nor_addin1[] = {0};
static const unsigned char kat38_nor_retbytes[] = {
    0x60, 0xec, 0xf6, 0xd8, 0x9a, 0xe6, 0x3b, 0x0c, 0xa3, 0xc3, 0x95, 0xec,
    0xba, 0x93, 0xfa, 0x16, 0x77, 0x76, 0xdd, 0x46, 0xfa, 0x20, 0x18, 0xf1,
    0x1b, 0x5d, 0x4c, 0xfb, 0x29, 0xce, 0x1f, 0xb1, 0x9d, 0x14, 0xca, 0x7c,
    0xe8, 0x88, 0x7c, 0x5b, 0x19, 0x5b, 0xd1, 0xbc, 0x0d, 0xf4, 0x73, 0x50,
    0x5d, 0x4b, 0x0e, 0x69, 0xa7, 0xbb, 0x54, 0xff, 0x31, 0xc4, 0xb4, 0x8f,
    0xd0, 0x6b, 0xbc, 0x78, 0x3b, 0xfa, 0x85, 0xc8, 0x71, 0x5a, 0xef, 0x86,
    0xfd, 0xee, 0xca, 0x52, 0x12, 0xa1, 0x2b, 0xac,
};
static const struct drbg_kat_no_reseed kat38_nor_t = {
    7, kat38_nor_entropyin, kat38_nor_nonce, kat38_nor_persstr,
    kat38_nor_addin0, kat38_nor_addin1, kat38_nor_retbytes
};
static const struct drbg_kat kat38_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat38_nor_t
};

static const unsigned char kat39_nor_entropyin[] = {
    0xb9, 0xb6, 0x6a, 0xea, 0x1e, 0x07, 0xb6, 0x0b, 0x30, 0xfd, 0x32, 0xff,
    0x2c, 0xdc, 0xba, 0x61,
};
static const unsigned char kat39_nor_nonce[] = {
    0x9b, 0x49, 0x7b, 0xed, 0x54, 0x94, 0x5c, 0x93,
};
static const unsigned char kat39_nor_persstr[] = {
    0xed, 0xde, 0x3a, 0xe3, 0x95, 0x78, 0xea, 0x81, 0x9f, 0x67, 0xb3, 0x8e,
    0x55, 0x2b, 0xe3, 0xb8,
};
static const unsigned char kat39_nor_addin0[] = {0};
static const unsigned char kat39_nor_addin1[] = {0};
static const unsigned char kat39_nor_retbytes[] = {
    0x7d, 0xfb, 0x2c, 0xfa, 0x07, 0x47, 0x07, 0x1c, 0x97, 0xf1, 0x1a, 0x4d,
    0xda, 0x37, 0x80, 0xf3, 0xca, 0xd2, 0x0b, 0x1c, 0x59, 0x8d, 0x3e, 0x74,
    0xb2, 0xa9, 0x0f, 0xd3, 0x9f, 0xf6, 0xb6, 0x59, 0x01, 0x2c, 0xac, 0xbf,
    0x60, 0x4b, 0x27, 0x07, 0x62, 0xbf, 0x50, 0x08, 0xa9, 0xf7, 0x79, 0xac,
    0xbf, 0xb2, 0xc4, 0xb1, 0x60, 0xd4, 0x22, 0x85, 0xd6, 0x3e, 0x5a, 0x90,
    0x19, 0x62, 0x63, 0x61, 0xbb, 0xd9, 0x8b, 0xfa, 0xd1, 0x48, 0xcc, 0x81,
    0x42, 0x6c, 0xec, 0xbc, 0x07, 0x6e, 0x92, 0x6f,
};
static const struct drbg_kat_no_reseed kat39_nor_t = {
    8, kat39_nor_entropyin, kat39_nor_nonce, kat39_nor_persstr,
    kat39_nor_addin0, kat39_nor_addin1, kat39_nor_retbytes
};
static const struct drbg_kat kat39_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat39_nor_t
};

static const unsigned char kat40_nor_entropyin[] = {
    0x81, 0x1b, 0xd6, 0x4f, 0xa9, 0x15, 0xfa, 0xdf, 0x72, 0xc1, 0x14, 0xb7,
    0xb9, 0xa3, 0xcd, 0x40,
};
static const unsigned char kat40_nor_nonce[] = {
    0x88, 0x70, 0x42, 0x0e, 0x78, 0xb3, 0x3a, 0xf8,
};
static const unsigned char kat40_nor_persstr[] = {
    0xe0, 0xdc, 0xc2, 0x80, 0x4e, 0xd0, 0x7b, 0x8f, 0x4e, 0xee, 0xa4, 0xe4,
    0x02, 0x68, 0x44, 0x3b,
};
static const unsigned char kat40_nor_addin0[] = {0};
static const unsigned char kat40_nor_addin1[] = {0};
static const unsigned char kat40_nor_retbytes[] = {
    0x7f, 0x4d, 0xbd, 0xaa, 0xce, 0xdc, 0xce, 0x57, 0x6b, 0xdf, 0xa0, 0x3a,
    0x80, 0xc3, 0x86, 0x1c, 0xb5, 0x2d, 0x8b, 0x14, 0xa6, 0xf6, 0x3e, 0x03,
    0x48, 0x68, 0xdc, 0x3f, 0xde, 0xbb, 0x5f, 0x2c, 0xd1, 0xae, 0x57, 0x90,
    0xb7, 0x55, 0xca, 0x61, 0x7c, 0x91, 0x71, 0x4b, 0xe9, 0x29, 0x02, 0xd0,
    0x08, 0x8a, 0x60, 0xd2, 0x50, 0x11, 0x23, 0x94, 0x5d, 0x53, 0x3e, 0xbc,
    0x8b, 0x9a, 0x13, 0xfc, 0x81, 0xb6, 0xcf, 0x22, 0xcc, 0x0d, 0xca, 0x1c,
    0x86, 0xf2, 0x8a, 0xa5, 0xbf, 0x38, 0xa6, 0xfa,
};
static const struct drbg_kat_no_reseed kat40_nor_t = {
    9, kat40_nor_entropyin, kat40_nor_nonce, kat40_nor_persstr,
    kat40_nor_addin0, kat40_nor_addin1, kat40_nor_retbytes
};
static const struct drbg_kat kat40_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat40_nor_t
};

static const unsigned char kat41_nor_entropyin[] = {
    0x37, 0x5a, 0xde, 0x01, 0x2a, 0xae, 0xfb, 0x48, 0x0d, 0xd9, 0xe0, 0x1d,
    0xca, 0xc6, 0xb0, 0x9d,
};
static const unsigned char kat41_nor_nonce[] = {
    0x4b, 0x52, 0x18, 0xc5, 0x85, 0x46, 0x4f, 0xa1,
};
static const unsigned char kat41_nor_persstr[] = {
    0x1d, 0x04, 0xbe, 0xea, 0xf7, 0x92, 0xcf, 0x33, 0x60, 0x13, 0xf6, 0xdf,
    0xec, 0xf8, 0xbf, 0x37,
};
static const unsigned char kat41_nor_addin0[] = {0};
static const unsigned char kat41_nor_addin1[] = {0};
static const unsigned char kat41_nor_retbytes[] = {
    0xea, 0xbf, 0xcd, 0xfc, 0x39, 0x6b, 0x5e, 0x9c, 0x10, 0xe0, 0x9b, 0x45,
    0xaa, 0x0f, 0xe9, 0x78, 0x55, 0x8a, 0xaf, 0x8a, 0x13, 0xc4, 0xce, 0x7c,
    0xdc, 0xba, 0xb0, 0x21, 0x68, 0x82, 0xd3, 0x95, 0x55, 0xb3, 0x69, 0xc6,
    0x8e, 0x62, 0x48, 0x4d, 0x1c, 0xcd, 0x51, 0xfd, 0xcc, 0x4c, 0xb6, 0x1b,
    0x22, 0xa0, 0x12, 0x01, 0x47, 0xe6, 0x68, 0x6c, 0x26, 0x3c, 0x32, 0xf6,
    0x62, 0x33, 0x27, 0x3a, 0x30, 0x11, 0xda, 0xd2, 0x42, 0x8d, 0x4d, 0x6f,
    0x5e, 0x9e, 0x20, 0xb9, 0xef, 0x64, 0x74, 0xe4,
};
static const struct drbg_kat_no_reseed kat41_nor_t = {
    10, kat41_nor_entropyin, kat41_nor_nonce, kat41_nor_persstr,
    kat41_nor_addin0, kat41_nor_addin1, kat41_nor_retbytes
};
static const struct drbg_kat kat41_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat41_nor_t
};

static const unsigned char kat42_nor_entropyin[] = {
    0x86, 0xe5, 0x9a, 0x04, 0x69, 0x83, 0x7c, 0x69, 0xff, 0xec, 0xa5, 0x66,
    0x65, 0x6f, 0x02, 0x54,
};
static const unsigned char kat42_nor_nonce[] = {
    0x67, 0x7e, 0x6c, 0xb3, 0x13, 0xa3, 0x4a, 0x19,
};
static const unsigned char kat42_nor_persstr[] = {
    0x38, 0x09, 0x01, 0x3d, 0x72, 0xfc, 0x55, 0x99, 0x01, 0xca, 0x1b, 0x06,
    0xbe, 0x58, 0x78, 0xf8,
};
static const unsigned char kat42_nor_addin0[] = {0};
static const unsigned char kat42_nor_addin1[] = {0};
static const unsigned char kat42_nor_retbytes[] = {
    0x8f, 0xa6, 0x23, 0x14, 0xbf, 0x1a, 0x1f, 0xf5, 0x84, 0x31, 0x79, 0x96,
    0xa1, 0x3a, 0x64, 0x80, 0x06, 0x95, 0x8f, 0xe8, 0xcd, 0x52, 0x63, 0xc6,
    0x1a, 0x2c, 0x68, 0x94, 0x19, 0x35, 0x8d, 0xc3, 0x56, 0xf3, 0x4e, 0x1b,
    0x19, 0x81, 0xaa, 0x8f, 0x4d, 0x18, 0x99, 0xa5, 0x1f, 0xed, 0x1a, 0x9c,
    0xfc, 0x8d, 0xb7, 0xe8, 0x4b, 0x73, 0x50, 0x63, 0x77, 0xcd, 0x12, 0x36,
    0x6b, 0xbc, 0x14, 0x40, 0x32, 0x1e, 0xf1, 0xd6, 0x4a, 0x0a, 0x6e, 0xb0,
    0xff, 0xe2, 0x54, 0x00, 0xeb, 0x82, 0xa7, 0xac,
};
static const struct drbg_kat_no_reseed kat42_nor_t = {
    11, kat42_nor_entropyin, kat42_nor_nonce, kat42_nor_persstr,
    kat42_nor_addin0, kat42_nor_addin1, kat42_nor_retbytes
};
static const struct drbg_kat kat42_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat42_nor_t
};

static const unsigned char kat43_nor_entropyin[] = {
    0x66, 0xd1, 0xab, 0x7b, 0x31, 0x15, 0x24, 0x93, 0x61, 0xdb, 0x41, 0x48,
    0x25, 0x38, 0xd1, 0x42,
};
static const unsigned char kat43_nor_nonce[] = {
    0xa1, 0x7c, 0xbb, 0xde, 0x72, 0x00, 0xd2, 0xbe,
};
static const unsigned char kat43_nor_persstr[] = {
    0xaa, 0xd7, 0xb6, 0xa9, 0xde, 0x71, 0x7a, 0xcc, 0x17, 0xfa, 0x7c, 0xed,
    0xc2, 0x85, 0x8e, 0xa9,
};
static const unsigned char kat43_nor_addin0[] = {0};
static const unsigned char kat43_nor_addin1[] = {0};
static const unsigned char kat43_nor_retbytes[] = {
    0x18, 0xe3, 0xfa, 0x3d, 0x86, 0x1a, 0xb8, 0x10, 0xe4, 0xfc, 0xd0, 0x52,
    0x06, 0xb3, 0xec, 0x49, 0x87, 0xc1, 0x60, 0x00, 0xf6, 0xa8, 0x4a, 0x28,
    0xe2, 0xe8, 0x93, 0x87, 0xb4, 0x01, 0x0c, 0x84, 0x0f, 0x79, 0x64, 0x84,
    0xd4, 0x73, 0x0c, 0x5b, 0x3a, 0x46, 0xe3, 0x7c, 0x99, 0xe5, 0xe1, 0xea,
    0x02, 0xa6, 0xd1, 0x53, 0xd5, 0x0e, 0x97, 0x19, 0xe0, 0x41, 0x9e, 0x58,
    0xe9, 0xf6, 0x25, 0x4b, 0xa3, 0xb9, 0xca, 0x80, 0xbb, 0x1b, 0x27, 0xf7,
    0x0a, 0x05, 0xc9, 0x44, 0xcb, 0xbf, 0x9e, 0x89,
};
static const struct drbg_kat_no_reseed kat43_nor_t = {
    12, kat43_nor_entropyin, kat43_nor_nonce, kat43_nor_persstr,
    kat43_nor_addin0, kat43_nor_addin1, kat43_nor_retbytes
};
static const struct drbg_kat kat43_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat43_nor_t
};

static const unsigned char kat44_nor_entropyin[] = {
    0x61, 0x49, 0x06, 0x0c, 0x71, 0xc6, 0xd6, 0xfa, 0xc0, 0xd3, 0x08, 0x83,
    0x69, 0xa9, 0x62, 0x8e,
};
static const unsigned char kat44_nor_nonce[] = {
    0x6a, 0x97, 0xf8, 0x19, 0x9b, 0x36, 0xb3, 0xca,
};
static const unsigned char kat44_nor_persstr[] = {
    0x23, 0x1c, 0x6f, 0x78, 0x38, 0x8f, 0xb6, 0x3c, 0xfe, 0x94, 0x22, 0xde,
    0xd4, 0x1c, 0xaa, 0x7a,
};
static const unsigned char kat44_nor_addin0[] = {0};
static const unsigned char kat44_nor_addin1[] = {0};
static const unsigned char kat44_nor_retbytes[] = {
    0x9d, 0x7c, 0xea, 0x99, 0x27, 0xe5, 0xe9, 0x55, 0x2d, 0xe9, 0xda, 0x9a,
    0x8c, 0x29, 0x1b, 0x94, 0x00, 0xe3, 0x8f, 0xb3, 0x2c, 0x30, 0xdc, 0x33,
    0x4f, 0x57, 0x2b, 0xac, 0xe0, 0x72, 0x81, 0x5b, 0xcc, 0x48, 0x06, 0xfc,
    0x07, 0x7b, 0x7a, 0xe6, 0xe4, 0xc8, 0x2e, 0xd9, 0xa8, 0x0b, 0x28, 0x1f,
    0xe1, 0x86, 0x81, 0x8c, 0x38, 0xbc, 0x1d, 0x0f, 0x16, 0xce, 0x05, 0xee,
    0x64, 0x44, 0x50, 0xd2, 0x0d, 0xe5, 0xec, 0xdb, 0xa5, 0x75, 0xe4, 0x36,
    0xac, 0xb8, 0x52, 0x62, 0x30, 0xcd, 0x8b, 0xce,
};
static const struct drbg_kat_no_reseed kat44_nor_t = {
    13, kat44_nor_entropyin, kat44_nor_nonce, kat44_nor_persstr,
    kat44_nor_addin0, kat44_nor_addin1, kat44_nor_retbytes
};
static const struct drbg_kat kat44_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat44_nor_t
};

static const unsigned char kat45_nor_entropyin[] = {
    0xf3, 0x68, 0xb8, 0x15, 0x01, 0x33, 0x8a, 0x3c, 0xca, 0x94, 0x83, 0x53,
    0xfe, 0x16, 0x97, 0x0d,
};
static const unsigned char kat45_nor_nonce[] = {
    0xfa, 0x9a, 0xda, 0xe9, 0x24, 0x41, 0x71, 0x50,
};
static const unsigned char kat45_nor_persstr[] = {
    0xdb, 0xad, 0x22, 0xc3, 0x89, 0xc5, 0x27, 0x71, 0x5d, 0x21, 0xa5, 0xbd,
    0xf3, 0x8c, 0x1f, 0xad,
};
static const unsigned char kat45_nor_addin0[] = {0};
static const unsigned char kat45_nor_addin1[] = {0};
static const unsigned char kat45_nor_retbytes[] = {
    0xa1, 0x8d, 0x57, 0xe6, 0x72, 0x21, 0x89, 0x56, 0xe6, 0xc8, 0xcb, 0x99,
    0x01, 0xd0, 0x28, 0x88, 0xf3, 0x58, 0x71, 0x77, 0xc3, 0xe1, 0x1e, 0x1a,
    0x99, 0xea, 0x72, 0x37, 0x03, 0x47, 0xb9, 0x53, 0xa9, 0xf1, 0x22, 0xc9,
    0x44, 0x6d, 0xfa, 0x10, 0x97, 0x23, 0xb2, 0x7f, 0x36, 0xfb, 0xf1, 0x5e,
    0xdf, 0x10, 0x3a, 0x56, 0x74, 0x1c, 0x24, 0x96, 0x85, 0x92, 0x47, 0x9c,
    0xfe, 0x30, 0xbc, 0x00, 0x53, 0xfa, 0x7b, 0x98, 0x18, 0xe9, 0xde, 0xbc,
    0xc4, 0x94, 0xdb, 0x64, 0xd1, 0x5d, 0x03, 0x8b,
};
static const struct drbg_kat_no_reseed kat45_nor_t = {
    14, kat45_nor_entropyin, kat45_nor_nonce, kat45_nor_persstr,
    kat45_nor_addin0, kat45_nor_addin1, kat45_nor_retbytes
};
static const struct drbg_kat kat45_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat45_nor_t
};

static const unsigned char kat46_nor_entropyin[] = {
    0x64, 0x66, 0xe1, 0x79, 0x9a, 0x68, 0x01, 0x23, 0x79, 0x63, 0x1b, 0x3a,
    0xae, 0x41, 0xf5, 0x9b,
};
static const unsigned char kat46_nor_nonce[] = {
    0x6b, 0x0c, 0x61, 0x26, 0x9f, 0x67, 0xc5, 0x76,
};
static const unsigned char kat46_nor_persstr[] = {
    0xcc, 0x93, 0x6b, 0x87, 0xc8, 0xc8, 0xc1, 0xab, 0x85, 0xdd, 0xe0, 0xad,
    0x2e, 0x92, 0x42, 0xb4,
};
static const unsigned char kat46_nor_addin0[] = {
    0xd1, 0x03, 0x3a, 0xc5, 0x53, 0xef, 0x08, 0xf2, 0x2f, 0xd3, 0x8f, 0x12,
    0xb4, 0x9b, 0x45, 0xbc,
};
static const unsigned char kat46_nor_addin1[] = {
    0xf0, 0x04, 0xba, 0x01, 0xf5, 0x14, 0x55, 0x43, 0x0d, 0x84, 0x36, 0x2e,
    0x37, 0x6e, 0xb7, 0x75,
};
static const unsigned char kat46_nor_retbytes[] = {
    0x5d, 0x67, 0x5d, 0x1e, 0x92, 0x49, 0x09, 0x52, 0x70, 0x3c, 0x19, 0x41,
    0x94, 0xe1, 0xb0, 0x61, 0xb6, 0xec, 0x4e, 0x21, 0x9d, 0xc2, 0xe1, 0xed,
    0xaa, 0x89, 0x1e, 0xf2, 0xd1, 0xb7, 0xed, 0x05, 0x0a, 0x06, 0x34, 0x2d,
    0x3c, 0x09, 0x50, 0x11, 0xeb, 0x33, 0x9f, 0x19, 0x85, 0x19, 0x77, 0x9b,
    0x01, 0xab, 0x1a, 0x58, 0x0b, 0xd2, 0xe3, 0x4d, 0x6c, 0xf4, 0xe4, 0x7c,
    0x1b, 0xef, 0xe0, 0xc7, 0xdc, 0x37, 0xb4, 0xaa, 0xfb, 0x31, 0x12, 0x8f,
    0xa3, 0x96, 0x26, 0x7f, 0x37, 0x32, 0x09, 0x5a,
};
static const struct drbg_kat_no_reseed kat46_nor_t = {
    0, kat46_nor_entropyin, kat46_nor_nonce, kat46_nor_persstr,
    kat46_nor_addin0, kat46_nor_addin1, kat46_nor_retbytes
};
static const struct drbg_kat kat46_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat46_nor_t
};

static const unsigned char kat47_nor_entropyin[] = {
    0x7e, 0xd2, 0x84, 0xc8, 0x59, 0x6b, 0x6f, 0x01, 0x07, 0xde, 0x3b, 0x4b,
    0x95, 0xcd, 0x91, 0x14,
};
static const unsigned char kat47_nor_nonce[] = {
    0x69, 0x11, 0x6c, 0x8a, 0x07, 0x3b, 0xf4, 0xf4,
};
static const unsigned char kat47_nor_persstr[] = {
    0xaf, 0x66, 0x8e, 0xb6, 0x5b, 0x0f, 0x4d, 0xf2, 0xed, 0x49, 0xb3, 0x7c,
    0x84, 0x2b, 0x27, 0x2e,
};
static const unsigned char kat47_nor_addin0[] = {
    0xf0, 0xdd, 0xc4, 0x59, 0x22, 0x36, 0x3f, 0x40, 0x27, 0x1e, 0x7a, 0x01,
    0xed, 0x67, 0xba, 0x84,
};
static const unsigned char kat47_nor_addin1[] = {
    0xc2, 0xda, 0x95, 0xcb, 0xfe, 0xfd, 0x2a, 0xc7, 0x39, 0x5c, 0xf9, 0x89,
    0x71, 0x1c, 0x13, 0x50,
};
static const unsigned char kat47_nor_retbytes[] = {
    0x30, 0x6f, 0x55, 0x30, 0x77, 0x03, 0x4e, 0xc2, 0xa8, 0x6a, 0x64, 0xcd,
    0xf3, 0x43, 0x87, 0xc5, 0x97, 0x69, 0xaa, 0xe1, 0x49, 0xd8, 0xcd, 0xb0,
    0x04, 0x1a, 0x5c, 0x2e, 0x6d, 0xf9, 0xdd, 0x1e, 0x00, 0xef, 0x08, 0x98,
    0x17, 0x9d, 0x6c, 0x6b, 0xf0, 0x2b, 0x2c, 0x42, 0xb4, 0xa8, 0x98, 0xf4,
    0x8c, 0xdd, 0x9b, 0xa4, 0xad, 0x2b, 0x99, 0xfb, 0x13, 0x3c, 0x5a, 0xce,
    0x42, 0x49, 0xa6, 0x7b, 0xc3, 0x0e, 0x64, 0x52, 0x49, 0x5c, 0x18, 0x62,
    0x44, 0xdc, 0x5a, 0xc6, 0x92, 0x8b, 0x1f, 0xf3,
};
static const struct drbg_kat_no_reseed kat47_nor_t = {
    1, kat47_nor_entropyin, kat47_nor_nonce, kat47_nor_persstr,
    kat47_nor_addin0, kat47_nor_addin1, kat47_nor_retbytes
};
static const struct drbg_kat kat47_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat47_nor_t
};

static const unsigned char kat48_nor_entropyin[] = {
    0x48, 0xfe, 0xfe, 0x99, 0x7c, 0xef, 0x3b, 0xa2, 0xb7, 0x3e, 0x76, 0xaa,
    0xf0, 0x03, 0x59, 0x0a,
};
static const unsigned char kat48_nor_nonce[] = {
    0x19, 0xdb, 0x21, 0xe0, 0xd0, 0x5d, 0x99, 0xb9,
};
static const unsigned char kat48_nor_persstr[] = {
    0xd8, 0x17, 0x59, 0xf6, 0xa2, 0x89, 0x2c, 0x5b, 0x2d, 0x51, 0xb8, 0x5f,
    0x94, 0xd8, 0x98, 0x72,
};
static const unsigned char kat48_nor_addin0[] = {
    0x70, 0x08, 0x42, 0x84, 0x15, 0x73, 0xe6, 0xae, 0xfc, 0x1f, 0xcf, 0x20,
    0xaa, 0x87, 0x74, 0x16,
};
static const unsigned char kat48_nor_addin1[] = {
    0xa9, 0x1e, 0xfe, 0x14, 0x15, 0xec, 0xfe, 0xe7, 0x1d, 0xf0, 0x68, 0xa1,
    0x6b, 0xaf, 0x03, 0x4d,
};
static const unsigned char kat48_nor_retbytes[] = {
    0xe6, 0xbe, 0xd7, 0x8b, 0xe0, 0x92, 0x15, 0xd9, 0xd1, 0xbe, 0xb0, 0xb8,
    0xb0, 0x88, 0x9e, 0x67, 0x79, 0x62, 0xa6, 0x1b, 0xf9, 0x01, 0xdf, 0x09,
    0xad, 0x8e, 0x36, 0xce, 0xec, 0x05, 0x25, 0x88, 0xb5, 0x7c, 0xdd, 0x44,
    0x5c, 0xe5, 0xa2, 0xc5, 0x7f, 0xd4, 0x0b, 0x74, 0x32, 0x86, 0x0f, 0xa3,
    0xc1, 0xc9, 0x78, 0x47, 0xd7, 0x90, 0xab, 0xd5, 0xd7, 0x04, 0xee, 0xfc,
    0x12, 0xff, 0x16, 0x36, 0xa6, 0xc9, 0x99, 0xc4, 0xda, 0xe9, 0x40, 0x0a,
    0x7c, 0x96, 0x8e, 0x84, 0xbc, 0x7e, 0xc3, 0xe6,
};
static const struct drbg_kat_no_reseed kat48_nor_t = {
    2, kat48_nor_entropyin, kat48_nor_nonce, kat48_nor_persstr,
    kat48_nor_addin0, kat48_nor_addin1, kat48_nor_retbytes
};
static const struct drbg_kat kat48_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat48_nor_t
};

static const unsigned char kat49_nor_entropyin[] = {
    0x31, 0xc4, 0x90, 0x0e, 0x6e, 0x37, 0x25, 0x0a, 0x80, 0xdb, 0x3b, 0x59,
    0xcb, 0xaa, 0x10, 0xee,
};
static const unsigned char kat49_nor_nonce[] = {
    0x8e, 0x19, 0x4c, 0x49, 0x4a, 0x41, 0x9d, 0x9e,
};
static const unsigned char kat49_nor_persstr[] = {
    0x12, 0x16, 0x74, 0x89, 0x8e, 0x64, 0x77, 0x5d, 0x34, 0x4f, 0x02, 0x2e,
    0xa6, 0x92, 0xd9, 0xe7,
};
static const unsigned char kat49_nor_addin0[] = {
    0x06, 0x62, 0x0c, 0x19, 0x98, 0xde, 0x33, 0x31, 0xb9, 0x2f, 0x24, 0x09,
    0xe8, 0x63, 0x58, 0x61,
};
static const unsigned char kat49_nor_addin1[] = {
    0xe0, 0xbc, 0x04, 0x53, 0x44, 0xe4, 0x59, 0x27, 0x36, 0x4b, 0x27, 0xd9,
    0xfc, 0xa2, 0xc5, 0x3f,
};
static const unsigned char kat49_nor_retbytes[] = {
    0x1b, 0x6e, 0x68, 0xec, 0x91, 0xba, 0x97, 0x08, 0x56, 0x2a, 0xaf, 0x2c,
    0xe6, 0xe3, 0xb4, 0x99, 0xf4, 0x2a, 0x1b, 0x72, 0xb1, 0x4e, 0x5c, 0x20,
    0xa8, 0x8e, 0x37, 0x3b, 0x85, 0xcb, 0xff, 0xc9, 0x24, 0xda, 0x24, 0xb4,
    0xc0, 0x88, 0x36, 0xb7, 0xa8, 0x49, 0x27, 0x06, 0x4e, 0xb0, 0xcc, 0x82,
    0x4f, 0x2d, 0xfa, 0x0f, 0x5a, 0xa8, 0xd4, 0x80, 0xcb, 0xbf, 0x7c, 0xa7,
    0x50, 0x8d, 0x96, 0x37, 0x0d, 0x92, 0xa5, 0x9f, 0x81, 0x37, 0xea, 0x39,
    0x16, 0x7a, 0x37, 0xda, 0x7f, 0x75, 0xc8, 0x9b,
};
static const struct drbg_kat_no_reseed kat49_nor_t = {
    3, kat49_nor_entropyin, kat49_nor_nonce, kat49_nor_persstr,
    kat49_nor_addin0, kat49_nor_addin1, kat49_nor_retbytes
};
static const struct drbg_kat kat49_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat49_nor_t
};

static const unsigned char kat50_nor_entropyin[] = {
    0xe8, 0xb4, 0x1a, 0x5a, 0xf5, 0xc4, 0x51, 0x52, 0x82, 0xf3, 0x07, 0x86,
    0x88, 0xbe, 0x48, 0xa2,
};
static const unsigned char kat50_nor_nonce[] = {
    0x4c, 0x3b, 0x5a, 0x55, 0x9c, 0xec, 0xa5, 0x1d,
};
static const unsigned char kat50_nor_persstr[] = {
    0xcf, 0x1a, 0xd8, 0xf0, 0x83, 0x98, 0x17, 0x06, 0x1b, 0xa5, 0x34, 0xe0,
    0x04, 0xf6, 0x13, 0x11,
};
static const unsigned char kat50_nor_addin0[] = {
    0x3c, 0x78, 0x3a, 0xa5, 0xe0, 0x96, 0xf5, 0x30, 0x5d, 0x84, 0xaa, 0xef,
    0x0d, 0xe0, 0x86, 0x5e,
};
static const unsigned char kat50_nor_addin1[] = {
    0x62, 0x27, 0x1e, 0x83, 0x41, 0x72, 0x5f, 0xad, 0xfc, 0x41, 0xef, 0x13,
    0x64, 0x92, 0xc6, 0xb3,
};
static const unsigned char kat50_nor_retbytes[] = {
    0x48, 0xa1, 0x17, 0x29, 0xd1, 0x98, 0xf9, 0x08, 0x26, 0x90, 0x65, 0xbc,
    0x85, 0x2d, 0x06, 0x2b, 0x93, 0x32, 0x04, 0x75, 0x35, 0xc0, 0x4d, 0x8c,
    0x0a, 0xd0, 0xb6, 0xb9, 0x9e, 0x9a, 0x90, 0xe6, 0x86, 0xdf, 0xca, 0x34,
    0x4d, 0xf6, 0x8c, 0x7b, 0xb7, 0x2e, 0xcb, 0xc7, 0x39, 0x63, 0x45, 0x19,
    0x07, 0x7f, 0xa6, 0xf1, 0xf0, 0x0e, 0x8d, 0xf4, 0xe3, 0x71, 0x2c, 0xfa,
    0x79, 0xef, 0x4d, 0x2d, 0x7e, 0x95, 0x00, 0x85, 0x99, 0x4b, 0xd5, 0x53,
    0x50, 0x2f, 0x67, 0x3f, 0x8f, 0xfe, 0xc1, 0x7a,
};
static const struct drbg_kat_no_reseed kat50_nor_t = {
    4, kat50_nor_entropyin, kat50_nor_nonce, kat50_nor_persstr,
    kat50_nor_addin0, kat50_nor_addin1, kat50_nor_retbytes
};
static const struct drbg_kat kat50_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat50_nor_t
};

static const unsigned char kat51_nor_entropyin[] = {
    0xe9, 0x00, 0xd8, 0xf9, 0x80, 0x3f, 0x7f, 0xde, 0xb4, 0xfe, 0x52, 0x22,
    0xd9, 0x9c, 0xe8, 0xb5,
};
static const unsigned char kat51_nor_nonce[] = {
    0xa5, 0x41, 0x2a, 0x73, 0xd5, 0x9f, 0xfd, 0x2f,
};
static const unsigned char kat51_nor_persstr[] = {
    0x10, 0x52, 0x3f, 0x20, 0xa6, 0x90, 0x46, 0x5f, 0x73, 0x39, 0xf6, 0x6f,
    0x8d, 0x91, 0xf9, 0xaf,
};
static const unsigned char kat51_nor_addin0[] = {
    0x75, 0x4f, 0xb2, 0xb6, 0x2c, 0xe1, 0x79, 0x1c, 0x0a, 0x17, 0x31, 0xec,
    0xa8, 0xd0, 0x06, 0xdc,
};
static const unsigned char kat51_nor_addin1[] = {
    0x66, 0xa8, 0xfa, 0x07, 0x27, 0xf4, 0xf8, 0x80, 0xaf, 0xd4, 0x00, 0x65,
    0x74, 0xa5, 0x1b, 0xd8,
};
static const unsigned char kat51_nor_retbytes[] = {
    0xa7, 0x2c, 0x09, 0xf2, 0x3d, 0xe8, 0xcd, 0xab, 0x9d, 0x60, 0x67, 0xb8,
    0xee, 0x0d, 0xbc, 0xbb, 0xa9, 0x73, 0x06, 0xad, 0x71, 0x18, 0x60, 0xf7,
    0x95, 0x8d, 0x4c, 0x58, 0x89, 0xe6, 0x15, 0x49, 0xe3, 0x0f, 0xea, 0x41,
    0x5a, 0xbc, 0xb6, 0x21, 0x1d, 0x0e, 0xbd, 0xdd, 0x27, 0xda, 0x0c, 0xa1,
    0x10, 0x4a, 0x4c, 0x58, 0x54, 0x86, 0xea, 0x49, 0x5e, 0xde, 0xa8, 0x72,
    0x57, 0xd2, 0xb7, 0x54, 0x62, 0x50, 0x45, 0x38, 0xf2, 0x1d, 0x18, 0x03,
    0xf2, 0x67, 0x91, 0x28, 0xf8, 0xfd, 0xab, 0x7a,
};
static const struct drbg_kat_no_reseed kat51_nor_t = {
    5, kat51_nor_entropyin, kat51_nor_nonce, kat51_nor_persstr,
    kat51_nor_addin0, kat51_nor_addin1, kat51_nor_retbytes
};
static const struct drbg_kat kat51_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat51_nor_t
};

static const unsigned char kat52_nor_entropyin[] = {
    0x39, 0x5b, 0x1d, 0xa9, 0x17, 0x18, 0x83, 0x71, 0xad, 0xbb, 0x0d, 0x50,
    0x27, 0xdf, 0xc2, 0xd8,
};
static const unsigned char kat52_nor_nonce[] = {
    0x77, 0x39, 0xdd, 0x16, 0x0f, 0x53, 0xbf, 0x02,
};
static const unsigned char kat52_nor_persstr[] = {
    0x3e, 0x86, 0x4c, 0x12, 0x87, 0xdc, 0x15, 0xee, 0x2c, 0x58, 0x5f, 0xae,
    0xd1, 0x22, 0x26, 0xc5,
};
static const unsigned char kat52_nor_addin0[] = {
    0xb4, 0x06, 0xd8, 0xb8, 0x9f, 0x3d, 0x60, 0x25, 0x60, 0x25, 0x7f, 0xa4,
    0x76, 0x03, 0x16, 0xe1,
};
static const unsigned char kat52_nor_addin1[] = {
    0x94, 0xb7, 0x2e, 0x35, 0x5f, 0xd1, 0x2a, 0x08, 0x34, 0xcd, 0x24, 0x48,
    0xf4, 0x34, 0x3d, 0x16,
};
static const unsigned char kat52_nor_retbytes[] = {
    0xe0, 0xa3, 0xa7, 0xc3, 0x6b, 0x94, 0x61, 0xa0, 0x3f, 0x75, 0x35, 0x6e,
    0x58, 0x8b, 0x64, 0x34, 0xc1, 0xea, 0x08, 0xd9, 0x16, 0x83, 0x58, 0xc8,
    0x28, 0x7e, 0x7a, 0x3d, 0x99, 0xfc, 0x15, 0x53, 0x0e, 0x15, 0x61, 0xd3,
    0x12, 0xc7, 0x41, 0xdf, 0x9e, 0x8f, 0xac, 0x58, 0x29, 0x9e, 0xbd, 0xe1,
    0x8f, 0xc9, 0x0a, 0xd8, 0xa9, 0x52, 0xc8, 0x98, 0x4e, 0xf3, 0x1d, 0xcd,
    0xc9, 0xa6, 0x02, 0x3d, 0x5e, 0xdb, 0x4b, 0xc3, 0x93, 0xb3, 0x79, 0xfe,
    0xb9, 0xa9, 0x18, 0x89, 0x8d, 0x8c, 0x46, 0x8e,
};
static const struct drbg_kat_no_reseed kat52_nor_t = {
    6, kat52_nor_entropyin, kat52_nor_nonce, kat52_nor_persstr,
    kat52_nor_addin0, kat52_nor_addin1, kat52_nor_retbytes
};
static const struct drbg_kat kat52_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat52_nor_t
};

static const unsigned char kat53_nor_entropyin[] = {
    0x0a, 0xcd, 0x1d, 0xa3, 0xcf, 0xeb, 0xd6, 0xc6, 0x6d, 0x4f, 0x2a, 0xc3,
    0x6a, 0xec, 0x6d, 0xee,
};
static const unsigned char kat53_nor_nonce[] = {
    0xfb, 0xa4, 0xc2, 0x00, 0xff, 0x69, 0x1b, 0xcc,
};
static const unsigned char kat53_nor_persstr[] = {
    0x6a, 0xfc, 0xa0, 0x03, 0x4d, 0xbc, 0xf6, 0x32, 0xc8, 0x15, 0xe3, 0x3e,
    0x69, 0xf7, 0x33, 0x40,
};
static const unsigned char kat53_nor_addin0[] = {
    0x0f, 0x3d, 0xea, 0xdd, 0xdf, 0x2c, 0x53, 0xbf, 0xee, 0xdb, 0x47, 0xfe,
    0xd1, 0x0c, 0xf4, 0xe3,
};
static const unsigned char kat53_nor_addin1[] = {
    0x32, 0x6e, 0xd6, 0x10, 0xfe, 0x03, 0x8f, 0x5f, 0x16, 0x8c, 0xf7, 0xee,
    0x4a, 0x1e, 0xf3, 0x8e,
};
static const unsigned char kat53_nor_retbytes[] = {
    0xa6, 0x5e, 0xf7, 0x22, 0xc9, 0x17, 0x21, 0xba, 0xf5, 0xe6, 0x09, 0xea,
    0x12, 0x9a, 0x32, 0x11, 0xd0, 0x5c, 0x53, 0x67, 0x7a, 0x57, 0x6a, 0xc5,
    0x55, 0x2e, 0x0d, 0x03, 0x2c, 0xa1, 0x72, 0x28, 0xa1, 0xa6, 0xce, 0xdc,
    0x41, 0xa0, 0x6e, 0x5a, 0xdd, 0x7d, 0x9c, 0x44, 0xc4, 0x11, 0x10, 0xaf,
    0xdf, 0x5f, 0xa2, 0x7f, 0x9c, 0xd4, 0xb4, 0xfd, 0x1b, 0x65, 0xc7, 0xa3,
    0x46, 0xc3, 0x9f, 0xc3, 0xc2, 0x2d, 0x1e, 0x04, 0x2e, 0x39, 0x6a, 0xe1,
    0x96, 0x28, 0xb2, 0xee, 0x4f, 0x3c, 0xb5, 0xd9,
};
static const struct drbg_kat_no_reseed kat53_nor_t = {
    7, kat53_nor_entropyin, kat53_nor_nonce, kat53_nor_persstr,
    kat53_nor_addin0, kat53_nor_addin1, kat53_nor_retbytes
};
static const struct drbg_kat kat53_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat53_nor_t
};

static const unsigned char kat54_nor_entropyin[] = {
    0xe9, 0x49, 0x29, 0x67, 0xa0, 0x6e, 0x84, 0xc8, 0x0b, 0xd0, 0x49, 0xfe,
    0x9f, 0x2d, 0x08, 0xa7,
};
static const unsigned char kat54_nor_nonce[] = {
    0x28, 0x7d, 0x30, 0xdc, 0x97, 0x2f, 0xfc, 0x8c,
};
static const unsigned char kat54_nor_persstr[] = {
    0xfa, 0x60, 0xa8, 0x94, 0xd9, 0x95, 0x5f, 0xd1, 0x4d, 0x36, 0x8a, 0xef,
    0xcb, 0xe5, 0x0e, 0x67,
};
static const unsigned char kat54_nor_addin0[] = {
    0xba, 0xe9, 0xd7, 0x92, 0xa1, 0xd6, 0x67, 0xfa, 0xdb, 0xee, 0xfe, 0x98,
    0xd6, 0x45, 0x78, 0x0e,
};
static const unsigned char kat54_nor_addin1[] = {
    0x41, 0x82, 0x02, 0x16, 0x0d, 0x92, 0xcb, 0x59, 0xd1, 0xe7, 0x37, 0x88,
    0x38, 0x64, 0x3c, 0x1e,
};
static const unsigned char kat54_nor_retbytes[] = {
    0xa6, 0xdb, 0xef, 0xea, 0xf6, 0xf8, 0x59, 0xd7, 0xac, 0x41, 0x37, 0xaf,
    0x79, 0x8c, 0xaa, 0x8f, 0xbb, 0x4f, 0xa2, 0xfb, 0xe4, 0x96, 0x98, 0x6c,
    0xdc, 0x8c, 0x86, 0xba, 0xbc, 0x72, 0xb1, 0x65, 0x5e, 0x98, 0xe5, 0x2c,
    0x40, 0xf6, 0x66, 0x81, 0xc4, 0xd4, 0xcb, 0x13, 0x19, 0xb3, 0xcf, 0xa9,
    0x7d, 0xa7, 0x33, 0x33, 0x10, 0x9c, 0xe4, 0x14, 0xce, 0x41, 0xc9, 0x3b,
    0x1e, 0x62, 0x76, 0x7f, 0xd2, 0x6a, 0x75, 0x7e, 0xb8, 0x2d, 0xce, 0x2d,
    0x44, 0x54, 0xf8, 0xe9, 0xdf, 0xfb, 0x79, 0x1f,
};
static const struct drbg_kat_no_reseed kat54_nor_t = {
    8, kat54_nor_entropyin, kat54_nor_nonce, kat54_nor_persstr,
    kat54_nor_addin0, kat54_nor_addin1, kat54_nor_retbytes
};
static const struct drbg_kat kat54_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat54_nor_t
};

static const unsigned char kat55_nor_entropyin[] = {
    0x3a, 0x65, 0x5b, 0x77, 0x52, 0x7c, 0x15, 0x24, 0xaa, 0x13, 0xc7, 0x1a,
    0x3d, 0x9e, 0x19, 0x16,
};
static const unsigned char kat55_nor_nonce[] = {
    0x66, 0x12, 0xc7, 0x46, 0xe4, 0xbc, 0xc0, 0xef,
};
static const unsigned char kat55_nor_persstr[] = {
    0x8a, 0x1c, 0x40, 0x2e, 0x8a, 0xbc, 0x71, 0xff, 0x11, 0xd7, 0xe4, 0x73,
    0x8f, 0x14, 0xb0, 0x0f,
};
static const unsigned char kat55_nor_addin0[] = {
    0x39, 0xaf, 0x3c, 0xe9, 0x9e, 0xb8, 0xe9, 0xea, 0x59, 0x64, 0x31, 0x0c,
    0x27, 0xc5, 0x06, 0x73,
};
static const unsigned char kat55_nor_addin1[] = {
    0xb3, 0x35, 0x3b, 0xf5, 0x22, 0xfd, 0xf8, 0xca, 0x87, 0xe8, 0x40, 0x66,
    0x5d, 0xb7, 0x3a, 0x1e,
};
static const unsigned char kat55_nor_retbytes[] = {
    0x2b, 0xe2, 0x93, 0x38, 0xc9, 0x15, 0xbd, 0x42, 0x39, 0x7d, 0x73, 0x90,
    0x7d, 0x70, 0xf5, 0xd9, 0x5b, 0xa9, 0xbf, 0xf4, 0x99, 0xd1, 0x1f, 0x7a,
    0x27, 0x39, 0x9d, 0xb6, 0x64, 0x70, 0xe2, 0x65, 0x98, 0xcd, 0xac, 0xb5,
    0xab, 0x06, 0xd1, 0xd1, 0x37, 0x57, 0x04, 0x3c, 0x8d, 0xbf, 0x56, 0xa0,
    0x5d, 0x30, 0x39, 0x2c, 0xc3, 0xb2, 0x1f, 0x42, 0x10, 0xcc, 0x14, 0x16,
    0x52, 0xea, 0x42, 0xb0, 0x2e, 0x4b, 0x95, 0x7c, 0x84, 0x6e, 0xfd, 0x21,
    0x4a, 0x4b, 0xe0, 0xf0, 0x15, 0xb5, 0x14, 0xd0,
};
static const struct drbg_kat_no_reseed kat55_nor_t = {
    9, kat55_nor_entropyin, kat55_nor_nonce, kat55_nor_persstr,
    kat55_nor_addin0, kat55_nor_addin1, kat55_nor_retbytes
};
static const struct drbg_kat kat55_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat55_nor_t
};

static const unsigned char kat56_nor_entropyin[] = {
    0x12, 0x42, 0x0e, 0xb0, 0x2f, 0xd4, 0x38, 0x32, 0x63, 0x81, 0x49, 0x68,
    0xdc, 0xf6, 0x07, 0x91,
};
static const unsigned char kat56_nor_nonce[] = {
    0x37, 0x7a, 0x3f, 0xca, 0xa2, 0x67, 0x0b, 0x8b,
};
static const unsigned char kat56_nor_persstr[] = {
    0x4e, 0xfc, 0x59, 0x26, 0xbd, 0x99, 0x9a, 0xd4, 0x6c, 0x30, 0xe9, 0xbc,
    0x72, 0x96, 0xb2, 0xba,
};
static const unsigned char kat56_nor_addin0[] = {
    0x5c, 0x68, 0xc0, 0xf0, 0x79, 0x11, 0x04, 0x76, 0xc1, 0x48, 0x77, 0xc5,
    0xfd, 0xed, 0x2b, 0x7f,
};
static const unsigned char kat56_nor_addin1[] = {
    0x3f, 0x17, 0x47, 0xb2, 0xa6, 0x4b, 0xb3, 0x71, 0xeb, 0x56, 0x6a, 0x02,
    0x04, 0xc8, 0x2e, 0x7b,
};
static const unsigned char kat56_nor_retbytes[] = {
    0x1d, 0x73, 0x39, 0x29, 0x28, 0x43, 0x8d, 0xa9, 0xe9, 0xf7, 0xa2, 0x8f,
    0xff, 0x33, 0x32, 0x74, 0x6f, 0x7f, 0xf9, 0x05, 0x58, 0xc9, 0x87, 0x38,
    0x33, 0x05, 0xc8, 0xdb, 0x81, 0x75, 0xc9, 0x01, 0x2d, 0xeb, 0xda, 0xf7,
    0x36, 0x59, 0xfd, 0xb4, 0xc4, 0x04, 0x17, 0xb5, 0xd8, 0xda, 0xa4, 0xfe,
    0xf8, 0xb4, 0x26, 0x96, 0x35, 0x9f, 0x05, 0x58, 0x45, 0x42, 0xd1, 0x47,
    0x21, 0xd6, 0x8f, 0x8c, 0xa7, 0x80, 0x7b, 0x75, 0x1d, 0x64, 0x9e, 0xf2,
    0x4e, 0xb4, 0x29, 0x46, 0xc1, 0xc5, 0x00, 0x3c,
};
static const struct drbg_kat_no_reseed kat56_nor_t = {
    10, kat56_nor_entropyin, kat56_nor_nonce, kat56_nor_persstr,
    kat56_nor_addin0, kat56_nor_addin1, kat56_nor_retbytes
};
static const struct drbg_kat kat56_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat56_nor_t
};

static const unsigned char kat57_nor_entropyin[] = {
    0x84, 0x7b, 0x94, 0xcb, 0xbf, 0xf8, 0xbc, 0xe7, 0x09, 0x53, 0x0a, 0xa0,
    0x19, 0xd1, 0x02, 0x81,
};
static const unsigned char kat57_nor_nonce[] = {
    0xf1, 0x26, 0x6d, 0x65, 0x2c, 0x32, 0x7a, 0xa5,
};
static const unsigned char kat57_nor_persstr[] = {
    0xaf, 0x28, 0xaf, 0x29, 0x7a, 0xb2, 0x98, 0xa8, 0x38, 0x61, 0x6f, 0x1f,
    0xdd, 0xe1, 0xce, 0xed,
};
static const unsigned char kat57_nor_addin0[] = {
    0x09, 0x33, 0x4c, 0xe2, 0x3d, 0xf4, 0xcf, 0x76, 0x29, 0x79, 0x4a, 0xc0,
    0x16, 0xd4, 0xab, 0x61,
};
static const unsigned char kat57_nor_addin1[] = {
    0x59, 0x2f, 0x2a, 0x1f, 0x07, 0x5a, 0xe6, 0x9f, 0x31, 0x08, 0x27, 0xd1,
    0x5e, 0xb2, 0xd6, 0xc9,
};
static const unsigned char kat57_nor_retbytes[] = {
    0x00, 0x35, 0x68, 0x4b, 0x66, 0xbd, 0x9e, 0x73, 0xcb, 0x23, 0xfb, 0x04,
    0xfd, 0x47, 0xc4, 0xe1, 0xf3, 0x54, 0x67, 0x03, 0x1a, 0xcf, 0x05, 0xdc,
    0xd9, 0x77, 0x1a, 0x81, 0x93, 0x18, 0x0d, 0x7c, 0x69, 0xbf, 0x87, 0x52,
    0x1a, 0x1d, 0xf1, 0x9b, 0xf8, 0x47, 0xc6, 0x59, 0xa6, 0xc4, 0xcd, 0xbc,
    0x0c, 0x21, 0xe2, 0xb9, 0xcb, 0x26, 0x03, 0x4c, 0xcc, 0xb9, 0x02, 0x56,
    0x3d, 0x16, 0xc6, 0x3f, 0x39, 0x31, 0x4d, 0x8c, 0xb5, 0x95, 0x04, 0x87,
    0xbc, 0x37, 0x8f, 0x77, 0x5b, 0x8c, 0xfd, 0x8f,
};
static const struct drbg_kat_no_reseed kat57_nor_t = {
    11, kat57_nor_entropyin, kat57_nor_nonce, kat57_nor_persstr,
    kat57_nor_addin0, kat57_nor_addin1, kat57_nor_retbytes
};
static const struct drbg_kat kat57_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat57_nor_t
};

static const unsigned char kat58_nor_entropyin[] = {
    0x3d, 0xaf, 0x9f, 0x11, 0x0b, 0xfe, 0x02, 0xbb, 0x4b, 0x1c, 0xad, 0xb6,
    0x1f, 0x79, 0xda, 0x85,
};
static const unsigned char kat58_nor_nonce[] = {
    0x1a, 0x0e, 0x5f, 0x88, 0xd5, 0x0f, 0xd8, 0x6e,
};
static const unsigned char kat58_nor_persstr[] = {
    0x16, 0xb3, 0x20, 0xd8, 0xbf, 0x28, 0xc7, 0x26, 0x42, 0xaa, 0x32, 0xd4,
    0xd9, 0x79, 0x8c, 0x4b,
};
static const unsigned char kat58_nor_addin0[] = {
    0x15, 0x3a, 0xb4, 0x30, 0xf0, 0x75, 0x06, 0x59, 0xfd, 0xff, 0xa6, 0xf9,
    0x8c, 0x6e, 0xa9, 0xf6,
};
static const unsigned char kat58_nor_addin1[] = {
    0x7e, 0xb0, 0xd6, 0x25, 0x9b, 0x23, 0x7b, 0x7c, 0x38, 0x72, 0x8f, 0xe7,
    0x64, 0x9e, 0x64, 0x3f,
};
static const unsigned char kat58_nor_retbytes[] = {
    0x92, 0xd0, 0x66, 0xeb, 0x16, 0xcb, 0x87, 0x53, 0x63, 0x08, 0x88, 0xe4,
    0x40, 0x18, 0x59, 0x28, 0x80, 0xc6, 0x25, 0x35, 0x76, 0x53, 0x1d, 0x90,
    0xd0, 0x42, 0xd1, 0xbe, 0x96, 0x71, 0xc1, 0xb7, 0x8c, 0xb8, 0x42, 0x61,
    0xb5, 0xce, 0x0b, 0x12, 0xf3, 0xfc, 0x1e, 0x12, 0xa8, 0x8f, 0x97, 0x99,
    0xcb, 0xa1, 0xa8, 0xa0, 0x5b, 0x1c, 0xb0, 0xc5, 0x89, 0x25, 0xa7, 0x91,
    0x3a, 0xb2, 0x74, 0xf8, 0x75, 0xcc, 0x71, 0xc4, 0x74, 0x03, 0xfb, 0x24,
    0x2f, 0x85, 0xae, 0x85, 0x8e, 0xc5, 0x85, 0x05,
};
static const struct drbg_kat_no_reseed kat58_nor_t = {
    12, kat58_nor_entropyin, kat58_nor_nonce, kat58_nor_persstr,
    kat58_nor_addin0, kat58_nor_addin1, kat58_nor_retbytes
};
static const struct drbg_kat kat58_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat58_nor_t
};

static const unsigned char kat59_nor_entropyin[] = {
    0xdd, 0x71, 0x99, 0x3a, 0xb8, 0xd7, 0xf4, 0xde, 0x58, 0xf4, 0xa9, 0x4a,
    0xec, 0xb0, 0x96, 0x2a,
};
static const unsigned char kat59_nor_nonce[] = {
    0xb6, 0x17, 0xbe, 0x93, 0x44, 0x50, 0x43, 0x1b,
};
static const unsigned char kat59_nor_persstr[] = {
    0x84, 0xc6, 0x95, 0xd8, 0x3b, 0xc7, 0xd5, 0xef, 0x90, 0x2a, 0x58, 0x3b,
    0xd7, 0x99, 0xf2, 0xb8,
};
static const unsigned char kat59_nor_addin0[] = {
    0x13, 0xee, 0x6c, 0xce, 0xd4, 0x02, 0xef, 0x8c, 0x85, 0x43, 0xec, 0x3a,
    0xc1, 0x6a, 0x38, 0xc6,
};
static const unsigned char kat59_nor_addin1[] = {
    0x9d, 0x0a, 0x0f, 0xc6, 0x78, 0x84, 0x98, 0x60, 0x32, 0x05, 0x68, 0xf9,
    0xb1, 0x90, 0x53, 0xf9,
};
static const unsigned char kat59_nor_retbytes[] = {
    0xfb, 0xda, 0x06, 0x0b, 0xd0, 0xd0, 0x0b, 0x0c, 0x31, 0xc8, 0xcd, 0x13,
    0x50, 0x7c, 0x36, 0x57, 0x9e, 0x6a, 0xf5, 0xe1, 0x56, 0xc9, 0x6b, 0xd8,
    0x60, 0x64, 0x6b, 0x83, 0xe8, 0x1c, 0x24, 0x37, 0xf6, 0x36, 0xde, 0xba,
    0x7b, 0x2e, 0xf6, 0xe5, 0x95, 0x8f, 0xb3, 0xda, 0xe2, 0x65, 0x93, 0x44,
    0x76, 0xa6, 0x54, 0x99, 0x0d, 0x08, 0x92, 0x6d, 0xee, 0xde, 0xcf, 0x1e,
    0x43, 0xa4, 0x1f, 0xfb, 0x61, 0xbc, 0xad, 0x56, 0x0d, 0xcb, 0xce, 0x85,
    0x8c, 0x6a, 0xc3, 0xf7, 0x48, 0x95, 0x56, 0x60,
};
static const struct drbg_kat_no_reseed kat59_nor_t = {
    13, kat59_nor_entropyin, kat59_nor_nonce, kat59_nor_persstr,
    kat59_nor_addin0, kat59_nor_addin1, kat59_nor_retbytes
};
static const struct drbg_kat kat59_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat59_nor_t
};

static const unsigned char kat60_nor_entropyin[] = {
    0xd7, 0x7f, 0xf8, 0x6d, 0x65, 0x3f, 0x38, 0x97, 0x53, 0x22, 0xd0, 0xb7,
    0xf1, 0xc0, 0x2b, 0x6b,
};
static const unsigned char kat60_nor_nonce[] = {
    0x4b, 0x2d, 0x2a, 0x31, 0xb1, 0x36, 0x7c, 0x86,
};
static const unsigned char kat60_nor_persstr[] = {
    0x84, 0x21, 0x08, 0xe0, 0x74, 0xfd, 0xea, 0x38, 0x7c, 0xfe, 0x71, 0x52,
    0x6e, 0xaa, 0xc9, 0xb4,
};
static const unsigned char kat60_nor_addin0[] = {
    0xe4, 0x88, 0xe1, 0x6f, 0x48, 0xc6, 0x1d, 0xd2, 0x15, 0x2a, 0xfe, 0x92,
    0x5e, 0xce, 0xee, 0x92,
};
static const unsigned char kat60_nor_addin1[] = {
    0x12, 0xc6, 0x92, 0xab, 0xd9, 0x0a, 0xb4, 0x85, 0xf4, 0xd9, 0x49, 0x96,
    0x80, 0xa6, 0x89, 0x3f,
};
static const unsigned char kat60_nor_retbytes[] = {
    0x8b, 0xa0, 0x46, 0x17, 0xa1, 0x35, 0xd8, 0xab, 0xe0, 0xc3, 0xc0, 0xa1,
    0x70, 0xe7, 0x47, 0x2e, 0x7e, 0xd7, 0x50, 0xea, 0xc7, 0x06, 0xe5, 0xc3,
    0xed, 0x83, 0x05, 0xd6, 0xf6, 0xf8, 0xa1, 0xa5, 0x3e, 0x0c, 0x52, 0xd4,
    0x85, 0x3b, 0x21, 0xab, 0x89, 0x51, 0xe8, 0x09, 0x70, 0xb4, 0x26, 0x00,
    0x8a, 0xe1, 0x19, 0x52, 0xff, 0x36, 0x48, 0x17, 0xb6, 0x85, 0x6e, 0xf0,
    0x81, 0x08, 0x60, 0xdc, 0x65, 0xfa, 0xea, 0x48, 0x7b, 0x5d, 0x7c, 0x3f,
    0x3d, 0x63, 0xfd, 0x44, 0x37, 0x56, 0xd2, 0xa8,
};
static const struct drbg_kat_no_reseed kat60_nor_t = {
    14, kat60_nor_entropyin, kat60_nor_nonce, kat60_nor_persstr,
    kat60_nor_addin0, kat60_nor_addin1, kat60_nor_retbytes
};
static const struct drbg_kat kat60_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat60_nor_t
};

static const unsigned char kat61_nor_entropyin[] = {
    0x66, 0xb3, 0xc7, 0x8f, 0xa8, 0xc8, 0x6d, 0xd8, 0x91, 0xaa, 0xb0, 0xe5,
    0x3f, 0x36, 0x06, 0xde,
};
static const unsigned char kat61_nor_nonce[] = {
    0x84, 0x14, 0x2c, 0x2d, 0x1f, 0x1c, 0xda, 0x88,
};
static const unsigned char kat61_nor_persstr[] = {0};
static const unsigned char kat61_nor_addin0[] = {0};
static const unsigned char kat61_nor_addin1[] = {0};
static const unsigned char kat61_nor_retbytes[] = {
    0xe6, 0x0d, 0x83, 0xed, 0x6b, 0xbe, 0x75, 0x35, 0x1f, 0x2f, 0xbd, 0x35,
    0x6a, 0x4b, 0xc8, 0x90, 0x4b, 0x97, 0x0d, 0x45, 0x98, 0x2e, 0xde, 0xb8,
    0x98, 0xd0, 0xf2, 0xe6, 0xf4, 0x70, 0x7f, 0xaf, 0x49, 0x74, 0x71, 0xc3,
    0xdf, 0x74, 0x60, 0x78, 0x3e, 0x4b, 0xef, 0xf0, 0x01, 0x5d, 0x23, 0xd8,
    0xc7, 0xba, 0x27, 0x0b, 0x6f, 0x56, 0x2c, 0x4a, 0x7b, 0x76, 0xc0, 0x69,
    0x0b, 0x26, 0x09, 0xad, 0xe3, 0xfe, 0x37, 0x2b, 0x53, 0xf1, 0xc5, 0xff,
    0x4d, 0x5f, 0x0d, 0xd9, 0xa3, 0x96, 0x68, 0x25,
};
static const struct drbg_kat_no_reseed kat61_nor_t = {
    0, kat61_nor_entropyin, kat61_nor_nonce, kat61_nor_persstr,
    kat61_nor_addin0, kat61_nor_addin1, kat61_nor_retbytes
};
static const struct drbg_kat kat61_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat61_nor_t
};

static const unsigned char kat62_nor_entropyin[] = {
    0x22, 0xbf, 0x7d, 0x4b, 0x28, 0xe0, 0xb0, 0xcd, 0x26, 0x3c, 0x11, 0x4e,
    0x59, 0x4c, 0xe5, 0xa1,
};
static const unsigned char kat62_nor_nonce[] = {
    0x20, 0x9e, 0xab, 0x89, 0x79, 0xe7, 0xb5, 0x9a,
};
static const unsigned char kat62_nor_persstr[] = {0};
static const unsigned char kat62_nor_addin0[] = {0};
static const unsigned char kat62_nor_addin1[] = {0};
static const unsigned char kat62_nor_retbytes[] = {
    0x4c, 0xa3, 0xbb, 0x37, 0xcf, 0x63, 0xc5, 0x89, 0xed, 0xfd, 0x06, 0x2a,
    0xbd, 0x77, 0x11, 0x65, 0x42, 0xb5, 0xb7, 0x53, 0xc3, 0xcd, 0xca, 0x97,
    0x49, 0x7e, 0xb6, 0xe1, 0x3b, 0xea, 0xed, 0xde, 0x8c, 0x43, 0xbe, 0xed,
    0xbc, 0xf9, 0xa7, 0x2c, 0xc8, 0x71, 0x50, 0x70, 0xcb, 0xdd, 0xf0, 0xf6,
    0xdd, 0xc6, 0x14, 0x69, 0x71, 0xb2, 0x2c, 0xcd, 0xb1, 0xe3, 0xc9, 0xee,
    0x73, 0x0d, 0xa1, 0xb2, 0x12, 0x6b, 0xff, 0x2c, 0x3c, 0xb0, 0xef, 0xab,
    0xc8, 0x71, 0xfe, 0xa4, 0x94, 0x70, 0xa7, 0xcb,
};
static const struct drbg_kat_no_reseed kat62_nor_t = {
    1, kat62_nor_entropyin, kat62_nor_nonce, kat62_nor_persstr,
    kat62_nor_addin0, kat62_nor_addin1, kat62_nor_retbytes
};
static const struct drbg_kat kat62_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat62_nor_t
};

static const unsigned char kat63_nor_entropyin[] = {
    0x20, 0x80, 0x7f, 0x2c, 0x0e, 0x86, 0x55, 0x14, 0xf6, 0xfc, 0xdb, 0xbb,
    0xaa, 0xc5, 0x98, 0x95,
};
static const unsigned char kat63_nor_nonce[] = {
    0xe6, 0x0b, 0xaa, 0x00, 0x3c, 0x2c, 0x18, 0xd5,
};
static const unsigned char kat63_nor_persstr[] = {0};
static const unsigned char kat63_nor_addin0[] = {0};
static const unsigned char kat63_nor_addin1[] = {0};
static const unsigned char kat63_nor_retbytes[] = {
    0xff, 0x89, 0x7d, 0x95, 0x7d, 0x8b, 0xe8, 0x63, 0x91, 0x54, 0x61, 0x29,
    0xce, 0x6c, 0x98, 0xdd, 0xc4, 0x38, 0x06, 0x78, 0x4c, 0x24, 0xa0, 0xd3,
    0xcb, 0x30, 0x80, 0xe9, 0xea, 0xc3, 0xc2, 0x06, 0x05, 0xff, 0x6d, 0x1c,
    0xaa, 0x24, 0xbd, 0x03, 0x8d, 0x8b, 0x68, 0x3f, 0x07, 0xb2, 0x4f, 0x38,
    0x46, 0xac, 0xa8, 0x23, 0x65, 0x81, 0xb4, 0xc4, 0xd0, 0x4e, 0xb8, 0xb8,
    0x70, 0x20, 0xc0, 0x7b, 0x55, 0x88, 0x83, 0x47, 0xa2, 0x63, 0x82, 0xd6,
    0x2f, 0x9d, 0x10, 0xd4, 0x12, 0x1a, 0x21, 0xc9,
};
static const struct drbg_kat_no_reseed kat63_nor_t = {
    2, kat63_nor_entropyin, kat63_nor_nonce, kat63_nor_persstr,
    kat63_nor_addin0, kat63_nor_addin1, kat63_nor_retbytes
};
static const struct drbg_kat kat63_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat63_nor_t
};

static const unsigned char kat64_nor_entropyin[] = {
    0xbe, 0x8d, 0x9f, 0xec, 0x6b, 0x80, 0x91, 0x4c, 0x70, 0xa9, 0xc2, 0x8e,
    0xa6, 0xbf, 0xe7, 0xc4,
};
static const unsigned char kat64_nor_nonce[] = {
    0xd6, 0xf7, 0xda, 0x93, 0xc2, 0x69, 0x33, 0x41,
};
static const unsigned char kat64_nor_persstr[] = {0};
static const unsigned char kat64_nor_addin0[] = {0};
static const unsigned char kat64_nor_addin1[] = {0};
static const unsigned char kat64_nor_retbytes[] = {
    0x7a, 0xab, 0x1f, 0x40, 0x8b, 0x87, 0x1a, 0x49, 0x4b, 0x9d, 0xf4, 0x22,
    0x08, 0xd8, 0x3b, 0xfe, 0xf1, 0xba, 0x64, 0x76, 0xa5, 0x1c, 0xc2, 0xf3,
    0xb7, 0x08, 0x96, 0x81, 0x6b, 0x1c, 0x1e, 0x41, 0x0d, 0x50, 0x6a, 0xac,
    0xd3, 0xe1, 0xf8, 0x91, 0x6d, 0xc8, 0xbe, 0x0b, 0xce, 0x42, 0xf9, 0x86,
    0xf6, 0x7b, 0xd2, 0xf9, 0xf0, 0x23, 0x95, 0xfb, 0xa1, 0xff, 0xe4, 0x99,
    0xe5, 0xee, 0xf7, 0x0b, 0x3b, 0xf7, 0x43, 0xf5, 0x19, 0xfd, 0x8c, 0xa1,
    0xdb, 0x69, 0x4f, 0x89, 0x70, 0xf8, 0x24, 0x21,
};
static const struct drbg_kat_no_reseed kat64_nor_t = {
    3, kat64_nor_entropyin, kat64_nor_nonce, kat64_nor_persstr,
    kat64_nor_addin0, kat64_nor_addin1, kat64_nor_retbytes
};
static const struct drbg_kat kat64_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat64_nor_t
};

static const unsigned char kat65_nor_entropyin[] = {
    0x29, 0x58, 0x12, 0x8d, 0xf4, 0x64, 0x9e, 0x1a, 0xe1, 0xf0, 0xdd, 0x5d,
    0xd3, 0x74, 0x0c, 0xe9,
};
static const unsigned char kat65_nor_nonce[] = {
    0x18, 0x1c, 0xe1, 0x94, 0xbb, 0x3a, 0xa4, 0xec,
};
static const unsigned char kat65_nor_persstr[] = {0};
static const unsigned char kat65_nor_addin0[] = {0};
static const unsigned char kat65_nor_addin1[] = {0};
static const unsigned char kat65_nor_retbytes[] = {
    0xb7, 0xcd, 0xa6, 0xec, 0x47, 0xed, 0x3f, 0xf8, 0xda, 0xfb, 0x78, 0xed,
    0x04, 0xfc, 0x5c, 0x0e, 0xc2, 0xd6, 0xcc, 0xb1, 0x86, 0x94, 0x22, 0x66,
    0x5d, 0xc4, 0xbd, 0xf9, 0x84, 0x25, 0x96, 0xee, 0xe0, 0x0f, 0xd5, 0x93,
    0x4d, 0xc1, 0x7c, 0xbb, 0x17, 0xc5, 0x7f, 0xb9, 0x70, 0xce, 0x9a, 0x9b,
    0xdf, 0xdc, 0xa7, 0x5a, 0xaa, 0xa4, 0x26, 0x3a, 0x87, 0x5a, 0x2e, 0xd0,
    0x12, 0xb6, 0xd5, 0x60, 0x49, 0xf1, 0x3b, 0x51, 0xe2, 0xa4, 0x4a, 0xcb,
    0x37, 0xf9, 0x22, 0xf8, 0xd1, 0x3a, 0x4d, 0x19,
};
static const struct drbg_kat_no_reseed kat65_nor_t = {
    4, kat65_nor_entropyin, kat65_nor_nonce, kat65_nor_persstr,
    kat65_nor_addin0, kat65_nor_addin1, kat65_nor_retbytes
};
static const struct drbg_kat kat65_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat65_nor_t
};

static const unsigned char kat66_nor_entropyin[] = {
    0x91, 0xea, 0xff, 0xb7, 0xb6, 0x3f, 0x5d, 0xba, 0x78, 0x24, 0xd5, 0x62,
    0xf8, 0x63, 0xb0, 0x53,
};
static const unsigned char kat66_nor_nonce[] = {
    0xfe, 0x53, 0x0c, 0xa5, 0xb6, 0xe3, 0x24, 0x3f,
};
static const unsigned char kat66_nor_persstr[] = {0};
static const unsigned char kat66_nor_addin0[] = {0};
static const unsigned char kat66_nor_addin1[] = {0};
static const unsigned char kat66_nor_retbytes[] = {
    0xea, 0x37, 0x08, 0xf5, 0x6e, 0x3a, 0x51, 0x5b, 0x5e, 0xef, 0x9e, 0x36,
    0x1c, 0x4e, 0x0a, 0x8c, 0x38, 0x5f, 0x38, 0xc1, 0x7e, 0x82, 0x16, 0xf4,
    0x2a, 0x2d, 0xb5, 0x8e, 0x4c, 0xe5, 0xb4, 0x8c, 0x75, 0x21, 0xb2, 0xff,
    0xdf, 0xe2, 0xe0, 0xbe, 0x55, 0xd4, 0x30, 0x15, 0x0b, 0x6a, 0x21, 0x21,
    0xde, 0x11, 0xca, 0x71, 0x97, 0x16, 0x7e, 0xc0, 0xd3, 0xbd, 0x63, 0x83,
    0xbe, 0x4b, 0x3b, 0x47, 0xd9, 0xc5, 0x7f, 0xc2, 0x24, 0x08, 0x8d, 0x05,
    0x01, 0x14, 0x75, 0x71, 0x15, 0xee, 0xe0, 0x02,
};
static const struct drbg_kat_no_reseed kat66_nor_t = {
    5, kat66_nor_entropyin, kat66_nor_nonce, kat66_nor_persstr,
    kat66_nor_addin0, kat66_nor_addin1, kat66_nor_retbytes
};
static const struct drbg_kat kat66_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat66_nor_t
};

static const unsigned char kat67_nor_entropyin[] = {
    0x22, 0xc8, 0x9c, 0x67, 0x1b, 0x47, 0x80, 0x15, 0xca, 0xe3, 0xda, 0x53,
    0x03, 0x56, 0x85, 0x2d,
};
static const unsigned char kat67_nor_nonce[] = {
    0x96, 0x32, 0xeb, 0xd1, 0x3a, 0xc4, 0x06, 0xd6,
};
static const unsigned char kat67_nor_persstr[] = {0};
static const unsigned char kat67_nor_addin0[] = {0};
static const unsigned char kat67_nor_addin1[] = {0};
static const unsigned char kat67_nor_retbytes[] = {
    0x86, 0x1a, 0xec, 0x24, 0x88, 0xdf, 0x56, 0xea, 0x7f, 0x0b, 0xc4, 0xbd,
    0x14, 0x2c, 0x91, 0x62, 0xce, 0x6a, 0x64, 0xb7, 0x39, 0x12, 0xd9, 0x49,
    0x81, 0xb7, 0x15, 0xaa, 0x14, 0xdd, 0x1f, 0xd6, 0x49, 0xad, 0x1d, 0x86,
    0xfc, 0x4a, 0x53, 0x65, 0xc2, 0x30, 0x7d, 0x9f, 0x67, 0x71, 0xdd, 0xfd,
    0xb7, 0x2d, 0x6d, 0x89, 0xd9, 0x8a, 0x10, 0x34, 0x1e, 0x44, 0x05, 0x62,
    0xc7, 0x61, 0xc9, 0xab, 0x11, 0x90, 0x52, 0x9b, 0xf7, 0x21, 0xdb, 0x2a,
    0xc0, 0xd0, 0x0c, 0xfb, 0x61, 0x8e, 0x34, 0xa5,
};
static const struct drbg_kat_no_reseed kat67_nor_t = {
    6, kat67_nor_entropyin, kat67_nor_nonce, kat67_nor_persstr,
    kat67_nor_addin0, kat67_nor_addin1, kat67_nor_retbytes
};
static const struct drbg_kat kat67_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat67_nor_t
};

static const unsigned char kat68_nor_entropyin[] = {
    0xed, 0xa4, 0xdd, 0x91, 0x7f, 0xd7, 0x76, 0x5a, 0x8b, 0x10, 0x2e, 0x83,
    0x1c, 0x39, 0x38, 0x4b,
};
static const unsigned char kat68_nor_nonce[] = {
    0x25, 0x5d, 0x24, 0x8c, 0xdb, 0xa3, 0xcf, 0x78,
};
static const unsigned char kat68_nor_persstr[] = {0};
static const unsigned char kat68_nor_addin0[] = {0};
static const unsigned char kat68_nor_addin1[] = {0};
static const unsigned char kat68_nor_retbytes[] = {
    0x83, 0x26, 0x3f, 0xec, 0x6c, 0x1c, 0x70, 0xcb, 0xc0, 0x88, 0xe1, 0x3d,
    0xde, 0x4d, 0xe3, 0x1f, 0x14, 0x2d, 0x7d, 0x20, 0x42, 0x45, 0x7d, 0x79,
    0xf0, 0xc0, 0x33, 0xf0, 0xd7, 0x90, 0x62, 0x1f, 0x75, 0x82, 0x3c, 0x55,
    0xe4, 0x1e, 0x9a, 0x44, 0x61, 0xad, 0x7c, 0xcd, 0xdf, 0x52, 0x37, 0x73,
    0x40, 0xd5, 0x1f, 0x72, 0x7b, 0x0f, 0x9b, 0x81, 0x2d, 0x15, 0x10, 0x36,
    0xcf, 0xa6, 0x86, 0xbc, 0x28, 0xaf, 0xea, 0xae, 0x99, 0x5e, 0x13, 0x0a,
    0x1e, 0xe0, 0xdd, 0x11, 0x10, 0x4e, 0x5a, 0xb1,
};
static const struct drbg_kat_no_reseed kat68_nor_t = {
    7, kat68_nor_entropyin, kat68_nor_nonce, kat68_nor_persstr,
    kat68_nor_addin0, kat68_nor_addin1, kat68_nor_retbytes
};
static const struct drbg_kat kat68_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat68_nor_t
};

static const unsigned char kat69_nor_entropyin[] = {
    0x4e, 0xf4, 0x97, 0x80, 0x78, 0xec, 0x29, 0x82, 0x06, 0x36, 0xbb, 0x17,
    0x7f, 0xfb, 0x4b, 0x2f,
};
static const unsigned char kat69_nor_nonce[] = {
    0xe4, 0xc9, 0xc3, 0x45, 0xce, 0xdf, 0xc1, 0x23,
};
static const unsigned char kat69_nor_persstr[] = {0};
static const unsigned char kat69_nor_addin0[] = {0};
static const unsigned char kat69_nor_addin1[] = {0};
static const unsigned char kat69_nor_retbytes[] = {
    0x22, 0xc7, 0xec, 0xa6, 0x33, 0xda, 0xb4, 0x83, 0x68, 0x81, 0x39, 0x39,
    0x7a, 0x2c, 0xd8, 0xc4, 0x26, 0x50, 0x6f, 0x60, 0xdd, 0xe1, 0xbf, 0xe5,
    0xe4, 0xe7, 0x99, 0x99, 0x90, 0x87, 0xf7, 0xf2, 0x04, 0x6f, 0x7d, 0x8e,
    0x04, 0x06, 0x17, 0x9d, 0x48, 0x59, 0xff, 0x61, 0x9d, 0x70, 0xce, 0x6d,
    0xb7, 0x57, 0x3e, 0xc4, 0xa2, 0x08, 0x5c, 0x48, 0x24, 0xd7, 0xe8, 0x2d,
    0x4f, 0xef, 0xa6, 0xfa, 0x43, 0xcf, 0xe2, 0x3b, 0x24, 0x5d, 0xab, 0x95,
    0xc6, 0x24, 0x86, 0x6a, 0x72, 0xc1, 0x67, 0xa0,
};
static const struct drbg_kat_no_reseed kat69_nor_t = {
    8, kat69_nor_entropyin, kat69_nor_nonce, kat69_nor_persstr,
    kat69_nor_addin0, kat69_nor_addin1, kat69_nor_retbytes
};
static const struct drbg_kat kat69_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat69_nor_t
};

static const unsigned char kat70_nor_entropyin[] = {
    0xa2, 0x68, 0x07, 0x50, 0xd8, 0x73, 0x1c, 0xd0, 0x93, 0xd6, 0x28, 0x4f,
    0x57, 0x04, 0x66, 0xad,
};
static const unsigned char kat70_nor_nonce[] = {
    0x76, 0x58, 0xf3, 0x00, 0x66, 0x6b, 0x66, 0xd2,
};
static const unsigned char kat70_nor_persstr[] = {0};
static const unsigned char kat70_nor_addin0[] = {0};
static const unsigned char kat70_nor_addin1[] = {0};
static const unsigned char kat70_nor_retbytes[] = {
    0xda, 0x6c, 0xdf, 0x35, 0x48, 0xef, 0x1a, 0xe5, 0x95, 0x58, 0x14, 0xe3,
    0x6c, 0x73, 0xb7, 0xae, 0xaf, 0x5b, 0x4d, 0x53, 0x3b, 0x03, 0x7a, 0xfc,
    0xe4, 0x9b, 0xf7, 0x2b, 0xf0, 0x41, 0x6c, 0x91, 0x18, 0x25, 0xae, 0x05,
    0xfd, 0x97, 0xcc, 0x5f, 0xed, 0x23, 0x01, 0xcb, 0xfb, 0x20, 0x54, 0x5a,
    0x20, 0xd5, 0x60, 0x96, 0xaa, 0x47, 0x4a, 0x39, 0xb3, 0xda, 0xbf, 0xc8,
    0x88, 0xa8, 0xa6, 0x5e, 0xf3, 0x23, 0x0e, 0x69, 0xf2, 0xce, 0x5b, 0x0d,
    0x7d, 0x30, 0x73, 0x8f, 0x70, 0x8d, 0x5f, 0x58,
};
static const struct drbg_kat_no_reseed kat70_nor_t = {
    9, kat70_nor_entropyin, kat70_nor_nonce, kat70_nor_persstr,
    kat70_nor_addin0, kat70_nor_addin1, kat70_nor_retbytes
};
static const struct drbg_kat kat70_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat70_nor_t
};

static const unsigned char kat71_nor_entropyin[] = {
    0x23, 0x8f, 0x4a, 0x64, 0xdb, 0x2c, 0xd8, 0xd1, 0x40, 0xf9, 0xfb, 0xfd,
    0xea, 0x10, 0x23, 0x38,
};
static const unsigned char kat71_nor_nonce[] = {
    0xb1, 0x9a, 0xf3, 0xbc, 0x12, 0x55, 0x2f, 0xbb,
};
static const unsigned char kat71_nor_persstr[] = {0};
static const unsigned char kat71_nor_addin0[] = {0};
static const unsigned char kat71_nor_addin1[] = {0};
static const unsigned char kat71_nor_retbytes[] = {
    0x27, 0xce, 0x1e, 0xfb, 0x77, 0xdf, 0x4a, 0x84, 0x8f, 0x0e, 0xb7, 0x24,
    0x51, 0xc0, 0x77, 0x1f, 0x08, 0xb8, 0x0f, 0x45, 0xe2, 0x90, 0x09, 0x1f,
    0x8c, 0x38, 0x52, 0xfd, 0xa0, 0xb7, 0x1b, 0x28, 0x1d, 0x01, 0xa0, 0x70,
    0xb3, 0x80, 0xda, 0xf7, 0xef, 0xa9, 0x46, 0x97, 0x80, 0x6d, 0x3d, 0x13,
    0x8e, 0x94, 0x51, 0x13, 0x00, 0x2a, 0x62, 0x84, 0x83, 0xdd, 0x7e, 0x34,
    0x1c, 0x7a, 0x9a, 0x6f, 0x83, 0x6d, 0x99, 0x57, 0x7b, 0x39, 0xfb, 0x0f,
    0x9a, 0x73, 0x2f, 0x14, 0x45, 0x95, 0x95, 0xfc,
};
static const struct drbg_kat_no_reseed kat71_nor_t = {
    10, kat71_nor_entropyin, kat71_nor_nonce, kat71_nor_persstr,
    kat71_nor_addin0, kat71_nor_addin1, kat71_nor_retbytes
};
static const struct drbg_kat kat71_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat71_nor_t
};

static const unsigned char kat72_nor_entropyin[] = {
    0x1a, 0x3a, 0x03, 0x33, 0x34, 0x0f, 0x1d, 0x67, 0xdd, 0xa7, 0x15, 0x28,
    0x5b, 0xbf, 0xb6, 0x17,
};
static const unsigned char kat72_nor_nonce[] = {
    0x37, 0x4b, 0x8f, 0x8b, 0xe9, 0xf2, 0x82, 0x0e,
};
static const unsigned char kat72_nor_persstr[] = {0};
static const unsigned char kat72_nor_addin0[] = {0};
static const unsigned char kat72_nor_addin1[] = {0};
static const unsigned char kat72_nor_retbytes[] = {
    0xf2, 0x8c, 0xd9, 0x13, 0x65, 0xd0, 0x2d, 0x20, 0x4f, 0xa7, 0x1f, 0xe8,
    0x6e, 0xe4, 0x0f, 0xb0, 0x5f, 0x47, 0x78, 0x64, 0xbe, 0x55, 0x6f, 0x2e,
    0x8e, 0x0f, 0x21, 0x2d, 0xcb, 0x1e, 0xd9, 0x8e, 0xf5, 0x9a, 0xdd, 0xe4,
    0x85, 0xcb, 0x28, 0xaf, 0xee, 0x97, 0xac, 0x44, 0xa7, 0x89, 0x62, 0x65,
    0x50, 0xd5, 0x33, 0xe6, 0xcb, 0x8a, 0x13, 0xe0, 0xcd, 0x03, 0x9a, 0x7d,
    0x56, 0xf2, 0x70, 0xb8, 0x92, 0xdf, 0x74, 0x2c, 0x49, 0x9f, 0x5c, 0x60,
    0x9e, 0xf7, 0x44, 0x54, 0xf1, 0xa8, 0xcb, 0xf5,
};
static const struct drbg_kat_no_reseed kat72_nor_t = {
    11, kat72_nor_entropyin, kat72_nor_nonce, kat72_nor_persstr,
    kat72_nor_addin0, kat72_nor_addin1, kat72_nor_retbytes
};
static const struct drbg_kat kat72_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat72_nor_t
};

static const unsigned char kat73_nor_entropyin[] = {
    0xd6, 0x22, 0xc4, 0x7e, 0x98, 0x38, 0xe0, 0xfc, 0xd8, 0xbf, 0xb2, 0x09,
    0xf4, 0x2a, 0x3c, 0x29,
};
static const unsigned char kat73_nor_nonce[] = {
    0xb2, 0x9b, 0xba, 0x7c, 0xfe, 0xba, 0x90, 0xcc,
};
static const unsigned char kat73_nor_persstr[] = {0};
static const unsigned char kat73_nor_addin0[] = {0};
static const unsigned char kat73_nor_addin1[] = {0};
static const unsigned char kat73_nor_retbytes[] = {
    0x93, 0x40, 0xbb, 0x46, 0x92, 0xe5, 0x27, 0x92, 0x69, 0x3e, 0x8e, 0xdc,
    0x0c, 0x28, 0x83, 0xf5, 0x4f, 0x8b, 0x93, 0x94, 0x2a, 0xe3, 0x6f, 0x06,
    0xd5, 0x53, 0x29, 0x64, 0xf1, 0xf9, 0xb5, 0x81, 0xc0, 0x2a, 0x86, 0xb9,
    0x52, 0x05, 0x66, 0xf5, 0xf7, 0x96, 0xe3, 0xcb, 0x21, 0x94, 0x85, 0x3e,
    0xd9, 0xec, 0x18, 0xb6, 0x93, 0xbc, 0x7a, 0xc9, 0xa8, 0x3c, 0x0f, 0x8d,
    0xb8, 0x96, 0xd9, 0x3f, 0xa2, 0x09, 0x53, 0xd4, 0x59, 0xaf, 0xe4, 0x43,
    0xb6, 0x3d, 0x12, 0x41, 0xd0, 0xb4, 0x7c, 0xc1,
};
static const struct drbg_kat_no_reseed kat73_nor_t = {
    12, kat73_nor_entropyin, kat73_nor_nonce, kat73_nor_persstr,
    kat73_nor_addin0, kat73_nor_addin1, kat73_nor_retbytes
};
static const struct drbg_kat kat73_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat73_nor_t
};

static const unsigned char kat74_nor_entropyin[] = {
    0xb5, 0x4f, 0xa4, 0xfb, 0xef, 0xc6, 0x3f, 0x96, 0xa4, 0x50, 0x70, 0xdb,
    0xa6, 0x69, 0xc3, 0xfa,
};
static const unsigned char kat74_nor_nonce[] = {
    0xdc, 0x9b, 0xb5, 0xe0, 0x14, 0xdf, 0x5f, 0x2b,
};
static const unsigned char kat74_nor_persstr[] = {0};
static const unsigned char kat74_nor_addin0[] = {0};
static const unsigned char kat74_nor_addin1[] = {0};
static const unsigned char kat74_nor_retbytes[] = {
    0x07, 0x45, 0x2a, 0xa8, 0x5d, 0x24, 0x16, 0xbb, 0x09, 0x1b, 0x9c, 0xc8,
    0xf4, 0x9a, 0xdd, 0x2d, 0xc8, 0x97, 0x46, 0x59, 0x98, 0x78, 0x2a, 0x01,
    0xe6, 0x28, 0xe9, 0xfe, 0xd6, 0xf0, 0xf1, 0x85, 0xfd, 0x5d, 0x41, 0x2d,
    0xc4, 0x06, 0x25, 0xde, 0xc7, 0x10, 0xeb, 0xf2, 0x67, 0x29, 0xd0, 0x34,
    0x6c, 0x34, 0x4e, 0xf2, 0xb1, 0xd3, 0xd4, 0x2a, 0x9a, 0xd6, 0x0f, 0x05,
    0x38, 0x64, 0x13, 0xe4, 0x09, 0x78, 0xd2, 0x06, 0xcf, 0x3f, 0xc8, 0xec,
    0xf4, 0x09, 0x50, 0xa2, 0x80, 0xe4, 0x47, 0x7c,
};
static const struct drbg_kat_no_reseed kat74_nor_t = {
    13, kat74_nor_entropyin, kat74_nor_nonce, kat74_nor_persstr,
    kat74_nor_addin0, kat74_nor_addin1, kat74_nor_retbytes
};
static const struct drbg_kat kat74_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat74_nor_t
};

static const unsigned char kat75_nor_entropyin[] = {
    0xce, 0xb3, 0x54, 0x44, 0x4d, 0x1a, 0x29, 0xc0, 0xc3, 0xe8, 0xa1, 0xcc,
    0x24, 0xd0, 0x28, 0x46,
};
static const unsigned char kat75_nor_nonce[] = {
    0x86, 0xd3, 0xfd, 0x9f, 0xc5, 0x1f, 0x8b, 0x19,
};
static const unsigned char kat75_nor_persstr[] = {0};
static const unsigned char kat75_nor_addin0[] = {0};
static const unsigned char kat75_nor_addin1[] = {0};
static const unsigned char kat75_nor_retbytes[] = {
    0x6f, 0x90, 0xad, 0x61, 0x19, 0x87, 0xa3, 0x7b, 0xac, 0x54, 0xbe, 0xa0,
    0x78, 0x2a, 0xc7, 0x82, 0x15, 0xb7, 0xd1, 0x7e, 0xcd, 0xd3, 0x99, 0x1a,
    0x81, 0xa3, 0x6d, 0x0e, 0x26, 0x3c, 0x6f, 0x0d, 0xda, 0x2c, 0x10, 0x2c,
    0xfb, 0xa5, 0x6b, 0x26, 0xc7, 0xb7, 0x4b, 0x5d, 0xd2, 0x54, 0x8b, 0xe9,
    0xbc, 0x81, 0xc7, 0x95, 0x8e, 0x9d, 0x19, 0x82, 0x15, 0x83, 0xc6, 0xf3,
    0x88, 0x13, 0x2b, 0x9e, 0x19, 0xae, 0x76, 0x09, 0xad, 0xd9, 0xa2, 0x96,
    0xc1, 0xe9, 0x2d, 0x66, 0xa2, 0xef, 0x54, 0x64,
};
static const struct drbg_kat_no_reseed kat75_nor_t = {
    14, kat75_nor_entropyin, kat75_nor_nonce, kat75_nor_persstr,
    kat75_nor_addin0, kat75_nor_addin1, kat75_nor_retbytes
};
static const struct drbg_kat kat75_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat75_nor_t
};

static const unsigned char kat76_nor_entropyin[] = {
    0xea, 0x97, 0xce, 0xcb, 0xca, 0xa4, 0xed, 0x7d, 0xa1, 0x3b, 0xec, 0xfe,
    0xda, 0x62, 0x05, 0xb4,
};
static const unsigned char kat76_nor_nonce[] = {
    0x59, 0x31, 0x5c, 0xd4, 0x61, 0x66, 0x3f, 0xe0,
};
static const unsigned char kat76_nor_persstr[] = {0};
static const unsigned char kat76_nor_addin0[] = {
    0x2b, 0x4b, 0xc4, 0x95, 0xeb, 0x17, 0x1e, 0x88, 0xbd, 0x9f, 0x63, 0x42,
    0x70, 0x80, 0xfa, 0x75,
};
static const unsigned char kat76_nor_addin1[] = {
    0xda, 0xc9, 0x7b, 0xb5, 0x11, 0x6f, 0x63, 0xbd, 0xb7, 0x2f, 0x29, 0xf4,
    0x6d, 0x38, 0xe5, 0x7c,
};
static const unsigned char kat76_nor_retbytes[] = {
    0xd7, 0x53, 0x6d, 0xab, 0x2b, 0x3b, 0x6c, 0x71, 0x86, 0x44, 0x3d, 0xe2,
    0xa7, 0xc6, 0x9e, 0x69, 0x65, 0xa0, 0x9c, 0xa8, 0xc7, 0x0f, 0x5d, 0xa0,
    0x69, 0xea, 0xa9, 0xd1, 0xb9, 0xff, 0xb2, 0x71, 0x4b, 0xda, 0x75, 0x1e,
    0x47, 0x9d, 0x83, 0x7f, 0x7d, 0xe4, 0xc8, 0xc9, 0x7b, 0xc8, 0xc1, 0x49,
    0x25, 0x79, 0x5f, 0xf1, 0x98, 0x3a, 0xf7, 0x1f, 0x04, 0x2d, 0xf3, 0xc8,
    0x73, 0xe8, 0xc4, 0x00, 0x81, 0xb9, 0xa4, 0x20, 0x53, 0xd4, 0x21, 0x53,
    0xdd, 0x16, 0x80, 0xbb, 0x2e, 0x8a, 0x45, 0x56,
};
static const struct drbg_kat_no_reseed kat76_nor_t = {
    0, kat76_nor_entropyin, kat76_nor_nonce, kat76_nor_persstr,
    kat76_nor_addin0, kat76_nor_addin1, kat76_nor_retbytes
};
static const struct drbg_kat kat76_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat76_nor_t
};

static const unsigned char kat77_nor_entropyin[] = {
    0xbe, 0x8b, 0xb1, 0x86, 0xbd, 0x76, 0xc6, 0xe3, 0xc2, 0xaa, 0x01, 0x67,
    0x85, 0xef, 0x0c, 0x94,
};
static const unsigned char kat77_nor_nonce[] = {
    0x0c, 0xdf, 0x98, 0x0a, 0x92, 0x49, 0x18, 0x0d,
};
static const unsigned char kat77_nor_persstr[] = {0};
static const unsigned char kat77_nor_addin0[] = {
    0x57, 0x62, 0xc6, 0x47, 0x8f, 0x21, 0x3d, 0xf4, 0x41, 0x18, 0xa8, 0x72,
    0xa7, 0x61, 0x62, 0x3a,
};
static const unsigned char kat77_nor_addin1[] = {
    0x98, 0x44, 0x75, 0xdb, 0x54, 0x03, 0xd4, 0xac, 0xf7, 0x1b, 0xad, 0x13,
    0x56, 0x72, 0x28, 0xc6,
};
static const unsigned char kat77_nor_retbytes[] = {
    0x15, 0x19, 0xc0, 0x13, 0x2b, 0xec, 0x98, 0x81, 0xf8, 0x03, 0xfe, 0xe1,
    0x6f, 0xcc, 0xf0, 0x4d, 0xb1, 0x1d, 0xa7, 0x10, 0x57, 0xa8, 0x83, 0xfe,
    0x26, 0x8d, 0x3c, 0x12, 0x0e, 0x41, 0xb6, 0xef, 0x86, 0x30, 0xa1, 0x91,
    0xad, 0xc2, 0xc9, 0xa4, 0x18, 0x33, 0x03, 0x96, 0x6f, 0x6f, 0x81, 0x5a,
    0x1c, 0xbc, 0xfd, 0xc2, 0x2e, 0xdc, 0x1f, 0x95, 0xbc, 0x82, 0x07, 0x6a,
    0xb5, 0x26, 0x1e, 0x8e, 0xa7, 0x49, 0x66, 0x0d, 0xd3, 0xa8, 0xf1, 0x07,
    0x72, 0x01, 0x28, 0x4a, 0x75, 0x29, 0x70, 0xa5,
};
static const struct drbg_kat_no_reseed kat77_nor_t = {
    1, kat77_nor_entropyin, kat77_nor_nonce, kat77_nor_persstr,
    kat77_nor_addin0, kat77_nor_addin1, kat77_nor_retbytes
};
static const struct drbg_kat kat77_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat77_nor_t
};

static const unsigned char kat78_nor_entropyin[] = {
    0x8c, 0x3f, 0x3c, 0x3a, 0xd8, 0x5d, 0x8d, 0xbf, 0x0b, 0x75, 0x57, 0x14,
    0x47, 0x7b, 0x45, 0x6c,
};
static const unsigned char kat78_nor_nonce[] = {
    0x69, 0x62, 0x8b, 0xcc, 0x65, 0xbb, 0x13, 0x28,
};
static const unsigned char kat78_nor_persstr[] = {0};
static const unsigned char kat78_nor_addin0[] = {
    0x5d, 0x85, 0xf4, 0x40, 0x86, 0x24, 0x69, 0x2f, 0x0d, 0x02, 0x58, 0xda,
    0xde, 0xb7, 0x0e, 0x16,
};
static const unsigned char kat78_nor_addin1[] = {
    0x4f, 0x94, 0x3e, 0x04, 0x3d, 0xd2, 0x24, 0xdb, 0x1a, 0x14, 0x4f, 0xf7,
    0x4f, 0xe9, 0x13, 0xad,
};
static const unsigned char kat78_nor_retbytes[] = {
    0x1e, 0x39, 0x68, 0x7f, 0x18, 0xb0, 0x8f, 0x56, 0xbc, 0xe6, 0xc2, 0x02,
    0x2e, 0x5a, 0x7d, 0xe1, 0xb0, 0x4c, 0xed, 0x77, 0x16, 0x30, 0xef, 0xeb,
    0x32, 0xb6, 0xc4, 0xa1, 0x25, 0xba, 0x83, 0xec, 0xed, 0x08, 0x08, 0x4c,
    0x9a, 0xd1, 0x79, 0xc6, 0x26, 0xce, 0x93, 0x07, 0x04, 0xa0, 0x60, 0xb2,
    0x5d, 0x15, 0xbe, 0x03, 0x2e, 0x15, 0xee, 0x67, 0x69, 0x52, 0x80, 0xda,
    0xc2, 0xe2, 0xb3, 0x86, 0x4a, 0xd3, 0x08, 0xbb, 0x54, 0xf9, 0xb6, 0x0b,
    0x0f, 0x1b, 0xd9, 0x21, 0x2e, 0x03, 0x21, 0xe1,
};
static const struct drbg_kat_no_reseed kat78_nor_t = {
    2, kat78_nor_entropyin, kat78_nor_nonce, kat78_nor_persstr,
    kat78_nor_addin0, kat78_nor_addin1, kat78_nor_retbytes
};
static const struct drbg_kat kat78_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat78_nor_t
};

static const unsigned char kat79_nor_entropyin[] = {
    0x5f, 0x5a, 0x57, 0xc0, 0xe7, 0x83, 0x90, 0xba, 0x0e, 0x9a, 0xa6, 0x07,
    0x31, 0x02, 0xa5, 0x63,
};
static const unsigned char kat79_nor_nonce[] = {
    0x46, 0x78, 0x27, 0x5a, 0x8b, 0x8d, 0xfc, 0xbd,
};
static const unsigned char kat79_nor_persstr[] = {0};
static const unsigned char kat79_nor_addin0[] = {
    0x1e, 0x45, 0x7c, 0xd7, 0x04, 0x84, 0xf7, 0x64, 0xfd, 0x44, 0x92, 0x1e,
    0xdd, 0x93, 0x1a, 0x93,
};
static const unsigned char kat79_nor_addin1[] = {
    0xc3, 0xdc, 0x45, 0xc2, 0x20, 0xb7, 0xbc, 0x21, 0xb3, 0x18, 0x05, 0x3f,
    0x36, 0x29, 0xf5, 0x6d,
};
static const unsigned char kat79_nor_retbytes[] = {
    0x1f, 0x58, 0x2c, 0xec, 0x88, 0xd7, 0x1a, 0xcd, 0xc1, 0x54, 0x4a, 0x89,
    0x91, 0x86, 0x28, 0xe9, 0xab, 0xb9, 0xf5, 0x8b, 0x8f, 0xf8, 0xb9, 0x62,
    0x59, 0xdf, 0x55, 0xbb, 0x1f, 0xa2, 0xb5, 0xa7, 0x52, 0x9c, 0x0e, 0x79,
    0x0e, 0xac, 0xea, 0xa8, 0xf4, 0xa8, 0x1f, 0x86, 0x22, 0x69, 0x8e, 0xbf,
    0xb2, 0xdb, 0x9d, 0xa2, 0x55, 0xa3, 0xf2, 0xb4, 0x51, 0x56, 0xa4, 0x21,
    0x9f, 0x6d, 0xea, 0x24, 0x98, 0xd8, 0xdf, 0xff, 0x59, 0x9a, 0x2d, 0x0c,
    0x0b, 0xc8, 0xf0, 0xfb, 0x22, 0x0d, 0xc6, 0xd5,
};
static const struct drbg_kat_no_reseed kat79_nor_t = {
    3, kat79_nor_entropyin, kat79_nor_nonce, kat79_nor_persstr,
    kat79_nor_addin0, kat79_nor_addin1, kat79_nor_retbytes
};
static const struct drbg_kat kat79_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat79_nor_t
};

static const unsigned char kat80_nor_entropyin[] = {
    0x90, 0xad, 0xa5, 0x5b, 0xf6, 0xde, 0xf3, 0x9b, 0xf8, 0xf6, 0x16, 0x99,
    0xe7, 0xc4, 0x58, 0x34,
};
static const unsigned char kat80_nor_nonce[] = {
    0xf0, 0x34, 0xdb, 0x6d, 0x32, 0x51, 0xb3, 0xea,
};
static const unsigned char kat80_nor_persstr[] = {0};
static const unsigned char kat80_nor_addin0[] = {
    0xe5, 0xf1, 0x58, 0x80, 0x93, 0x5a, 0xe7, 0x75, 0xfa, 0xc2, 0xc1, 0x5d,
    0xe2, 0x0a, 0x32, 0xa9,
};
static const unsigned char kat80_nor_addin1[] = {
    0xbf, 0xf5, 0x26, 0xe1, 0x08, 0xb7, 0x81, 0x8d, 0x5a, 0x30, 0xe9, 0xb0,
    0x62, 0x6a, 0x87, 0x94,
};
static const unsigned char kat80_nor_retbytes[] = {
    0x78, 0x79, 0x38, 0x71, 0xb3, 0xd2, 0x96, 0x67, 0xce, 0xdd, 0x43, 0x96,
    0x38, 0x90, 0x70, 0xc8, 0xae, 0xdf, 0x11, 0x5a, 0x0c, 0xbf, 0x3a, 0x4a,
    0xd2, 0xeb, 0xa1, 0xc3, 0xbc, 0xa0, 0x58, 0xf8, 0xc3, 0xde, 0xc4, 0xe2,
    0x3c, 0x63, 0x53, 0xff, 0x50, 0x16, 0x14, 0xea, 0x37, 0x67, 0x83, 0xe9,
    0xec, 0xd8, 0x39, 0x69, 0x10, 0xf0, 0x4f, 0xb7, 0x00, 0x5f, 0xb9, 0xf2,
    0x79, 0xea, 0xbd, 0x1b, 0x26, 0x20, 0x62, 0x64, 0x99, 0xcd, 0x77, 0x02,
    0xf4, 0x1e, 0x78, 0x1f, 0xd3, 0x41, 0x3d, 0x7a,
};
static const struct drbg_kat_no_reseed kat80_nor_t = {
    4, kat80_nor_entropyin, kat80_nor_nonce, kat80_nor_persstr,
    kat80_nor_addin0, kat80_nor_addin1, kat80_nor_retbytes
};
static const struct drbg_kat kat80_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat80_nor_t
};

static const unsigned char kat81_nor_entropyin[] = {
    0x46, 0x7d, 0x32, 0x15, 0x3d, 0x45, 0x03, 0x25, 0x5b, 0xfa, 0xbb, 0xbb,
    0xe4, 0x45, 0xb3, 0x88,
};
static const unsigned char kat81_nor_nonce[] = {
    0x18, 0x2b, 0x13, 0x92, 0x2f, 0xa8, 0x6d, 0x97,
};
static const unsigned char kat81_nor_persstr[] = {0};
static const unsigned char kat81_nor_addin0[] = {
    0x84, 0xe9, 0x61, 0x47, 0x21, 0x25, 0xdf, 0x37, 0x39, 0x6b, 0xe2, 0xd8,
    0xf4, 0x18, 0x7a, 0x9d,
};
static const unsigned char kat81_nor_addin1[] = {
    0x12, 0x32, 0x64, 0x4a, 0xaa, 0xb0, 0xb9, 0xa6, 0x61, 0xfa, 0x2a, 0xef,
    0x59, 0x76, 0xde, 0xe7,
};
static const unsigned char kat81_nor_retbytes[] = {
    0x20, 0x4a, 0xb4, 0x9a, 0x98, 0x3d, 0xbd, 0xda, 0x27, 0xbc, 0xd1, 0xa0,
    0xb8, 0x3d, 0xcd, 0x23, 0x33, 0x87, 0x7d, 0x6d, 0xb6, 0x7d, 0x6c, 0x31,
    0x0d, 0x49, 0x2c, 0x18, 0x27, 0x27, 0x4e, 0x34, 0x22, 0x6e, 0xfa, 0x02,
    0x88, 0x70, 0xc5, 0xc1, 0x25, 0x65, 0x6d, 0xdd, 0xaa, 0x77, 0xba, 0xbc,
    0x47, 0xdb, 0x6b, 0xda, 0xfa, 0x3c, 0x2f, 0xad, 0xe1, 0x2d, 0x8e, 0xc5,
    0xf3, 0xc8, 0xb8, 0x1f, 0x81, 0xb1, 0xe9, 0xee, 0xf3, 0x8f, 0x33, 0xcd,
    0x84, 0x02, 0x34, 0xd3, 0x97, 0x67, 0xa9, 0x7e,
};
static const struct drbg_kat_no_reseed kat81_nor_t = {
    5, kat81_nor_entropyin, kat81_nor_nonce, kat81_nor_persstr,
    kat81_nor_addin0, kat81_nor_addin1, kat81_nor_retbytes
};
static const struct drbg_kat kat81_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat81_nor_t
};

static const unsigned char kat82_nor_entropyin[] = {
    0x1d, 0x22, 0xfa, 0x6c, 0xda, 0x79, 0x05, 0x3b, 0x68, 0xbb, 0x6b, 0xa9,
    0x8b, 0x98, 0x5a, 0x52,
};
static const unsigned char kat82_nor_nonce[] = {
    0x2f, 0x7d, 0x01, 0xa0, 0xfb, 0x26, 0x14, 0x5a,
};
static const unsigned char kat82_nor_persstr[] = {0};
static const unsigned char kat82_nor_addin0[] = {
    0x6b, 0x78, 0x6c, 0xe4, 0x3d, 0xa7, 0x7a, 0x58, 0xa5, 0x31, 0xfb, 0xac,
    0x74, 0x93, 0xec, 0x2e,
};
static const unsigned char kat82_nor_addin1[] = {
    0x91, 0x7b, 0x22, 0x0b, 0x80, 0x97, 0x93, 0xa4, 0x19, 0x38, 0x27, 0xe4,
    0xfb, 0xf0, 0xdb, 0x70,
};
static const unsigned char kat82_nor_retbytes[] = {
    0xbf, 0xdc, 0x3c, 0x80, 0x06, 0x81, 0xac, 0xb5, 0x33, 0x7b, 0xd3, 0xad,
    0x90, 0x4e, 0x26, 0xfa, 0xe6, 0x17, 0x4c, 0x3e, 0x57, 0x65, 0x79, 0xc4,
    0x95, 0xcc, 0xce, 0x73, 0xec, 0xab, 0xd5, 0x4a, 0x5e, 0x14, 0xe6, 0xa8,
    0x98, 0x0f, 0x6e, 0x8f, 0xb2, 0xce, 0xe6, 0x21, 0x34, 0x30, 0x92, 0xd0,
    0x45, 0x59, 0x45, 0x1e, 0xe5, 0xe1, 0x6d, 0xee, 0xcb, 0xaf, 0x57, 0xd1,
    0xbc, 0x4c, 0xb5, 0x08, 0x7b, 0x32, 0x91, 0x94, 0x1e, 0xf7, 0xe6, 0xc6,
    0xf5, 0x11, 0xa8, 0x69, 0xee, 0x2d, 0x57, 0xee,
};
static const struct drbg_kat_no_reseed kat82_nor_t = {
    6, kat82_nor_entropyin, kat82_nor_nonce, kat82_nor_persstr,
    kat82_nor_addin0, kat82_nor_addin1, kat82_nor_retbytes
};
static const struct drbg_kat kat82_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat82_nor_t
};

static const unsigned char kat83_nor_entropyin[] = {
    0x54, 0xb5, 0xfd, 0xd7, 0xdd, 0x2e, 0x7e, 0x1c, 0x4f, 0x09, 0xa6, 0x11,
    0x8e, 0x4e, 0x60, 0xeb,
};
static const unsigned char kat83_nor_nonce[] = {
    0x81, 0x62, 0x37, 0x9c, 0x2b, 0x3d, 0xd9, 0x7f,
};
static const unsigned char kat83_nor_persstr[] = {0};
static const unsigned char kat83_nor_addin0[] = {
    0xb2, 0xed, 0x50, 0x78, 0xe1, 0x76, 0x44, 0x7a, 0x17, 0x98, 0x6e, 0x98,
    0xf9, 0x82, 0x7a, 0x43,
};
static const unsigned char kat83_nor_addin1[] = {
    0x5e, 0x6a, 0x48, 0x3c, 0xc9, 0xf2, 0x3e, 0x5c, 0x9d, 0x96, 0x95, 0x2b,
    0xa7, 0xcf, 0x10, 0x5e,
};
static const unsigned char kat83_nor_retbytes[] = {
    0x86, 0x7e, 0xc6, 0xf9, 0x25, 0xc5, 0x22, 0x7c, 0x74, 0x83, 0x67, 0x8b,
    0x4b, 0x09, 0x85, 0xf9, 0x02, 0xf1, 0xfd, 0x87, 0xf2, 0x2e, 0xda, 0xe2,
    0xa5, 0xad, 0xaf, 0x85, 0xcc, 0x77, 0x1b, 0x16, 0x40, 0x8e, 0xd6, 0x46,
    0x77, 0x8d, 0x87, 0x24, 0x37, 0x0b, 0xdb, 0x01, 0x37, 0xa8, 0x72, 0xb7,
    0x69, 0x94, 0xe8, 0x74, 0xde, 0x7f, 0xc2, 0xe7, 0x24, 0xde, 0x16, 0xf5,
    0xa0, 0x7d, 0xc9, 0x49, 0xd1, 0xc3, 0xbc, 0x9e, 0x73, 0xa4, 0xa0, 0xc2,
    0x99, 0x05, 0xc2, 0xb1, 0xcd, 0x99, 0xdf, 0x48,
};
static const struct drbg_kat_no_reseed kat83_nor_t = {
    7, kat83_nor_entropyin, kat83_nor_nonce, kat83_nor_persstr,
    kat83_nor_addin0, kat83_nor_addin1, kat83_nor_retbytes
};
static const struct drbg_kat kat83_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat83_nor_t
};

static const unsigned char kat84_nor_entropyin[] = {
    0xbb, 0x20, 0x61, 0x7c, 0xc0, 0xf7, 0xed, 0x8e, 0xf1, 0x29, 0xa7, 0xa0,
    0xe4, 0x78, 0x4c, 0x58,
};
static const unsigned char kat84_nor_nonce[] = {
    0xcb, 0x3b, 0x83, 0xd6, 0x13, 0xcc, 0x85, 0xdc,
};
static const unsigned char kat84_nor_persstr[] = {0};
static const unsigned char kat84_nor_addin0[] = {
    0xb1, 0x41, 0x9e, 0x3d, 0xbf, 0x91, 0xe1, 0x55, 0xcd, 0x32, 0x07, 0xea,
    0xd5, 0x28, 0xb2, 0x3d,
};
static const unsigned char kat84_nor_addin1[] = {
    0x9d, 0x54, 0x88, 0xca, 0x66, 0x4c, 0x2b, 0x19, 0x79, 0x48, 0x9a, 0x4c,
    0xf7, 0x42, 0x0e, 0x8d,
};
static const unsigned char kat84_nor_retbytes[] = {
    0x97, 0xee, 0x02, 0xe3, 0x86, 0xb0, 0x1b, 0x92, 0x9b, 0xde, 0x52, 0x2e,
    0x5f, 0x19, 0x8c, 0x85, 0xed, 0x03, 0x10, 0x25, 0xf4, 0xfd, 0x3f, 0x80,
    0x5c, 0x8a, 0xce, 0xab, 0x9d, 0x43, 0x28, 0x6f, 0x7e, 0xc7, 0x14, 0x22,
    0x82, 0x8b, 0x42, 0x70, 0x06, 0xd0, 0x35, 0xdf, 0x49, 0x4b, 0x86, 0xfb,
    0x9e, 0x71, 0x09, 0x01, 0x65, 0x18, 0x97, 0x18, 0x7e, 0x1c, 0xde, 0xa5,
    0xa6, 0x49, 0x41, 0x8c, 0x87, 0x24, 0xf0, 0xe6, 0x41, 0x26, 0x9a, 0x03,
    0x12, 0x72, 0x52, 0xb2, 0x9d, 0x79, 0x49, 0x4e,
};
static const struct drbg_kat_no_reseed kat84_nor_t = {
    8, kat84_nor_entropyin, kat84_nor_nonce, kat84_nor_persstr,
    kat84_nor_addin0, kat84_nor_addin1, kat84_nor_retbytes
};
static const struct drbg_kat kat84_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat84_nor_t
};

static const unsigned char kat85_nor_entropyin[] = {
    0xe9, 0xae, 0x7f, 0xab, 0x36, 0xe1, 0x64, 0x42, 0xfc, 0x6f, 0x88, 0xb7,
    0xd8, 0x90, 0xd7, 0x3d,
};
static const unsigned char kat85_nor_nonce[] = {
    0x37, 0xf7, 0x21, 0xf2, 0xb2, 0x27, 0xf1, 0x72,
};
static const unsigned char kat85_nor_persstr[] = {0};
static const unsigned char kat85_nor_addin0[] = {
    0x63, 0xf9, 0x2e, 0xc7, 0x0f, 0x47, 0x45, 0x8c, 0xa0, 0x38, 0x8a, 0xcb,
    0x1d, 0xc5, 0x51, 0xd2,
};
static const unsigned char kat85_nor_addin1[] = {
    0x85, 0x33, 0x11, 0xc7, 0x50, 0x63, 0xad, 0xa9, 0x9d, 0x25, 0xba, 0xa6,
    0x48, 0x4f, 0xc9, 0x53,
};
static const unsigned char kat85_nor_retbytes[] = {
    0x2c, 0xa7, 0x94, 0x59, 0xfa, 0x30, 0x12, 0x19, 0x88, 0x12, 0x13, 0xcd,
    0x3e, 0x0c, 0x14, 0xcb, 0x00, 0x4d, 0x12, 0xe1, 0xb0, 0xc7, 0x76, 0xfd,
    0x4f, 0x06, 0x32, 0xf9, 0x16, 0x73, 0xcb, 0x67, 0xa6, 0x65, 0x22, 0xfb,
    0x72, 0x4b, 0xe1, 0x21, 0xf8, 0xb4, 0x6c, 0xfb, 0xd1, 0x22, 0xcf, 0xd2,
    0xb1, 0x6d, 0x36, 0xf2, 0x7d, 0x17, 0x35, 0xe5, 0xf1, 0x73, 0x2a, 0xca,
    0xe8, 0x5b, 0xf2, 0x0f, 0xb4, 0xb5, 0x5c, 0x53, 0xbd, 0xc9, 0x30, 0x90,
    0x62, 0xa6, 0x64, 0x9f, 0xb5, 0x0f, 0xbc, 0xb2,
};
static const struct drbg_kat_no_reseed kat85_nor_t = {
    9, kat85_nor_entropyin, kat85_nor_nonce, kat85_nor_persstr,
    kat85_nor_addin0, kat85_nor_addin1, kat85_nor_retbytes
};
static const struct drbg_kat kat85_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat85_nor_t
};

static const unsigned char kat86_nor_entropyin[] = {
    0xef, 0x8b, 0x69, 0xc2, 0x11, 0x3d, 0x3e, 0x83, 0x87, 0x50, 0x0b, 0x8e,
    0x91, 0xe6, 0xd0, 0x77,
};
static const unsigned char kat86_nor_nonce[] = {
    0xa4, 0xb2, 0x5b, 0xcb, 0xe9, 0x2a, 0x9e, 0x5a,
};
static const unsigned char kat86_nor_persstr[] = {0};
static const unsigned char kat86_nor_addin0[] = {
    0x3a, 0x42, 0x62, 0x11, 0xe7, 0x6c, 0x02, 0x46, 0xcf, 0x25, 0x82, 0xa0,
    0x7e, 0x5c, 0xa6, 0x81,
};
static const unsigned char kat86_nor_addin1[] = {
    0xd8, 0x72, 0x1b, 0x09, 0xb2, 0x40, 0x48, 0x67, 0x61, 0xee, 0x0c, 0x67,
    0x0e, 0x5a, 0xf4, 0x73,
};
static const unsigned char kat86_nor_retbytes[] = {
    0x05, 0x5e, 0x5f, 0x05, 0x8b, 0x05, 0x43, 0x77, 0x83, 0x0b, 0xa4, 0xc7,
    0x31, 0x61, 0x7f, 0x9a, 0xcf, 0x16, 0x4f, 0xc5, 0xe2, 0x51, 0x74, 0x23,
    0x4e, 0x05, 0x20, 0x95, 0x41, 0x0b, 0x91, 0x24, 0x36, 0x66, 0xef, 0x57,
    0x33, 0x53, 0x29, 0x90, 0x89, 0x43, 0xc6, 0xd8, 0x8c, 0xfc, 0x00, 0xee,
    0x3f, 0x0d, 0xc0, 0x00, 0x23, 0xec, 0x32, 0xcf, 0xc4, 0x90, 0xe8, 0x2c,
    0xfb, 0xf4, 0x35, 0xe3, 0x88, 0x94, 0x98, 0x15, 0xdf, 0xc0, 0x05, 0xfc,
    0x03, 0x93, 0xd1, 0xd5, 0xef, 0xf7, 0xfd, 0xcf,
};
static const struct drbg_kat_no_reseed kat86_nor_t = {
    10, kat86_nor_entropyin, kat86_nor_nonce, kat86_nor_persstr,
    kat86_nor_addin0, kat86_nor_addin1, kat86_nor_retbytes
};
static const struct drbg_kat kat86_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat86_nor_t
};

static const unsigned char kat87_nor_entropyin[] = {
    0x1c, 0x98, 0x41, 0x2f, 0x62, 0xd9, 0x22, 0x0e, 0xbd, 0x84, 0x1b, 0x37,
    0x91, 0x28, 0xb1, 0x89,
};
static const unsigned char kat87_nor_nonce[] = {
    0xe1, 0xbe, 0x24, 0x32, 0x43, 0x1e, 0x83, 0xeb,
};
static const unsigned char kat87_nor_persstr[] = {0};
static const unsigned char kat87_nor_addin0[] = {
    0x6b, 0x43, 0xfe, 0x77, 0x0e, 0x72, 0x49, 0x7c, 0x13, 0x68, 0x5a, 0xa5,
    0x58, 0x72, 0x7f, 0x2d,
};
static const unsigned char kat87_nor_addin1[] = {
    0x96, 0x82, 0xfa, 0x8b, 0x49, 0x8d, 0x85, 0x3f, 0x3b, 0x43, 0x64, 0x98,
    0xaa, 0x68, 0x35, 0x8b,
};
static const unsigned char kat87_nor_retbytes[] = {
    0xca, 0x10, 0x79, 0xd6, 0x86, 0x75, 0x4c, 0x55, 0x38, 0x82, 0xab, 0x96,
    0x3d, 0xbd, 0x43, 0xa9, 0x53, 0xbc, 0x76, 0xbc, 0x77, 0x9b, 0x94, 0x5c,
    0xf3, 0x80, 0xac, 0x6d, 0x38, 0xd2, 0x12, 0x44, 0xa1, 0xbc, 0x14, 0x63,
    0x20, 0x21, 0x40, 0x41, 0xbb, 0xb5, 0x56, 0x04, 0x39, 0xfa, 0xb9, 0x97,
    0x21, 0xcc, 0xdc, 0xa6, 0x4f, 0x79, 0xb5, 0x57, 0xf6, 0xcd, 0x1e, 0x05,
    0x1c, 0x6b, 0x41, 0xd3, 0xa3, 0xbb, 0x86, 0x7c, 0xae, 0xdb, 0x45, 0x75,
    0x74, 0x9c, 0x5c, 0xd4, 0x51, 0xb6, 0x95, 0xe3,
};
static const struct drbg_kat_no_reseed kat87_nor_t = {
    11, kat87_nor_entropyin, kat87_nor_nonce, kat87_nor_persstr,
    kat87_nor_addin0, kat87_nor_addin1, kat87_nor_retbytes
};
static const struct drbg_kat kat87_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat87_nor_t
};

static const unsigned char kat88_nor_entropyin[] = {
    0xd7, 0xf7, 0x5f, 0x88, 0x96, 0xce, 0xcb, 0xd6, 0x49, 0x27, 0x32, 0x0d,
    0xbb, 0xe7, 0xa7, 0x1c,
};
static const unsigned char kat88_nor_nonce[] = {
    0xbb, 0x0e, 0x49, 0xae, 0x3b, 0x40, 0xe2, 0x8b,
};
static const unsigned char kat88_nor_persstr[] = {0};
static const unsigned char kat88_nor_addin0[] = {
    0xcd, 0xd3, 0x5d, 0x76, 0xe4, 0xdc, 0xec, 0xea, 0xe9, 0x18, 0xa3, 0xf5,
    0x8d, 0xc1, 0x1a, 0xe2,
};
static const unsigned char kat88_nor_addin1[] = {
    0xef, 0xdd, 0xd6, 0x12, 0x1e, 0x3f, 0x27, 0x66, 0x5e, 0x0e, 0x58, 0x5b,
    0x03, 0x5f, 0x77, 0x1f,
};
static const unsigned char kat88_nor_retbytes[] = {
    0x23, 0x6b, 0xf7, 0x25, 0x59, 0xc4, 0x8b, 0xa0, 0xea, 0x7d, 0x0d, 0x8a,
    0x6c, 0x76, 0xe0, 0x6d, 0xbf, 0x96, 0x6a, 0xf7, 0xcc, 0x29, 0xb1, 0x68,
    0xfd, 0x79, 0x8a, 0xec, 0xbb, 0x00, 0xdd, 0xda, 0x6e, 0x82, 0xcf, 0x03,
    0xd4, 0x30, 0xe0, 0x83, 0xcb, 0x4f, 0xe4, 0x2a, 0x9e, 0xb5, 0x47, 0xe2,
    0xc8, 0x2a, 0xd0, 0xdd, 0x62, 0x88, 0x2c, 0x17, 0x31, 0xde, 0x37, 0xe8,
    0x87, 0x44, 0x19, 0xe8, 0x96, 0xa4, 0xa5, 0x3e, 0x44, 0x8e, 0x98, 0xa0,
    0xb3, 0x08, 0x87, 0xa7, 0xfc, 0x8e, 0x50, 0xb5,
};
static const struct drbg_kat_no_reseed kat88_nor_t = {
    12, kat88_nor_entropyin, kat88_nor_nonce, kat88_nor_persstr,
    kat88_nor_addin0, kat88_nor_addin1, kat88_nor_retbytes
};
static const struct drbg_kat kat88_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat88_nor_t
};

static const unsigned char kat89_nor_entropyin[] = {
    0x40, 0x84, 0xb0, 0x5f, 0xd6, 0x2d, 0xa1, 0x89, 0x02, 0x45, 0xed, 0x6b,
    0xcc, 0xe6, 0xcd, 0x72,
};
static const unsigned char kat89_nor_nonce[] = {
    0xcf, 0xc3, 0x62, 0xec, 0x0d, 0xbf, 0x23, 0xe0,
};
static const unsigned char kat89_nor_persstr[] = {0};
static const unsigned char kat89_nor_addin0[] = {
    0xa5, 0x29, 0x2a, 0x17, 0x23, 0x39, 0xa4, 0x29, 0x1f, 0xb3, 0x0b, 0x9a,
    0x33, 0x25, 0xaa, 0x97,
};
static const unsigned char kat89_nor_addin1[] = {
    0xea, 0xe3, 0xdd, 0x07, 0x3e, 0xdc, 0xf8, 0x4c, 0xaa, 0xc2, 0xf6, 0x16,
    0x19, 0x73, 0xdd, 0xaa,
};
static const unsigned char kat89_nor_retbytes[] = {
    0xb0, 0x30, 0x97, 0x4c, 0xd3, 0xf0, 0xa2, 0x12, 0x51, 0x82, 0x1b, 0xe6,
    0x5a, 0x0a, 0x2c, 0x9d, 0x8b, 0xaa, 0xca, 0xaa, 0x06, 0x80, 0x4b, 0x6e,
    0x74, 0xbf, 0xe0, 0x0b, 0xa0, 0x69, 0x1a, 0x05, 0x52, 0x11, 0xac, 0xbd,
    0xeb, 0x2f, 0xa7, 0xda, 0x96, 0x41, 0xab, 0x4f, 0x8b, 0xbf, 0x74, 0xab,
    0x30, 0x32, 0xa9, 0x8f, 0x24, 0x17, 0x69, 0x7f, 0xbd, 0xf8, 0x0d, 0x74,
    0x88, 0x77, 0x52, 0x07, 0x26, 0x59, 0xf8, 0x17, 0xf1, 0x68, 0x17, 0x53,
    0x41, 0xdd, 0xef, 0x5b, 0x6e, 0xfd, 0x9f, 0xb6,
};
static const struct drbg_kat_no_reseed kat89_nor_t = {
    13, kat89_nor_entropyin, kat89_nor_nonce, kat89_nor_persstr,
    kat89_nor_addin0, kat89_nor_addin1, kat89_nor_retbytes
};
static const struct drbg_kat kat89_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat89_nor_t
};

static const unsigned char kat90_nor_entropyin[] = {
    0x46, 0xeb, 0xaf, 0x3f, 0xb0, 0x58, 0x3f, 0xc5, 0xb8, 0x01, 0x3e, 0x14,
    0x58, 0x4c, 0xe7, 0x17,
};
static const unsigned char kat90_nor_nonce[] = {
    0xd5, 0xd6, 0x8f, 0x69, 0x56, 0x7c, 0xb2, 0x90,
};
static const unsigned char kat90_nor_persstr[] = {0};
static const unsigned char kat90_nor_addin0[] = {
    0x32, 0xd0, 0x9b, 0x60, 0x4a, 0x65, 0xdc, 0x8d, 0xaa, 0x35, 0xcd, 0xc3,
    0x41, 0x41, 0xb7, 0x51,
};
static const unsigned char kat90_nor_addin1[] = {
    0xb8, 0x18, 0x6a, 0x29, 0x4c, 0x78, 0x24, 0xb7, 0xc5, 0x50, 0xc1, 0x05,
    0x4b, 0xad, 0xec, 0x00,
};
static const unsigned char kat90_nor_retbytes[] = {
    0xae, 0x9a, 0x09, 0x1c, 0xfa, 0xfb, 0xf0, 0xe7, 0x4c, 0x2b, 0xe8, 0xad,
    0x4b, 0x98, 0x4e, 0x82, 0x4a, 0x24, 0xe6, 0x5b, 0xa7, 0x61, 0x0b, 0x0f,
    0x3a, 0xb1, 0x75, 0x0e, 0x2f, 0x12, 0xde, 0x16, 0x20, 0xdb, 0x6b, 0xb8,
    0xc4, 0x93, 0xb3, 0xd8, 0xb0, 0x6a, 0xb7, 0x8e, 0x69, 0xcf, 0x2d, 0xff,
    0xd7, 0x3d, 0x43, 0x22, 0xa6, 0x7e, 0xe7, 0x72, 0x5a, 0xad, 0x84, 0xfb,
    0x45, 0x8b, 0x8f, 0x26, 0xcf, 0x04, 0x84, 0x68, 0x50, 0x20, 0x2e, 0x53,
    0xc8, 0x74, 0x21, 0x32, 0x21, 0xe7, 0x61, 0xe5,
};
static const struct drbg_kat_no_reseed kat90_nor_t = {
    14, kat90_nor_entropyin, kat90_nor_nonce, kat90_nor_persstr,
    kat90_nor_addin0, kat90_nor_addin1, kat90_nor_retbytes
};
static const struct drbg_kat kat90_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat90_nor_t
};

static const unsigned char kat91_nor_entropyin[] = {
    0xf0, 0xfd, 0x21, 0x99, 0xc6, 0x25, 0x26, 0x28, 0x63, 0x99, 0x76, 0x62,
    0x8c, 0xb7, 0x1c, 0x39,
};
static const unsigned char kat91_nor_nonce[] = {
    0xe2, 0xda, 0xeb, 0xbc, 0x0c, 0xe7, 0xa1, 0x83,
};
static const unsigned char kat91_nor_persstr[] = {
    0x5f, 0xa5, 0xee, 0xe4, 0xf3, 0x6f, 0x5d, 0x42, 0xf5, 0x93, 0xc5, 0x7d,
    0xe8, 0x75, 0x43, 0xab,
};
static const unsigned char kat91_nor_addin0[] = {0};
static const unsigned char kat91_nor_addin1[] = {0};
static const unsigned char kat91_nor_retbytes[] = {
    0x3e, 0x1a, 0x75, 0xdf, 0x86, 0x12, 0xb4, 0x0e, 0xce, 0xad, 0xe7, 0xa8,
    0xc7, 0x3f, 0x0c, 0x11, 0xca, 0x40, 0x46, 0xa3, 0xa1, 0x3e, 0x15, 0x9a,
    0xf3, 0xfb, 0x9b, 0x96, 0x9f, 0x38, 0x99, 0x4f, 0xd1, 0xbf, 0x68, 0x96,
    0xa2, 0x68, 0x12, 0x55, 0x55, 0x93, 0x4a, 0xee, 0xa0, 0x14, 0x5a, 0x8a,
    0x83, 0xa7, 0x80, 0xf2, 0xd7, 0x1c, 0x12, 0x68, 0x9c, 0xaa, 0x9f, 0xae,
    0xc6, 0xd0, 0xd1, 0x98, 0x75, 0x8d, 0x4a, 0x06, 0x64, 0x25, 0x0b, 0xd4,
    0xd8, 0x9f, 0xe9, 0x61, 0x4a, 0xdc, 0xa3, 0xe3,
};
static const struct drbg_kat_no_reseed kat91_nor_t = {
    0, kat91_nor_entropyin, kat91_nor_nonce, kat91_nor_persstr,
    kat91_nor_addin0, kat91_nor_addin1, kat91_nor_retbytes
};
static const struct drbg_kat kat91_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat91_nor_t
};

static const unsigned char kat92_nor_entropyin[] = {
    0x75, 0xb8, 0xf6, 0x6d, 0xb0, 0x2e, 0x8a, 0x39, 0xd4, 0x8f, 0xf6, 0xbd,
    0xb8, 0x64, 0x39, 0x2a,
};
static const unsigned char kat92_nor_nonce[] = {
    0x19, 0x88, 0x3b, 0xc6, 0x81, 0x29, 0x39, 0x91,
};
static const unsigned char kat92_nor_persstr[] = {
    0x1c, 0x5f, 0x89, 0x0d, 0xcf, 0x7c, 0x4c, 0x81, 0xfb, 0xe5, 0x9f, 0x12,
    0xdf, 0xa4, 0x87, 0xbf,
};
static const unsigned char kat92_nor_addin0[] = {0};
static const unsigned char kat92_nor_addin1[] = {0};
static const unsigned char kat92_nor_retbytes[] = {
    0x9a, 0x4d, 0x78, 0xb6, 0xee, 0x54, 0xbc, 0x20, 0x04, 0x88, 0xfb, 0xc5,
    0x22, 0xd9, 0x67, 0xc6, 0xe3, 0x97, 0xb1, 0x2f, 0x29, 0xde, 0xdd, 0x2a,
    0x44, 0xb4, 0xc2, 0xd2, 0xa5, 0xe0, 0xc5, 0xde, 0x7d, 0x19, 0xa5, 0xcd,
    0x77, 0x8f, 0xb0, 0xa0, 0xcb, 0xe1, 0xd9, 0x89, 0x38, 0x65, 0xa3, 0x78,
    0x38, 0x0d, 0x12, 0xb0, 0x58, 0x2c, 0x35, 0xbe, 0xfe, 0xba, 0xa1, 0x35,
    0xda, 0x44, 0xdc, 0x08, 0x40, 0x31, 0x65, 0xa0, 0x01, 0xbb, 0x31, 0x80,
    0xcf, 0xa5, 0x64, 0x49, 0x92, 0x11, 0x2f, 0x26,
};
static const struct drbg_kat_no_reseed kat92_nor_t = {
    1, kat92_nor_entropyin, kat92_nor_nonce, kat92_nor_persstr,
    kat92_nor_addin0, kat92_nor_addin1, kat92_nor_retbytes
};
static const struct drbg_kat kat92_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat92_nor_t
};

static const unsigned char kat93_nor_entropyin[] = {
    0x8d, 0xdc, 0x1d, 0x1a, 0xa7, 0x3c, 0x79, 0x75, 0x02, 0xf9, 0x71, 0xbb,
    0xd5, 0x9e, 0xfe, 0xc6,
};
static const unsigned char kat93_nor_nonce[] = {
    0xe5, 0xb5, 0x4a, 0x7a, 0xf7, 0x6f, 0xb1, 0x20,
};
static const unsigned char kat93_nor_persstr[] = {
    0x37, 0xec, 0x27, 0x25, 0x34, 0x7b, 0x49, 0x08, 0xab, 0xe8, 0xf6, 0x9e,
    0x22, 0x83, 0x6d, 0x4b,
};
static const unsigned char kat93_nor_addin0[] = {0};
static const unsigned char kat93_nor_addin1[] = {0};
static const unsigned char kat93_nor_retbytes[] = {
    0xbd, 0x96, 0x54, 0x9e, 0xd8, 0xc0, 0x94, 0xbb, 0x04, 0x38, 0x70, 0x42,
    0x4d, 0x38, 0xa7, 0x14, 0x79, 0xe4, 0x4a, 0x3f, 0x47, 0xe9, 0x2e, 0xc0,
    0x10, 0xa1, 0x01, 0x8a, 0xd5, 0xee, 0xaf, 0xe8, 0x32, 0xfd, 0xa5, 0x41,
    0x55, 0x39, 0x48, 0xc9, 0x75, 0x54, 0x0a, 0xd8, 0xe7, 0x3a, 0xcb, 0x36,
    0x48, 0x68, 0x14, 0x95, 0x04, 0xdf, 0x35, 0x74, 0xa0, 0x4b, 0x1c, 0x34,
    0xfe, 0x7b, 0xd7, 0x00, 0x92, 0x11, 0xc2, 0x7f, 0x30, 0xcd, 0x34, 0xc1,
    0xfd, 0xd4, 0x10, 0xb8, 0x32, 0x01, 0x94, 0x7a,
};
static const struct drbg_kat_no_reseed kat93_nor_t = {
    2, kat93_nor_entropyin, kat93_nor_nonce, kat93_nor_persstr,
    kat93_nor_addin0, kat93_nor_addin1, kat93_nor_retbytes
};
static const struct drbg_kat kat93_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat93_nor_t
};

static const unsigned char kat94_nor_entropyin[] = {
    0x3c, 0x03, 0x46, 0x41, 0x29, 0xb1, 0x3d, 0x93, 0xe4, 0xbc, 0x07, 0xf5,
    0x73, 0x1a, 0x59, 0xb6,
};
static const unsigned char kat94_nor_nonce[] = {
    0x5f, 0xfe, 0xa1, 0x2f, 0xd5, 0xf6, 0x2d, 0xcc,
};
static const unsigned char kat94_nor_persstr[] = {
    0x58, 0x82, 0x04, 0xa0, 0x22, 0x72, 0x7a, 0xfa, 0x3f, 0x94, 0xce, 0xb7,
    0xbb, 0x8e, 0xdf, 0xbb,
};
static const unsigned char kat94_nor_addin0[] = {0};
static const unsigned char kat94_nor_addin1[] = {0};
static const unsigned char kat94_nor_retbytes[] = {
    0x17, 0xa2, 0xfb, 0x65, 0xd6, 0x6d, 0x8a, 0x7f, 0x31, 0x92, 0x46, 0x38,
    0xa9, 0x44, 0x51, 0x36, 0xca, 0x6e, 0x5b, 0xdc, 0x35, 0x99, 0xa5, 0x50,
    0x1f, 0xfe, 0x97, 0x62, 0x89, 0x29, 0x7b, 0xe9, 0xee, 0x00, 0x28, 0x21,
    0x59, 0x01, 0xb4, 0xdf, 0xe6, 0x64, 0x30, 0xfc, 0xa4, 0xb0, 0xe4, 0xc8,
    0xf0, 0xe0, 0xc7, 0x4d, 0x98, 0x69, 0xb5, 0x8c, 0x7b, 0x67, 0xb5, 0xfa,
    0xc5, 0x60, 0x34, 0x9f, 0x2c, 0x60, 0x31, 0x8c, 0x79, 0xa7, 0xf3, 0xc1,
    0x43, 0xc5, 0xf6, 0x1c, 0xaa, 0x37, 0x14, 0xe5,
};
static const struct drbg_kat_no_reseed kat94_nor_t = {
    3, kat94_nor_entropyin, kat94_nor_nonce, kat94_nor_persstr,
    kat94_nor_addin0, kat94_nor_addin1, kat94_nor_retbytes
};
static const struct drbg_kat kat94_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat94_nor_t
};

static const unsigned char kat95_nor_entropyin[] = {
    0x05, 0xb1, 0x7d, 0x99, 0x62, 0xee, 0x6b, 0xa8, 0xcf, 0xd6, 0xf5, 0xe8,
    0x49, 0xb8, 0x45, 0x7b,
};
static const unsigned char kat95_nor_nonce[] = {
    0xf6, 0xd7, 0xa5, 0xae, 0xa8, 0x8a, 0x24, 0xbe,
};
static const unsigned char kat95_nor_persstr[] = {
    0x13, 0xd9, 0xc6, 0x0d, 0xe4, 0xb0, 0x4f, 0xc6, 0x3d, 0x07, 0x48, 0x5d,
    0x0e, 0xb2, 0x33, 0x7b,
};
static const unsigned char kat95_nor_addin0[] = {0};
static const unsigned char kat95_nor_addin1[] = {0};
static const unsigned char kat95_nor_retbytes[] = {
    0x80, 0xa2, 0x96, 0xe2, 0x08, 0x23, 0x24, 0xe1, 0xd7, 0xa8, 0x50, 0xab,
    0xb5, 0xee, 0xc2, 0x36, 0x94, 0xae, 0x69, 0x53, 0x45, 0xd5, 0x7f, 0x07,
    0xf3, 0xbc, 0x46, 0xc4, 0x2e, 0x5d, 0x80, 0x12, 0xb3, 0x7d, 0xad, 0xad,
    0xac, 0xa5, 0xf1, 0xfc, 0xb0, 0x08, 0x8c, 0x59, 0x00, 0x50, 0x58, 0x9e,
    0x5b, 0x10, 0xf8, 0xbf, 0xd9, 0x67, 0x22, 0xec, 0x0c, 0x56, 0x7b, 0x90,
    0xbe, 0xa5, 0x65, 0xed, 0xbe, 0xdd, 0x41, 0x5b, 0xab, 0x2f, 0xa6, 0x30,
    0xd9, 0x47, 0x38, 0xc4, 0x44, 0xdb, 0x14, 0xdc,
};
static const struct drbg_kat_no_reseed kat95_nor_t = {
    4, kat95_nor_entropyin, kat95_nor_nonce, kat95_nor_persstr,
    kat95_nor_addin0, kat95_nor_addin1, kat95_nor_retbytes
};
static const struct drbg_kat kat95_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat95_nor_t
};

static const unsigned char kat96_nor_entropyin[] = {
    0xe5, 0x36, 0xcf, 0x01, 0x38, 0x56, 0x6d, 0x88, 0x52, 0x55, 0x65, 0xc6,
    0xb6, 0xcd, 0x2f, 0x86,
};
static const unsigned char kat96_nor_nonce[] = {
    0xd0, 0x48, 0xa5, 0x18, 0x56, 0x90, 0x27, 0xf9,
};
static const unsigned char kat96_nor_persstr[] = {
    0x68, 0xe2, 0x28, 0x6e, 0xd4, 0x68, 0x33, 0x3e, 0xe2, 0x7c, 0x57, 0x34,
    0x53, 0x56, 0x99, 0x94,
};
static const unsigned char kat96_nor_addin0[] = {0};
static const unsigned char kat96_nor_addin1[] = {0};
static const unsigned char kat96_nor_retbytes[] = {
    0x61, 0x53, 0x84, 0xd1, 0x98, 0x2f, 0x75, 0xf8, 0xe8, 0xc0, 0x08, 0xfd,
    0x65, 0x6b, 0x3b, 0xe0, 0x95, 0xdc, 0x6a, 0x1b, 0x15, 0x72, 0x5c, 0xdc,
    0x22, 0x06, 0xab, 0xb5, 0xb0, 0xe7, 0xd7, 0x25, 0x14, 0xee, 0x4f, 0x47,
    0x7d, 0x12, 0x0a, 0x4b, 0x1f, 0xa8, 0x1c, 0x40, 0xec, 0x88, 0xbc, 0x7c,
    0xb4, 0x04, 0x8d, 0xf8, 0x3c, 0x94, 0xc7, 0xb2, 0xb6, 0xf7, 0xfd, 0x91,
    0x6f, 0xfc, 0x17, 0x67, 0x81, 0x51, 0x7d, 0x4d, 0xec, 0xb9, 0xb8, 0xab,
    0xe5, 0x3c, 0x22, 0x89, 0xdb, 0x4b, 0x41, 0xef,
};
static const struct drbg_kat_no_reseed kat96_nor_t = {
    5, kat96_nor_entropyin, kat96_nor_nonce, kat96_nor_persstr,
    kat96_nor_addin0, kat96_nor_addin1, kat96_nor_retbytes
};
static const struct drbg_kat kat96_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat96_nor_t
};

static const unsigned char kat97_nor_entropyin[] = {
    0x0f, 0x48, 0xee, 0xcd, 0x99, 0x13, 0xb7, 0x10, 0x4f, 0xf8, 0xb8, 0x80,
    0x0a, 0x2b, 0x13, 0x16,
};
static const unsigned char kat97_nor_nonce[] = {
    0x6d, 0xe0, 0x36, 0xfe, 0x5a, 0x20, 0x1b, 0x5f,
};
static const unsigned char kat97_nor_persstr[] = {
    0xd0, 0x0c, 0x23, 0x70, 0x30, 0xad, 0xba, 0x87, 0x4e, 0x1e, 0x6b, 0xcb,
    0xcf, 0x2a, 0x47, 0xbf,
};
static const unsigned char kat97_nor_addin0[] = {0};
static const unsigned char kat97_nor_addin1[] = {0};
static const unsigned char kat97_nor_retbytes[] = {
    0x36, 0xb7, 0x65, 0x62, 0xae, 0x5d, 0xc3, 0x05, 0x45, 0xe6, 0xce, 0x14,
    0xdd, 0xbc, 0xe1, 0x23, 0x30, 0xf1, 0xcf, 0xea, 0x88, 0x22, 0x7b, 0x8e,
    0xc1, 0x16, 0x54, 0xe6, 0x2b, 0x4e, 0x81, 0x06, 0x3a, 0x55, 0x74, 0x75,
    0x81, 0x63, 0xf4, 0x47, 0x8a, 0x28, 0x38, 0xaf, 0x25, 0x83, 0xec, 0xe2,
    0xe6, 0x9b, 0xfb, 0xf1, 0x45, 0x0f, 0x12, 0xac, 0x6e, 0x37, 0xa6, 0x94,
    0x8e, 0x7e, 0xc3, 0x67, 0x16, 0xdb, 0x22, 0xc4, 0x05, 0x65, 0xb1, 0xa1,
    0x6a, 0x4b, 0x06, 0xe2, 0x66, 0x59, 0xc3, 0x4f,
};
static const struct drbg_kat_no_reseed kat97_nor_t = {
    6, kat97_nor_entropyin, kat97_nor_nonce, kat97_nor_persstr,
    kat97_nor_addin0, kat97_nor_addin1, kat97_nor_retbytes
};
static const struct drbg_kat kat97_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat97_nor_t
};

static const unsigned char kat98_nor_entropyin[] = {
    0x66, 0x29, 0x30, 0xe8, 0xba, 0xe1, 0x4d, 0x3b, 0xe8, 0x7d, 0x96, 0x54,
    0x63, 0x5a, 0x20, 0x22,
};
static const unsigned char kat98_nor_nonce[] = {
    0x38, 0x36, 0xf4, 0xd7, 0xa5, 0x08, 0x59, 0x5d,
};
static const unsigned char kat98_nor_persstr[] = {
    0x74, 0x60, 0xd1, 0x6d, 0xa0, 0x1d, 0xfe, 0x50, 0x48, 0x70, 0xde, 0x63,
    0xbd, 0x4c, 0xf6, 0x0c,
};
static const unsigned char kat98_nor_addin0[] = {0};
static const unsigned char kat98_nor_addin1[] = {0};
static const unsigned char kat98_nor_retbytes[] = {
    0xc9, 0x9b, 0x9c, 0xea, 0x7f, 0x76, 0x10, 0xd5, 0x46, 0x32, 0xd3, 0x9b,
    0x45, 0x75, 0x25, 0xdf, 0xf4, 0xf0, 0x1c, 0x57, 0x98, 0x9f, 0x50, 0x94,
    0x9d, 0x18, 0x57, 0x99, 0x8f, 0x4a, 0xae, 0x14, 0xaa, 0xc5, 0x7a, 0x1c,
    0xc7, 0xad, 0x51, 0x38, 0xe9, 0x70, 0xca, 0x67, 0x8a, 0x7d, 0x4c, 0xbd,
    0x42, 0xc9, 0xd9, 0x61, 0xf1, 0x5b, 0x45, 0x66, 0x5d, 0xda, 0x08, 0xab,
    0x2c, 0xb1, 0xea, 0xff, 0x68, 0x90, 0xc7, 0xb6, 0x26, 0x84, 0xe2, 0x94,
    0xf6, 0xb3, 0x7f, 0x89, 0xf1, 0x11, 0x98, 0x35,
};
static const struct drbg_kat_no_reseed kat98_nor_t = {
    7, kat98_nor_entropyin, kat98_nor_nonce, kat98_nor_persstr,
    kat98_nor_addin0, kat98_nor_addin1, kat98_nor_retbytes
};
static const struct drbg_kat kat98_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat98_nor_t
};

static const unsigned char kat99_nor_entropyin[] = {
    0x3e, 0xf0, 0x96, 0xac, 0x9e, 0x72, 0xa9, 0x61, 0xf5, 0x68, 0x90, 0xf4,
    0xb5, 0x23, 0xf9, 0x5f,
};
static const unsigned char kat99_nor_nonce[] = {
    0x1c, 0x10, 0x04, 0x01, 0xa4, 0x94, 0x02, 0xcb,
};
static const unsigned char kat99_nor_persstr[] = {
    0xe2, 0x2d, 0x6f, 0x4b, 0x7f, 0xa5, 0x04, 0x63, 0xdb, 0x12, 0x13, 0x80,
    0x44, 0xcc, 0xfd, 0x37,
};
static const unsigned char kat99_nor_addin0[] = {0};
static const unsigned char kat99_nor_addin1[] = {0};
static const unsigned char kat99_nor_retbytes[] = {
    0x41, 0x38, 0x09, 0x17, 0x74, 0xbb, 0xf3, 0x94, 0xd9, 0x0b, 0x33, 0xc1,
    0xb6, 0x60, 0x17, 0xd8, 0x42, 0xc7, 0xc5, 0xe7, 0x22, 0x1f, 0x57, 0xf5,
    0x5e, 0xf0, 0xd6, 0x0d, 0xa4, 0x39, 0x8c, 0x50, 0xc7, 0xd2, 0xec, 0x98,
    0x13, 0x17, 0x76, 0x73, 0xb9, 0xf3, 0x36, 0xa0, 0xb8, 0x5a, 0x25, 0xf8,
    0x8e, 0x21, 0x56, 0xea, 0x5f, 0xb2, 0x25, 0x0b, 0x95, 0xeb, 0xc8, 0x79,
    0xa5, 0xac, 0x58, 0x58, 0x4a, 0xf8, 0x09, 0x4c, 0x31, 0xd7, 0x71, 0x83,
    0x5e, 0x26, 0x24, 0xd1, 0x1d, 0xb5, 0x01, 0x18,
};
static const struct drbg_kat_no_reseed kat99_nor_t = {
    8, kat99_nor_entropyin, kat99_nor_nonce, kat99_nor_persstr,
    kat99_nor_addin0, kat99_nor_addin1, kat99_nor_retbytes
};
static const struct drbg_kat kat99_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat99_nor_t
};

static const unsigned char kat100_nor_entropyin[] = {
    0xa3, 0x00, 0x92, 0xb4, 0xf4, 0x81, 0xc7, 0x4e, 0xfa, 0x85, 0xa2, 0x83,
    0x6a, 0x92, 0xdc, 0x2b,
};
static const unsigned char kat100_nor_nonce[] = {
    0xcd, 0x22, 0xe3, 0xe3, 0xe7, 0x02, 0x1c, 0x08,
};
static const unsigned char kat100_nor_persstr[] = {
    0x86, 0x02, 0xbf, 0x58, 0x9c, 0xf6, 0x70, 0xe6, 0xba, 0x78, 0x7b, 0x32,
    0x15, 0x3b, 0x2d, 0x3a,
};
static const unsigned char kat100_nor_addin0[] = {0};
static const unsigned char kat100_nor_addin1[] = {0};
static const unsigned char kat100_nor_retbytes[] = {
    0x67, 0x8f, 0x1b, 0xce, 0xdb, 0xc8, 0x5c, 0x20, 0x02, 0xe0, 0x4c, 0xf7,
    0x8b, 0xa8, 0xa8, 0x9e, 0x7b, 0x1d, 0x56, 0xfe, 0xe4, 0x20, 0xd2, 0x15,
    0xa0, 0xfb, 0xf3, 0x3e, 0x30, 0x1d, 0xe3, 0xf0, 0x75, 0xac, 0xde, 0xb3,
    0x36, 0x3b, 0x3c, 0x6c, 0x5d, 0x73, 0xed, 0x1d, 0x5a, 0x28, 0x43, 0x75,
    0xfc, 0x3b, 0x8b, 0x4c, 0x73, 0xa2, 0x74, 0x05, 0xd7, 0x57, 0x03, 0xea,
    0x6b, 0xd9, 0xb2, 0x95, 0x14, 0x8b, 0xa9, 0xb7, 0x03, 0x63, 0x9c, 0xff,
    0xee, 0x9b, 0x37, 0x80, 0x82, 0x31, 0x89, 0x40,
};
static const struct drbg_kat_no_reseed kat100_nor_t = {
    9, kat100_nor_entropyin, kat100_nor_nonce, kat100_nor_persstr,
    kat100_nor_addin0, kat100_nor_addin1, kat100_nor_retbytes
};
static const struct drbg_kat kat100_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat100_nor_t
};

static const unsigned char kat101_nor_entropyin[] = {
    0x25, 0x16, 0xc2, 0x2a, 0xa5, 0xae, 0xcf, 0xb0, 0x65, 0x94, 0xdb, 0x80,
    0x39, 0xfb, 0xab, 0x26,
};
static const unsigned char kat101_nor_nonce[] = {
    0x2b, 0x52, 0xac, 0x33, 0x99, 0xc0, 0xd7, 0xb4,
};
static const unsigned char kat101_nor_persstr[] = {
    0xed, 0x8c, 0xdb, 0xd1, 0x0a, 0x1f, 0xc7, 0xc4, 0x28, 0x0f, 0x39, 0x9a,
    0x93, 0xb8, 0xe4, 0x7a,
};
static const unsigned char kat101_nor_addin0[] = {0};
static const unsigned char kat101_nor_addin1[] = {0};
static const unsigned char kat101_nor_retbytes[] = {
    0xb9, 0x26, 0xbd, 0xf4, 0x38, 0xc6, 0xe3, 0x06, 0xa9, 0xac, 0x00, 0x84,
    0x29, 0x4c, 0x1a, 0x97, 0xef, 0x13, 0x68, 0xe1, 0x06, 0x24, 0x4e, 0xdd,
    0x40, 0x66, 0x24, 0x8a, 0x20, 0xd4, 0xf8, 0xd6, 0x01, 0xe4, 0x35, 0x83,
    0xb6, 0x82, 0xaf, 0xed, 0x51, 0x89, 0xf5, 0x1b, 0x6f, 0x06, 0xae, 0x00,
    0x44, 0x14, 0xcc, 0x66, 0x8c, 0x08, 0x2f, 0x88, 0x45, 0xa7, 0x07, 0x5e,
    0x5d, 0x39, 0x23, 0x15, 0x8f, 0xdd, 0x83, 0x77, 0xd1, 0x73, 0x51, 0xb9,
    0xd9, 0x24, 0xc8, 0xb8, 0xf3, 0x8b, 0xea, 0x4d,
};
static const struct drbg_kat_no_reseed kat101_nor_t = {
    10, kat101_nor_entropyin, kat101_nor_nonce, kat101_nor_persstr,
    kat101_nor_addin0, kat101_nor_addin1, kat101_nor_retbytes
};
static const struct drbg_kat kat101_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat101_nor_t
};

static const unsigned char kat102_nor_entropyin[] = {
    0x35, 0xed, 0xf0, 0x13, 0xd8, 0x0e, 0xec, 0x2d, 0x3b, 0x78, 0x57, 0x26,
    0x62, 0x9e, 0xe0, 0x74,
};
static const unsigned char kat102_nor_nonce[] = {
    0x14, 0x9c, 0x12, 0x60, 0xbd, 0x2b, 0x1d, 0x9c,
};
static const unsigned char kat102_nor_persstr[] = {
    0x92, 0x12, 0xa9, 0x5a, 0x4c, 0xf0, 0x85, 0x5e, 0x3e, 0xe8, 0xc5, 0x25,
    0x79, 0x14, 0xea, 0x36,
};
static const unsigned char kat102_nor_addin0[] = {0};
static const unsigned char kat102_nor_addin1[] = {0};
static const unsigned char kat102_nor_retbytes[] = {
    0x91, 0x0b, 0xfa, 0x60, 0x25, 0x83, 0xb1, 0xfa, 0x5c, 0x05, 0x91, 0x8d,
    0x34, 0x53, 0xac, 0x7e, 0xfc, 0x16, 0x30, 0x40, 0xcd, 0x34, 0x5f, 0xdd,
    0x2b, 0x5b, 0x34, 0x2d, 0x0d, 0xd6, 0xf4, 0x2e, 0xb2, 0xf8, 0x47, 0x93,
    0x27, 0x12, 0xcf, 0x12, 0x19, 0xae, 0x29, 0x76, 0x65, 0x32, 0xab, 0x16,
    0xe6, 0xd8, 0x11, 0x2c, 0xd6, 0xfe, 0xac, 0x56, 0x9f, 0x3e, 0x99, 0x42,
    0xc0, 0x45, 0x0b, 0x9f, 0xcc, 0xa2, 0x2c, 0xb9, 0x50, 0x4b, 0x28, 0x90,
    0x94, 0xc9, 0x6d, 0x5d, 0x1a, 0x7f, 0x7c, 0x3b,
};
static const struct drbg_kat_no_reseed kat102_nor_t = {
    11, kat102_nor_entropyin, kat102_nor_nonce, kat102_nor_persstr,
    kat102_nor_addin0, kat102_nor_addin1, kat102_nor_retbytes
};
static const struct drbg_kat kat102_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat102_nor_t
};

static const unsigned char kat103_nor_entropyin[] = {
    0x88, 0xeb, 0xea, 0x4a, 0xd8, 0x8d, 0x9e, 0x35, 0x18, 0x7c, 0xc7, 0x05,
    0x03, 0xd7, 0x73, 0xf5,
};
static const unsigned char kat103_nor_nonce[] = {
    0x53, 0xad, 0x0f, 0x40, 0xfa, 0xcc, 0x3f, 0x98,
};
static const unsigned char kat103_nor_persstr[] = {
    0x80, 0x95, 0xd8, 0xab, 0x6b, 0x61, 0x7f, 0xa1, 0xd1, 0x15, 0xd1, 0xbf,
    0xe5, 0x17, 0xf4, 0xef,
};
static const unsigned char kat103_nor_addin0[] = {0};
static const unsigned char kat103_nor_addin1[] = {0};
static const unsigned char kat103_nor_retbytes[] = {
    0x25, 0x6c, 0x83, 0x77, 0x81, 0xfa, 0x60, 0xcd, 0x1f, 0x77, 0x5d, 0xb1,
    0xd4, 0xd7, 0xb7, 0x65, 0xad, 0xbb, 0x3b, 0xb5, 0xda, 0xcc, 0x17, 0x16,
    0xd3, 0x9e, 0xfd, 0x71, 0x8f, 0x8e, 0x99, 0x95, 0x66, 0xf0, 0x9c, 0xeb,
    0xc1, 0xd8, 0xd0, 0x38, 0xc6, 0x3a, 0x31, 0x92, 0xd6, 0xab, 0xab, 0xf5,
    0xb3, 0xeb, 0x38, 0xf8, 0x79, 0x56, 0x62, 0xbe, 0x59, 0x4e, 0x63, 0x58,
    0x9d, 0x7a, 0xdc, 0xe5, 0x78, 0x84, 0xab, 0x62, 0xdc, 0x40, 0x99, 0xc4,
    0x9c, 0x1d, 0x25, 0xe8, 0x8a, 0x99, 0x28, 0x3f,
};
static const struct drbg_kat_no_reseed kat103_nor_t = {
    12, kat103_nor_entropyin, kat103_nor_nonce, kat103_nor_persstr,
    kat103_nor_addin0, kat103_nor_addin1, kat103_nor_retbytes
};
static const struct drbg_kat kat103_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat103_nor_t
};

static const unsigned char kat104_nor_entropyin[] = {
    0x7e, 0x9d, 0x7f, 0x2d, 0x74, 0x9f, 0xd4, 0x46, 0x74, 0x71, 0x4d, 0x0f,
    0x6b, 0x09, 0x8b, 0x47,
};
static const unsigned char kat104_nor_nonce[] = {
    0x29, 0x70, 0xb7, 0x16, 0xb2, 0x11, 0x68, 0x95,
};
static const unsigned char kat104_nor_persstr[] = {
    0x57, 0xac, 0xd2, 0x52, 0x0d, 0x9e, 0xbf, 0xb7, 0x62, 0xb7, 0x59, 0xb9,
    0x98, 0xfe, 0x5f, 0x36,
};
static const unsigned char kat104_nor_addin0[] = {0};
static const unsigned char kat104_nor_addin1[] = {0};
static const unsigned char kat104_nor_retbytes[] = {
    0xea, 0x84, 0x63, 0x03, 0xd9, 0xe1, 0xac, 0x84, 0x69, 0xfe, 0xd0, 0x16,
    0x3d, 0xeb, 0x49, 0xc6, 0xd6, 0xf6, 0x9d, 0xa2, 0x44, 0x3d, 0xd3, 0x1b,
    0x66, 0xc7, 0x28, 0x55, 0xac, 0x30, 0xe7, 0xc0, 0x2e, 0x06, 0x55, 0x8d,
    0x78, 0xe7, 0x33, 0xd9, 0xf5, 0xa0, 0xf6, 0xe5, 0x27, 0x63, 0x63, 0x9c,
    0xc2, 0xb8, 0xc4, 0xc0, 0xe0, 0x5b, 0x8c, 0x69, 0xf6, 0xa3, 0xdd, 0x84,
    0x06, 0x39, 0x97, 0x66, 0x3e, 0xd5, 0xf4, 0xe6, 0x90, 0xdd, 0xbb, 0x86,
    0x42, 0x38, 0x0f, 0x8e, 0x8c, 0x9a, 0xa8, 0xa2,
};
static const struct drbg_kat_no_reseed kat104_nor_t = {
    13, kat104_nor_entropyin, kat104_nor_nonce, kat104_nor_persstr,
    kat104_nor_addin0, kat104_nor_addin1, kat104_nor_retbytes
};
static const struct drbg_kat kat104_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat104_nor_t
};

static const unsigned char kat105_nor_entropyin[] = {
    0xd2, 0x0b, 0x5e, 0xd7, 0x01, 0x18, 0xec, 0xfe, 0xb2, 0xdc, 0xfe, 0x8c,
    0xfd, 0x7e, 0x6b, 0x6e,
};
static const unsigned char kat105_nor_nonce[] = {
    0x83, 0x68, 0xee, 0x0e, 0x29, 0xd3, 0x5c, 0x67,
};
static const unsigned char kat105_nor_persstr[] = {
    0xf1, 0x89, 0xa8, 0x0d, 0x56, 0x19, 0xf5, 0x3c, 0xce, 0x87, 0x8e, 0xd5,
    0x75, 0x22, 0xa4, 0x68,
};
static const unsigned char kat105_nor_addin0[] = {0};
static const unsigned char kat105_nor_addin1[] = {0};
static const unsigned char kat105_nor_retbytes[] = {
    0xae, 0xac, 0x59, 0x33, 0x06, 0x5c, 0x33, 0xce, 0x2a, 0xce, 0x25, 0x31,
    0xa1, 0x93, 0xe3, 0x67, 0xf7, 0x3c, 0x83, 0xfc, 0x32, 0x8f, 0x61, 0xee,
    0x26, 0x27, 0xf6, 0xf3, 0x84, 0x19, 0x14, 0xc6, 0xb8, 0xa3, 0xff, 0x76,
    0x7f, 0x96, 0xb3, 0xc3, 0xb6, 0x85, 0xba, 0xc9, 0x31, 0xaf, 0x9e, 0xc1,
    0x0c, 0x6f, 0x3e, 0xfe, 0x25, 0xb5, 0x10, 0x9b, 0xb6, 0x47, 0xb1, 0x20,
    0xe3, 0xa3, 0xf6, 0x97, 0x1a, 0x4e, 0xc4, 0x1f, 0x4e, 0xf0, 0xc7, 0xa9,
    0x00, 0xfd, 0xb0, 0x9d, 0x7f, 0xf3, 0xb2, 0x47,
};
static const struct drbg_kat_no_reseed kat105_nor_t = {
    14, kat105_nor_entropyin, kat105_nor_nonce, kat105_nor_persstr,
    kat105_nor_addin0, kat105_nor_addin1, kat105_nor_retbytes
};
static const struct drbg_kat kat105_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat105_nor_t
};

static const unsigned char kat106_nor_entropyin[] = {
    0xef, 0x3e, 0xb1, 0xe6, 0x3f, 0x98, 0x78, 0xd6, 0x29, 0x84, 0x87, 0x67,
    0x12, 0xe7, 0x6e, 0x56,
};
static const unsigned char kat106_nor_nonce[] = {
    0xf9, 0x59, 0xd3, 0x16, 0x50, 0x36, 0x4e, 0x70,
};
static const unsigned char kat106_nor_persstr[] = {
    0xfc, 0xfe, 0x42, 0xc0, 0xc6, 0x5b, 0xbd, 0xc4, 0xa2, 0x9e, 0xb0, 0xb5,
    0x98, 0x35, 0x3e, 0x17,
};
static const unsigned char kat106_nor_addin0[] = {
    0xd2, 0x18, 0xda, 0x50, 0x1f, 0xa0, 0x9b, 0x43, 0xed, 0x8e, 0x2f, 0xb8,
    0xc3, 0xec, 0xc5, 0x64,
};
static const unsigned char kat106_nor_addin1[] = {
    0x24, 0x79, 0xf4, 0x33, 0x0a, 0xf9, 0x91, 0x0d, 0x74, 0x08, 0x40, 0x46,
    0x90, 0xd8, 0xeb, 0x93,
};
static const unsigned char kat106_nor_retbytes[] = {
    0x5d, 0xe8, 0x1d, 0x92, 0xcd, 0xdd, 0xb3, 0xec, 0x23, 0x60, 0x37, 0xf4,
    0xdc, 0x80, 0xd6, 0xc7, 0x78, 0xca, 0x72, 0xe1, 0x28, 0x41, 0x1a, 0x64,
    0x68, 0x9f, 0x26, 0xa2, 0x43, 0xfb, 0x2b, 0x6f, 0xc5, 0x44, 0x11, 0xee,
    0x9f, 0x41, 0x7b, 0x1e, 0x11, 0x37, 0x60, 0x13, 0x9a, 0xe4, 0xb2, 0xf8,
    0x0d, 0x74, 0x75, 0xb6, 0x66, 0xaf, 0x16, 0xf8, 0x7b, 0xad, 0xc5, 0x8e,
    0x53, 0x04, 0x6e, 0xe6, 0x05, 0xae, 0x96, 0x06, 0xba, 0x97, 0x43, 0xc5,
    0xe5, 0x7a, 0x69, 0x59, 0xbe, 0xbe, 0xb3, 0xa8,
};
static const struct drbg_kat_no_reseed kat106_nor_t = {
    0, kat106_nor_entropyin, kat106_nor_nonce, kat106_nor_persstr,
    kat106_nor_addin0, kat106_nor_addin1, kat106_nor_retbytes
};
static const struct drbg_kat kat106_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat106_nor_t
};

static const unsigned char kat107_nor_entropyin[] = {
    0x74, 0x80, 0xaf, 0xff, 0x27, 0x4e, 0x31, 0x88, 0xc6, 0xc6, 0x86, 0x28,
    0xfc, 0x31, 0x5c, 0x0b,
};
static const unsigned char kat107_nor_nonce[] = {
    0xfd, 0x5e, 0xbb, 0xed, 0xaa, 0xdb, 0x6a, 0xe4,
};
static const unsigned char kat107_nor_persstr[] = {
    0x1c, 0xab, 0x57, 0x35, 0xb2, 0x49, 0xe5, 0x50, 0xff, 0x36, 0x09, 0xef,
    0xe8, 0x9d, 0xbc, 0x9e,
};
static const unsigned char kat107_nor_addin0[] = {
    0x83, 0x4e, 0xa3, 0xf1, 0x5f, 0x77, 0x3e, 0x6d, 0x9c, 0x27, 0xfa, 0x78,
    0x7a, 0xa6, 0x58, 0x86,
};
static const unsigned char kat107_nor_addin1[] = {
    0x85, 0xcb, 0x35, 0xd2, 0xe7, 0xcc, 0xef, 0x7d, 0x22, 0xe1, 0x63, 0x85,
    0x54, 0x22, 0x89, 0x12,
};
static const unsigned char kat107_nor_retbytes[] = {
    0xf8, 0xb0, 0x50, 0x22, 0x4e, 0xe8, 0x90, 0xde, 0xfa, 0xf9, 0xaa, 0x7d,
    0xc5, 0x0b, 0x0e, 0x3a, 0xda, 0xcc, 0x51, 0xf5, 0xd2, 0xb5, 0x4f, 0xbd,
    0x1c, 0xf9, 0x95, 0x25, 0x68, 0x18, 0x76, 0xd2, 0x1e, 0x5c, 0x45, 0x27,
    0x1f, 0x8a, 0x8e, 0x9e, 0xf8, 0x98, 0x70, 0xe5, 0x21, 0x42, 0xa7, 0xf9,
    0x90, 0xaf, 0xd4, 0x86, 0x89, 0x59, 0x12, 0xc3, 0xd4, 0x56, 0xdb, 0xc1,
    0x3e, 0x89, 0xb2, 0x86, 0x38, 0xe9, 0xa3, 0x13, 0x0e, 0x80, 0x8d, 0xdb,
    0xae, 0x96, 0xde, 0x74, 0xb3, 0xdc, 0x16, 0x17,
};
static const struct drbg_kat_no_reseed kat107_nor_t = {
    1, kat107_nor_entropyin, kat107_nor_nonce, kat107_nor_persstr,
    kat107_nor_addin0, kat107_nor_addin1, kat107_nor_retbytes
};
static const struct drbg_kat kat107_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat107_nor_t
};

static const unsigned char kat108_nor_entropyin[] = {
    0x72, 0x66, 0xe7, 0xe0, 0xa7, 0xad, 0x02, 0x52, 0xd2, 0x7b, 0x82, 0xa0,
    0x28, 0x98, 0x58, 0xd4,
};
static const unsigned char kat108_nor_nonce[] = {
    0xb0, 0x2a, 0xa9, 0x20, 0x33, 0xb4, 0x04, 0x5b,
};
static const unsigned char kat108_nor_persstr[] = {
    0xd0, 0xd2, 0x24, 0x78, 0xc0, 0x69, 0xde, 0xda, 0x19, 0x7e, 0x73, 0x51,
    0xed, 0xfd, 0x6a, 0x0c,
};
static const unsigned char kat108_nor_addin0[] = {
    0x36, 0x99, 0x12, 0x64, 0xcb, 0xc2, 0x17, 0x1d, 0x2e, 0x3f, 0xc9, 0x56,
    0xc2, 0xd8, 0x1f, 0x57,
};
static const unsigned char kat108_nor_addin1[] = {
    0xe4, 0xe2, 0xe9, 0x76, 0xbf, 0x83, 0x36, 0x95, 0xe0, 0xbb, 0xae, 0xa1,
    0xd9, 0x2b, 0xed, 0x7b,
};
static const unsigned char kat108_nor_retbytes[] = {
    0x06, 0xc9, 0xff, 0x2a, 0x14, 0x8a, 0x9b, 0x5f, 0x08, 0x0f, 0xd0, 0xfd,
    0x77, 0x42, 0xde, 0xe9, 0x1e, 0x5f, 0x1b, 0x4b, 0xff, 0x6f, 0xc4, 0xad,
    0x27, 0x46, 0x38, 0xac, 0xa0, 0x7e, 0x9e, 0xeb, 0xc2, 0x2d, 0x48, 0x47,
    0x33, 0xc8, 0x27, 0xb6, 0x1f, 0x47, 0xc6, 0xd0, 0xc4, 0x4b, 0x33, 0xdd,
    0xbb, 0xec, 0x1f, 0x2a, 0xb3, 0xad, 0x00, 0x42, 0xdd, 0x50, 0x1a, 0xee,
    0xe2, 0xc9, 0x19, 0xaf, 0x26, 0x63, 0x51, 0xe2, 0xf0, 0x70, 0xce, 0xe6,
    0xe7, 0x28, 0x8c, 0x30, 0x6b, 0x8f, 0x84, 0xd3,
};
static const struct drbg_kat_no_reseed kat108_nor_t = {
    2, kat108_nor_entropyin, kat108_nor_nonce, kat108_nor_persstr,
    kat108_nor_addin0, kat108_nor_addin1, kat108_nor_retbytes
};
static const struct drbg_kat kat108_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat108_nor_t
};

static const unsigned char kat109_nor_entropyin[] = {
    0x7c, 0x6c, 0x0d, 0x59, 0x1b, 0x81, 0xbd, 0x60, 0x16, 0x02, 0xf1, 0xf3,
    0xbe, 0x60, 0xa0, 0x53,
};
static const unsigned char kat109_nor_nonce[] = {
    0x84, 0xe2, 0xd7, 0xb9, 0xd8, 0x0a, 0xf9, 0xe2,
};
static const unsigned char kat109_nor_persstr[] = {
    0xe9, 0x6d, 0x18, 0x31, 0x21, 0x23, 0x9b, 0xa3, 0x50, 0x56, 0x26, 0x4e,
    0x73, 0xed, 0xbe, 0x79,
};
static const unsigned char kat109_nor_addin0[] = {
    0x67, 0x3f, 0x40, 0x9a, 0x0e, 0x13, 0x3c, 0x6f, 0x37, 0xf2, 0xf3, 0xc3,
    0xa3, 0x0e, 0xf0, 0xdc,
};
static const unsigned char kat109_nor_addin1[] = {
    0xc2, 0x68, 0x10, 0x12, 0x11, 0x61, 0xf3, 0xe0, 0x8c, 0x85, 0x4c, 0x29,
    0x9a, 0x3c, 0x5b, 0x45,
};
static const unsigned char kat109_nor_retbytes[] = {
    0x7b, 0x1c, 0x78, 0x4c, 0xce, 0x23, 0x8d, 0xa7, 0x65, 0x18, 0x4c, 0x6f,
    0xc0, 0x3f, 0x2b, 0xf5, 0x82, 0xd2, 0x65, 0x2c, 0x5b, 0x76, 0x3d, 0x42,
    0x5b, 0x5b, 0xe9, 0x32, 0xbd, 0x3d, 0x75, 0x65, 0xaa, 0xd0, 0x1e, 0xb9,
    0x02, 0x13, 0xd0, 0x0f, 0xd1, 0xd3, 0x8c, 0x59, 0xdf, 0xb3, 0xd5, 0xcd,
    0x92, 0xcf, 0x8a, 0x0b, 0x67, 0x6d, 0xe1, 0xc1, 0x13, 0xd7, 0xc5, 0x67,
    0xac, 0x92, 0xb8, 0x52, 0x85, 0x12, 0x10, 0xb6, 0x20, 0x7e, 0x48, 0x4f,
    0xdf, 0x4a, 0x3a, 0x62, 0x44, 0xa2, 0xff, 0xae,
};
static const struct drbg_kat_no_reseed kat109_nor_t = {
    3, kat109_nor_entropyin, kat109_nor_nonce, kat109_nor_persstr,
    kat109_nor_addin0, kat109_nor_addin1, kat109_nor_retbytes
};
static const struct drbg_kat kat109_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat109_nor_t
};

static const unsigned char kat110_nor_entropyin[] = {
    0x5f, 0x6a, 0xd5, 0xae, 0xfc, 0xc3, 0x9b, 0x33, 0x19, 0xaa, 0x7e, 0xda,
    0xbd, 0x5c, 0x3b, 0x5b,
};
static const unsigned char kat110_nor_nonce[] = {
    0xdc, 0xbe, 0xe7, 0x5d, 0xb3, 0x07, 0xb6, 0xf5,
};
static const unsigned char kat110_nor_persstr[] = {
    0x6b, 0xc3, 0xfb, 0x02, 0x3c, 0x6f, 0xd9, 0xce, 0xb6, 0x5e, 0xac, 0xce,
    0xaa, 0x5c, 0x4e, 0x76,
};
static const unsigned char kat110_nor_addin0[] = {
    0x8e, 0x9b, 0xa1, 0xa1, 0xf6, 0xa7, 0x31, 0x5b, 0x7b, 0x5c, 0x55, 0x7a,
    0xb0, 0x0c, 0x68, 0xd7,
};
static const unsigned char kat110_nor_addin1[] = {
    0xe5, 0x4b, 0xfb, 0xf4, 0x41, 0x51, 0x04, 0xe4, 0x76, 0xcc, 0x9f, 0x37,
    0xc5, 0x0e, 0xde, 0xa7,
};
static const unsigned char kat110_nor_retbytes[] = {
    0x33, 0x71, 0x86, 0x66, 0x66, 0xc2, 0xca, 0xbf, 0x02, 0x8f, 0x88, 0x8a,
    0xb1, 0x2b, 0x6a, 0xec, 0xe4, 0x3a, 0x7b, 0x20, 0x21, 0xd8, 0x09, 0x8e,
    0xcc, 0x42, 0x3c, 0x3e, 0x72, 0x60, 0x0b, 0xf4, 0x9a, 0xed, 0xc7, 0x1d,
    0x24, 0xda, 0x4c, 0x7d, 0x99, 0x2a, 0x7c, 0xe1, 0x58, 0x96, 0xf7, 0xae,
    0x11, 0xb3, 0x24, 0xfe, 0xc9, 0x3d, 0x76, 0x15, 0x0b, 0x13, 0x89, 0x2b,
    0x00, 0xe9, 0x11, 0x02, 0x6a, 0xc8, 0xe3, 0x54, 0x3b, 0xbb, 0xd0, 0xca,
    0xf1, 0x99, 0x6d, 0x9e, 0x07, 0xb0, 0x49, 0xb4,
};
static const struct drbg_kat_no_reseed kat110_nor_t = {
    4, kat110_nor_entropyin, kat110_nor_nonce, kat110_nor_persstr,
    kat110_nor_addin0, kat110_nor_addin1, kat110_nor_retbytes
};
static const struct drbg_kat kat110_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat110_nor_t
};

static const unsigned char kat111_nor_entropyin[] = {
    0x7f, 0x26, 0x7c, 0x4c, 0xb0, 0x7d, 0x85, 0xba, 0xc0, 0x61, 0x4d, 0xdd,
    0xd4, 0xd8, 0x9d, 0x80,
};
static const unsigned char kat111_nor_nonce[] = {
    0xd8, 0x19, 0xac, 0x8c, 0x6b, 0x43, 0xfd, 0x61,
};
static const unsigned char kat111_nor_persstr[] = {
    0x61, 0x7d, 0x25, 0x87, 0x40, 0xd8, 0xab, 0x7c, 0xb2, 0xc6, 0x32, 0xb6,
    0x37, 0x2e, 0xf3, 0xae,
};
static const unsigned char kat111_nor_addin0[] = {
    0x3a, 0x29, 0x9f, 0xd0, 0x7e, 0x86, 0x11, 0xe1, 0xc3, 0xbb, 0xff, 0xac,
    0x00, 0x1a, 0xdb, 0x2c,
};
static const unsigned char kat111_nor_addin1[] = {
    0x94, 0xf5, 0xbf, 0xb5, 0x3e, 0x32, 0x03, 0xb2, 0x12, 0xbf, 0xfd, 0x88,
    0x69, 0x5c, 0xfb, 0x58,
};
static const unsigned char kat111_nor_retbytes[] = {
    0x46, 0x8b, 0xed, 0xcb, 0xb2, 0xf9, 0xc6, 0xde, 0xb8, 0x14, 0x18, 0x88,
    0xa7, 0x58, 0xbd, 0x79, 0x31, 0xef, 0x12, 0x8d, 0xe4, 0xe6, 0x78, 0xbe,
    0x99, 0x85, 0x29, 0x8c, 0x26, 0x64, 0xd3, 0x39, 0x99, 0xd6, 0x83, 0x42,
    0xa0, 0x74, 0x7a, 0xcf, 0x18, 0xdf, 0xfc, 0xab, 0xc3, 0x2a, 0x0c, 0x09,
    0xce, 0x7c, 0x65, 0x22, 0x67, 0x3b, 0x67, 0x39, 0xef, 0xe6, 0x6c, 0xc2,
    0x5b, 0x28, 0xc0, 0x6e, 0x86, 0x77, 0xab, 0x3c, 0x6f, 0x1e, 0x38, 0x40,
    0x85, 0x2d, 0x24, 0x5e, 0x05, 0x53, 0xca, 0x47,
};
static const struct drbg_kat_no_reseed kat111_nor_t = {
    5, kat111_nor_entropyin, kat111_nor_nonce, kat111_nor_persstr,
    kat111_nor_addin0, kat111_nor_addin1, kat111_nor_retbytes
};
static const struct drbg_kat kat111_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat111_nor_t
};

static const unsigned char kat112_nor_entropyin[] = {
    0x85, 0x89, 0x23, 0xf0, 0x5c, 0x1a, 0x1a, 0x93, 0x70, 0xc4, 0x95, 0xfc,
    0x84, 0x02, 0x58, 0x9d,
};
static const unsigned char kat112_nor_nonce[] = {
    0xa8, 0x8f, 0x68, 0x8f, 0x22, 0xe8, 0xbd, 0x53,
};
static const unsigned char kat112_nor_persstr[] = {
    0x4c, 0xc6, 0x8f, 0x42, 0x20, 0x6a, 0xfd, 0x82, 0x60, 0x8d, 0xdf, 0x9f,
    0x18, 0x50, 0x73, 0x98,
};
static const unsigned char kat112_nor_addin0[] = {
    0x46, 0x70, 0x63, 0x1a, 0x5d, 0x2f, 0xbd, 0x86, 0xf5, 0x84, 0x87, 0x60,
    0x3b, 0x88, 0x00, 0x65,
};
static const unsigned char kat112_nor_addin1[] = {
    0x34, 0xf7, 0xb4, 0xc9, 0x2b, 0xa1, 0xaa, 0x90, 0x0f, 0x7b, 0xc6, 0x76,
    0xc2, 0x8f, 0x42, 0x83,
};
static const unsigned char kat112_nor_retbytes[] = {
    0xa3, 0x91, 0xbb, 0x3c, 0x2f, 0x2c, 0xbd, 0xe7, 0xda, 0xbd, 0x18, 0xa7,
    0x2b, 0x4e, 0x29, 0xb3, 0xb5, 0xc1, 0x1d, 0x07, 0xc5, 0xc7, 0x7a, 0x6b,
    0x34, 0xe9, 0x30, 0xf2, 0x80, 0x4d, 0x3a, 0xbd, 0x5a, 0xf4, 0x5f, 0x41,
    0x02, 0xad, 0x1e, 0x41, 0xda, 0x51, 0x25, 0x72, 0xed, 0x94, 0xd9, 0xd3,
    0x31, 0x7a, 0xc9, 0xc7, 0x6b, 0xd5, 0x6a, 0x0f, 0x10, 0xff, 0x10, 0xca,
    0x7a, 0x8b, 0xeb, 0x59, 0x2d, 0xc3, 0xc8, 0xa0, 0x17, 0x03, 0x0b, 0x0b,
    0x19, 0x0a, 0x7a, 0xbd, 0x50, 0x6e, 0xa5, 0x9b,
};
static const struct drbg_kat_no_reseed kat112_nor_t = {
    6, kat112_nor_entropyin, kat112_nor_nonce, kat112_nor_persstr,
    kat112_nor_addin0, kat112_nor_addin1, kat112_nor_retbytes
};
static const struct drbg_kat kat112_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat112_nor_t
};

static const unsigned char kat113_nor_entropyin[] = {
    0x31, 0x7d, 0xdf, 0xb1, 0xa0, 0x1b, 0xb0, 0xe1, 0x47, 0xea, 0xae, 0xd3,
    0x75, 0x99, 0x97, 0x30,
};
static const unsigned char kat113_nor_nonce[] = {
    0xd7, 0xc5, 0xa2, 0x7e, 0xbd, 0x92, 0x09, 0xee,
};
static const unsigned char kat113_nor_persstr[] = {
    0x8d, 0x53, 0x5a, 0x8d, 0x76, 0x04, 0x22, 0x33, 0xe4, 0xb2, 0xbc, 0xf6,
    0xae, 0x3a, 0x71, 0x42,
};
static const unsigned char kat113_nor_addin0[] = {
    0xb9, 0x86, 0x56, 0x84, 0x60, 0xca, 0xf3, 0x6f, 0x3e, 0x83, 0xcd, 0x27,
    0xd7, 0x9f, 0xcb, 0x69,
};
static const unsigned char kat113_nor_addin1[] = {
    0x1a, 0x4b, 0xe6, 0x76, 0x6a, 0x58, 0x99, 0xa2, 0x1b, 0xea, 0x32, 0x87,
    0xfa, 0x21, 0xcd, 0x88,
};
static const unsigned char kat113_nor_retbytes[] = {
    0x3e, 0x1c, 0x68, 0x8b, 0xb6, 0xb6, 0x80, 0x50, 0x20, 0x61, 0x51, 0x4b,
    0x59, 0xae, 0x90, 0x80, 0x9d, 0xfc, 0x72, 0x18, 0x2f, 0x54, 0xf0, 0xe8,
    0x37, 0x7f, 0xf5, 0x0d, 0xb2, 0xa8, 0x60, 0x82, 0xf9, 0x9e, 0x9f, 0x4c,
    0x9e, 0xac, 0x9e, 0x3c, 0x5f, 0x06, 0xee, 0xc4, 0x9d, 0x9e, 0x51, 0x43,
    0x40, 0x28, 0xbe, 0x4e, 0x08, 0xf7, 0x06, 0x34, 0xb1, 0x33, 0x9d, 0xb7,
    0x03, 0xbf, 0xc8, 0x48, 0xeb, 0xc6, 0x60, 0xa0, 0x6a, 0xb8, 0x44, 0x04,
    0x09, 0xed, 0x1e, 0xf4, 0x17, 0x55, 0x84, 0xcc,
};
static const struct drbg_kat_no_reseed kat113_nor_t = {
    7, kat113_nor_entropyin, kat113_nor_nonce, kat113_nor_persstr,
    kat113_nor_addin0, kat113_nor_addin1, kat113_nor_retbytes
};
static const struct drbg_kat kat113_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat113_nor_t
};

static const unsigned char kat114_nor_entropyin[] = {
    0xe3, 0xdd, 0x2e, 0x76, 0x7f, 0x52, 0x05, 0xb6, 0xd0, 0xc4, 0x90, 0x8d,
    0x46, 0x5c, 0xd2, 0xa0,
};
static const unsigned char kat114_nor_nonce[] = {
    0x4e, 0x35, 0x6c, 0x17, 0x17, 0x09, 0xce, 0xfe,
};
static const unsigned char kat114_nor_persstr[] = {
    0xeb, 0x84, 0xe2, 0xe5, 0xb2, 0x20, 0x88, 0x82, 0xd3, 0x72, 0xf2, 0x66,
    0xbd, 0x1b, 0x5b, 0xa1,
};
static const unsigned char kat114_nor_addin0[] = {
    0x83, 0x18, 0x68, 0xd3, 0x0c, 0x12, 0xad, 0x5a, 0x81, 0xe3, 0x9f, 0x74,
    0xed, 0xa9, 0xf8, 0x7e,
};
static const unsigned char kat114_nor_addin1[] = {
    0x0e, 0x34, 0x96, 0x0f, 0x94, 0x69, 0xe2, 0x3e, 0xd4, 0xfa, 0x28, 0x5e,
    0xdd, 0x6b, 0x83, 0x71,
};
static const unsigned char kat114_nor_retbytes[] = {
    0xe4, 0x7b, 0x96, 0xa3, 0x06, 0xa4, 0x07, 0x9a, 0x1a, 0x79, 0xe7, 0xe8,
    0xe9, 0x3d, 0x8a, 0x5e, 0x2f, 0x96, 0xfd, 0x99, 0x5a, 0x02, 0xbe, 0x29,
    0xc4, 0xc3, 0x6f, 0x98, 0x0f, 0x23, 0x42, 0xaa, 0x94, 0x99, 0x67, 0xa3,
    0xa4, 0xf3, 0x12, 0xa7, 0x02, 0x08, 0x1e, 0xf7, 0xbd, 0x95, 0x91, 0x25,
    0x28, 0x74, 0x11, 0xf9, 0x6c, 0xe6, 0x47, 0xb9, 0x1b, 0x56, 0xee, 0x93,
    0x2e, 0x55, 0x54, 0xd4, 0x55, 0x70, 0x3f, 0xaa, 0x6a, 0xbe, 0x41, 0x09,
    0x43, 0x63, 0x35, 0xc4, 0xd2, 0x31, 0x2b, 0x77,
};
static const struct drbg_kat_no_reseed kat114_nor_t = {
    8, kat114_nor_entropyin, kat114_nor_nonce, kat114_nor_persstr,
    kat114_nor_addin0, kat114_nor_addin1, kat114_nor_retbytes
};
static const struct drbg_kat kat114_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat114_nor_t
};

static const unsigned char kat115_nor_entropyin[] = {
    0xcb, 0xff, 0x42, 0xd1, 0x36, 0x1f, 0xa6, 0xcf, 0x28, 0x3e, 0xa3, 0x0f,
    0x49, 0xf8, 0x5a, 0xbd,
};
static const unsigned char kat115_nor_nonce[] = {
    0x82, 0x41, 0x8a, 0x77, 0x8a, 0xea, 0xfc, 0xcc,
};
static const unsigned char kat115_nor_persstr[] = {
    0x93, 0xde, 0x97, 0x31, 0xc0, 0xc7, 0x18, 0x07, 0x98, 0x46, 0x81, 0x69,
    0x9a, 0x78, 0x27, 0x98,
};
static const unsigned char kat115_nor_addin0[] = {
    0x7d, 0x11, 0xeb, 0xed, 0xcc, 0x74, 0xe2, 0x49, 0xb2, 0x7a, 0xb3, 0x09,
    0x87, 0x53, 0x97, 0x92,
};
static const unsigned char kat115_nor_addin1[] = {
    0xdd, 0x2c, 0x36, 0x32, 0xf5, 0x73, 0xeb, 0xdb, 0xbd, 0xb5, 0x62, 0x2c,
    0x04, 0xb3, 0x1d, 0xdc,
};
static const unsigned char kat115_nor_retbytes[] = {
    0x12, 0xaf, 0xcb, 0x02, 0x3c, 0x29, 0x79, 0x5c, 0x6b, 0x3d, 0xee, 0xb6,
    0x55, 0x8d, 0x59, 0xc7, 0xa0, 0x73, 0x9d, 0x49, 0x9c, 0x41, 0x35, 0xc3,
    0x7f, 0xf3, 0xdc, 0xd0, 0x02, 0x2c, 0x75, 0x15, 0xb1, 0xf8, 0x77, 0x7c,
    0x1f, 0x2e, 0xbb, 0x49, 0x32, 0x2f, 0x5d, 0xfd, 0xf9, 0xea, 0x14, 0x15,
    0x1d, 0x20, 0x86, 0x9b, 0x5c, 0xa1, 0x45, 0xfe, 0xec, 0x0d, 0xcf, 0xf8,
    0x42, 0x93, 0xed, 0x31, 0x55, 0x77, 0x44, 0xcc, 0x74, 0xea, 0xc3, 0xb1,
    0x5c, 0xf4, 0x50, 0xd1, 0x9d, 0x11, 0x36, 0xf8,
};
static const struct drbg_kat_no_reseed kat115_nor_t = {
    9, kat115_nor_entropyin, kat115_nor_nonce, kat115_nor_persstr,
    kat115_nor_addin0, kat115_nor_addin1, kat115_nor_retbytes
};
static const struct drbg_kat kat115_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat115_nor_t
};

static const unsigned char kat116_nor_entropyin[] = {
    0x9c, 0x50, 0x1a, 0x22, 0x07, 0xd4, 0x16, 0xc0, 0x84, 0xf8, 0xc6, 0xba,
    0x02, 0x2a, 0x7a, 0x35,
};
static const unsigned char kat116_nor_nonce[] = {
    0xcb, 0x44, 0x76, 0x7d, 0xe7, 0x04, 0x29, 0x09,
};
static const unsigned char kat116_nor_persstr[] = {
    0xf3, 0x23, 0x2c, 0xa9, 0xde, 0xf8, 0x3f, 0xbb, 0x54, 0x5e, 0x2c, 0xba,
    0xf8, 0x0b, 0xc8, 0x06,
};
static const unsigned char kat116_nor_addin0[] = {
    0xf4, 0xf6, 0x76, 0x89, 0x06, 0x34, 0xe1, 0x31, 0xd0, 0x37, 0x50, 0xe9,
    0x3b, 0x25, 0xfa, 0xe8,
};
static const unsigned char kat116_nor_addin1[] = {
    0x48, 0xbb, 0xaa, 0xd1, 0x91, 0xbd, 0xc7, 0x77, 0x93, 0x74, 0x44, 0xbd,
    0x23, 0x34, 0xda, 0xcf,
};
static const unsigned char kat116_nor_retbytes[] = {
    0x20, 0x02, 0x17, 0xf9, 0x15, 0x1c, 0x70, 0x28, 0x40, 0xd9, 0xfc, 0xd1,
    0x3d, 0xc3, 0x0b, 0x99, 0x3f, 0x1d, 0xda, 0xb2, 0x29, 0x17, 0x91, 0xa9,
    0x2e, 0xe5, 0x62, 0xe5, 0x2a, 0xbc, 0xd5, 0x35, 0x8a, 0xff, 0x1a, 0xb2,
    0x4e, 0x63, 0xf7, 0xdd, 0x1a, 0xf2, 0x70, 0xfd, 0x1e, 0x3e, 0x58, 0xc4,
    0xd4, 0x00, 0x25, 0x22, 0x53, 0xa8, 0x0d, 0x7e, 0xb0, 0x50, 0x1e, 0x4a,
    0x41, 0x13, 0xc0, 0x6d, 0x94, 0x5b, 0xce, 0x64, 0x0b, 0x77, 0x5a, 0xd6,
    0x57, 0xba, 0xd1, 0x05, 0xaf, 0x60, 0x28, 0x0d,
};
static const struct drbg_kat_no_reseed kat116_nor_t = {
    10, kat116_nor_entropyin, kat116_nor_nonce, kat116_nor_persstr,
    kat116_nor_addin0, kat116_nor_addin1, kat116_nor_retbytes
};
static const struct drbg_kat kat116_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat116_nor_t
};

static const unsigned char kat117_nor_entropyin[] = {
    0xa8, 0xc6, 0x88, 0x26, 0xec, 0x82, 0x54, 0xbe, 0xf2, 0x45, 0x17, 0xad,
    0xc8, 0x96, 0x2f, 0x0b,
};
static const unsigned char kat117_nor_nonce[] = {
    0xbd, 0x51, 0x4b, 0xfa, 0x52, 0xbc, 0xf8, 0x49,
};
static const unsigned char kat117_nor_persstr[] = {
    0x00, 0x07, 0x2c, 0x1d, 0x2e, 0x76, 0x70, 0xbd, 0x38, 0xf7, 0xe4, 0x09,
    0x3d, 0x89, 0x4b, 0x51,
};
static const unsigned char kat117_nor_addin0[] = {
    0x35, 0x1a, 0xf5, 0x5b, 0x61, 0x24, 0x9d, 0xba, 0xe6, 0x0a, 0xde, 0xe7,
    0x15, 0x6c, 0xfe, 0x40,
};
static const unsigned char kat117_nor_addin1[] = {
    0x02, 0xd9, 0x97, 0xaa, 0xe4, 0x80, 0x47, 0xee, 0xad, 0x33, 0x3c, 0x64,
    0x0a, 0x10, 0x95, 0x8d,
};
static const unsigned char kat117_nor_retbytes[] = {
    0x9e, 0x3d, 0xce, 0xa0, 0x56, 0x2a, 0x8c, 0x76, 0xa5, 0x69, 0xab, 0xcf,
    0xc8, 0x05, 0xa1, 0x78, 0x82, 0x5d, 0x8d, 0xe5, 0xd7, 0xeb, 0xcc, 0xaf,
    0x48, 0x74, 0x7a, 0x68, 0xd5, 0xe5, 0x48, 0x07, 0x06, 0xef, 0x6a, 0xcc,
    0x5f, 0xe5, 0xe0, 0xe2, 0x4c, 0xa9, 0x36, 0x2a, 0xb2, 0x3d, 0xf8, 0x4c,
    0xa6, 0x29, 0x87, 0x6b, 0x23, 0xbf, 0x4b, 0xbb, 0xe9, 0xee, 0xfe, 0x58,
    0xc9, 0x36, 0x76, 0xb4, 0x7c, 0x58, 0x2c, 0x1a, 0x71, 0x81, 0xf5, 0x05,
    0x74, 0x95, 0xe8, 0x3d, 0x30, 0xef, 0x98, 0x44,
};
static const struct drbg_kat_no_reseed kat117_nor_t = {
    11, kat117_nor_entropyin, kat117_nor_nonce, kat117_nor_persstr,
    kat117_nor_addin0, kat117_nor_addin1, kat117_nor_retbytes
};
static const struct drbg_kat kat117_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat117_nor_t
};

static const unsigned char kat118_nor_entropyin[] = {
    0x0e, 0x57, 0x9b, 0xc1, 0x1f, 0x69, 0xb1, 0xd1, 0x06, 0xe9, 0x2c, 0x0a,
    0x31, 0x83, 0x3a, 0xac,
};
static const unsigned char kat118_nor_nonce[] = {
    0x85, 0x57, 0xf8, 0x0c, 0x6f, 0x1e, 0xa3, 0x49,
};
static const unsigned char kat118_nor_persstr[] = {
    0xc6, 0xf3, 0x84, 0x8b, 0x26, 0xfd, 0xb4, 0xb4, 0x40, 0x2b, 0x4f, 0x95,
    0x16, 0xe7, 0x11, 0x45,
};
static const unsigned char kat118_nor_addin0[] = {
    0x1a, 0x06, 0x8f, 0x37, 0xe9, 0xc4, 0xa7, 0xa1, 0x55, 0xb8, 0xb7, 0x3e,
    0xb3, 0x1a, 0x72, 0x6f,
};
static const unsigned char kat118_nor_addin1[] = {
    0x8c, 0x35, 0x97, 0xd2, 0x3d, 0xa6, 0xcd, 0x67, 0x2b, 0x6e, 0xc8, 0x73,
    0xaa, 0xc9, 0x50, 0x2f,
};
static const unsigned char kat118_nor_retbytes[] = {
    0xdd, 0x60, 0x5d, 0x5f, 0x8b, 0x19, 0x31, 0x96, 0xad, 0x84, 0x30, 0x72,
    0x76, 0x8f, 0x87, 0x27, 0x59, 0x4c, 0x2c, 0x4e, 0x74, 0x1d, 0x22, 0x0c,
    0xb5, 0xd4, 0xe7, 0xd7, 0x00, 0x1b, 0x93, 0xce, 0xd7, 0x34, 0xf1, 0xa3,
    0xad, 0x26, 0x3b, 0xdb, 0xf2, 0x03, 0xfe, 0x5f, 0xbb, 0x07, 0xe6, 0x2a,
    0xf4, 0x56, 0xf9, 0xef, 0x4b, 0xfd, 0x93, 0xd0, 0xc4, 0xf6, 0xa7, 0x52,
    0x3a, 0xc8, 0x4b, 0x80, 0x88, 0xda, 0x6c, 0x57, 0x86, 0xc5, 0x2c, 0x64,
    0x89, 0x02, 0xb9, 0x72, 0xd0, 0x78, 0x1c, 0x15,
};
static const struct drbg_kat_no_reseed kat118_nor_t = {
    12, kat118_nor_entropyin, kat118_nor_nonce, kat118_nor_persstr,
    kat118_nor_addin0, kat118_nor_addin1, kat118_nor_retbytes
};
static const struct drbg_kat kat118_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat118_nor_t
};

static const unsigned char kat119_nor_entropyin[] = {
    0x06, 0xdd, 0xcf, 0xcd, 0x7c, 0x11, 0xf7, 0xca, 0xd4, 0x9e, 0x83, 0x29,
    0x0c, 0xcf, 0xe3, 0x20,
};
static const unsigned char kat119_nor_nonce[] = {
    0xbd, 0xd6, 0xf8, 0xb1, 0x9d, 0xda, 0xee, 0x6c,
};
static const unsigned char kat119_nor_persstr[] = {
    0xc4, 0x73, 0xd3, 0xe0, 0x77, 0x89, 0x27, 0x66, 0x5f, 0x2e, 0x36, 0x18,
    0x7b, 0xd8, 0xe8, 0xa0,
};
static const unsigned char kat119_nor_addin0[] = {
    0xd6, 0x46, 0x31, 0x60, 0x60, 0x5f, 0x3d, 0x40, 0x34, 0x4a, 0xe8, 0xfe,
    0x8d, 0x5b, 0x7c, 0x5e,
};
static const unsigned char kat119_nor_addin1[] = {
    0xeb, 0xba, 0xb9, 0xe5, 0xe4, 0xd8, 0x42, 0x27, 0xe8, 0xca, 0xbe, 0x8b,
    0xca, 0xac, 0x77, 0x11,
};
static const unsigned char kat119_nor_retbytes[] = {
    0x0e, 0x57, 0x83, 0x6e, 0x98, 0x29, 0x9c, 0x46, 0x80, 0xbd, 0x28, 0xfb,
    0x26, 0xca, 0x84, 0x80, 0x70, 0xad, 0x83, 0xe9, 0x99, 0x1f, 0x7d, 0x93,
    0x4c, 0x31, 0xe0, 0xae, 0x17, 0xf3, 0x2c, 0xd5, 0x91, 0xd4, 0xf2, 0xee,
    0xf1, 0x4b, 0x2e, 0xaf, 0x95, 0xb2, 0x7e, 0x3b, 0x1c, 0x75, 0xe9, 0xa0,
    0x49, 0x6a, 0x6d, 0xec, 0xea, 0xe8, 0xfe, 0xd5, 0x98, 0x2d, 0x03, 0xa0,
    0xba, 0xf8, 0x61, 0xee, 0x93, 0x24, 0xfc, 0x6e, 0x75, 0x09, 0x3c, 0x79,
    0x53, 0x46, 0x19, 0x18, 0xa6, 0x08, 0x45, 0x0b,
};
static const struct drbg_kat_no_reseed kat119_nor_t = {
    13, kat119_nor_entropyin, kat119_nor_nonce, kat119_nor_persstr,
    kat119_nor_addin0, kat119_nor_addin1, kat119_nor_retbytes
};
static const struct drbg_kat kat119_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat119_nor_t
};

static const unsigned char kat120_nor_entropyin[] = {
    0x03, 0xd3, 0x89, 0xe9, 0xa0, 0xfc, 0x11, 0x0d, 0x17, 0x3a, 0xdb, 0x2c,
    0xeb, 0x33, 0x58, 0xba,
};
static const unsigned char kat120_nor_nonce[] = {
    0x8b, 0x32, 0x55, 0x99, 0x29, 0x7c, 0x0d, 0x47,
};
static const unsigned char kat120_nor_persstr[] = {
    0xab, 0x25, 0xbe, 0xf8, 0xd0, 0x09, 0x7b, 0xbd, 0x42, 0xd5, 0x53, 0x16,
    0xaf, 0x32, 0x1b, 0xac,
};
static const unsigned char kat120_nor_addin0[] = {
    0xaf, 0x57, 0x8f, 0xbb, 0xb8, 0xa8, 0x30, 0x94, 0x7e, 0x9b, 0x4e, 0x2c,
    0x9e, 0x72, 0x93, 0x36,
};
static const unsigned char kat120_nor_addin1[] = {
    0x5a, 0x69, 0x86, 0x4c, 0xa3, 0x9d, 0xa1, 0xba, 0x47, 0x19, 0xdf, 0xe1,
    0xdc, 0x85, 0x0a, 0x4a,
};
static const unsigned char kat120_nor_retbytes[] = {
    0x8b, 0x84, 0x6f, 0x03, 0xcb, 0x66, 0xf7, 0xe4, 0x9f, 0xdd, 0xdf, 0x7c,
    0xc4, 0x49, 0xa5, 0xf3, 0xf6, 0xcc, 0xdc, 0x17, 0xae, 0x7e, 0x22, 0x65,
    0xa5, 0xd0, 0xe3, 0x9e, 0xa1, 0x0f, 0xc3, 0xe6, 0xcf, 0xfe, 0xfc, 0x04,
    0x14, 0x7b, 0x77, 0x3a, 0x15, 0x84, 0xe4, 0x29, 0xfe, 0x99, 0xe8, 0x85,
    0xf2, 0x78, 0xaf, 0xf7, 0x4a, 0x49, 0xd8, 0xc8, 0x42, 0xe7, 0xcc, 0xd8,
    0x70, 0xf1, 0x33, 0x06, 0x92, 0xfc, 0x9c, 0x48, 0x36, 0xda, 0xc5, 0x04,
    0x6c, 0x54, 0x4b, 0xe7, 0x46, 0x52, 0xda, 0x26,
};
static const struct drbg_kat_no_reseed kat120_nor_t = {
    14, kat120_nor_entropyin, kat120_nor_nonce, kat120_nor_persstr,
    kat120_nor_addin0, kat120_nor_addin1, kat120_nor_retbytes
};
static const struct drbg_kat kat120_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat120_nor_t
};

static const unsigned char kat121_nor_entropyin[] = {
    0xac, 0x1f, 0x51, 0xd4, 0x3f, 0xaf, 0xea, 0x51, 0x94, 0x53, 0xea, 0xca,
    0x3e, 0xbb, 0x2f, 0x95,
};
static const unsigned char kat121_nor_nonce[] = {
    0xa6, 0x66, 0x6f, 0xe2, 0x6d, 0x87, 0x02, 0xc1,
};
static const unsigned char kat121_nor_persstr[] = {0};
static const unsigned char kat121_nor_addin0[] = {0};
static const unsigned char kat121_nor_addin1[] = {0};
static const unsigned char kat121_nor_retbytes[] = {
    0x3e, 0xa6, 0x9d, 0xff, 0x9d, 0x56, 0x33, 0x69, 0x9d, 0x27, 0xfb, 0x4b,
    0xab, 0x4c, 0xa9, 0x72, 0x53, 0x6d, 0xae, 0xf1, 0x5d, 0xe5, 0x84, 0x42,
    0x73, 0x84, 0xd6, 0x6f, 0x00, 0x31, 0x2c, 0x31, 0xa2, 0x1e, 0xf0, 0x2d,
    0x88, 0xec, 0x9c, 0xa7, 0x95, 0x83, 0xb6, 0x9f, 0xca, 0x77, 0x6e, 0x2f,
    0x67, 0xfc, 0x6a, 0x9a, 0x1c, 0xa2, 0xc5, 0x19, 0x31, 0x7d, 0x25, 0x23,
    0x79, 0x3d, 0xec, 0x48, 0xe1, 0x1c, 0xde, 0xe2, 0x8b, 0x1f, 0xc6, 0x10,
    0x02, 0x4b, 0x03, 0xbc, 0x23, 0x46, 0x19, 0xba,
};
static const struct drbg_kat_no_reseed kat121_nor_t = {
    0, kat121_nor_entropyin, kat121_nor_nonce, kat121_nor_persstr,
    kat121_nor_addin0, kat121_nor_addin1, kat121_nor_retbytes
};
static const struct drbg_kat kat121_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat121_nor_t
};

static const unsigned char kat122_nor_entropyin[] = {
    0xbd, 0x67, 0x68, 0x78, 0x8f, 0x02, 0xe8, 0x4a, 0x2f, 0xc3, 0x58, 0x0a,
    0x87, 0xb4, 0x69, 0xfa,
};
static const unsigned char kat122_nor_nonce[] = {
    0xe2, 0x6b, 0x8c, 0x9c, 0x4b, 0xc6, 0x12, 0x8a,
};
static const unsigned char kat122_nor_persstr[] = {0};
static const unsigned char kat122_nor_addin0[] = {0};
static const unsigned char kat122_nor_addin1[] = {0};
static const unsigned char kat122_nor_retbytes[] = {
    0xd4, 0xe9, 0x00, 0xb0, 0xa6, 0xed, 0x2c, 0x88, 0xca, 0x0b, 0x28, 0xee,
    0x92, 0x12, 0xfa, 0xf0, 0x65, 0x7d, 0x28, 0xd7, 0xa5, 0x3d, 0x39, 0x3f,
    0x04, 0x75, 0xc4, 0x3c, 0xe5, 0xf0, 0xf2, 0x20, 0x7a, 0x3f, 0x18, 0xcd,
    0xea, 0x08, 0xd9, 0x6a, 0x2a, 0x53, 0x36, 0x47, 0xa6, 0x90, 0x05, 0xb6,
    0x75, 0x75, 0x8c, 0x93, 0x18, 0xa8, 0x34, 0x09, 0x79, 0xc4, 0xe1, 0xd0,
    0xbb, 0x46, 0x04, 0x0e, 0x3f, 0x65, 0x6a, 0x48, 0x62, 0xd4, 0xdb, 0xfd,
    0x07, 0x80, 0xca, 0xe9, 0xba, 0xa4, 0xab, 0x5a,
};
static const struct drbg_kat_no_reseed kat122_nor_t = {
    1, kat122_nor_entropyin, kat122_nor_nonce, kat122_nor_persstr,
    kat122_nor_addin0, kat122_nor_addin1, kat122_nor_retbytes
};
static const struct drbg_kat kat122_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat122_nor_t
};

static const unsigned char kat123_nor_entropyin[] = {
    0x51, 0x4c, 0xb0, 0x5a, 0xf1, 0x92, 0xb0, 0xb0, 0x44, 0x35, 0xe2, 0x1a,
    0x13, 0xde, 0x0a, 0x2b,
};
static const unsigned char kat123_nor_nonce[] = {
    0x27, 0x8c, 0x9a, 0x83, 0xac, 0xbd, 0xda, 0xff,
};
static const unsigned char kat123_nor_persstr[] = {0};
static const unsigned char kat123_nor_addin0[] = {0};
static const unsigned char kat123_nor_addin1[] = {0};
static const unsigned char kat123_nor_retbytes[] = {
    0x31, 0xf8, 0x64, 0x41, 0x7a, 0xc2, 0x79, 0x92, 0x78, 0x8b, 0xcd, 0x5e,
    0x07, 0xff, 0xf7, 0x88, 0x6e, 0x96, 0x9d, 0x06, 0xe7, 0x47, 0xfa, 0x5c,
    0x73, 0xa4, 0x8a, 0x19, 0x4a, 0x54, 0xf6, 0x12, 0x32, 0x7a, 0xec, 0xc3,
    0x12, 0x7b, 0x1e, 0x41, 0x0a, 0x5d, 0x3f, 0x56, 0x4a, 0xa5, 0x6b, 0x97,
    0x83, 0x50, 0xe9, 0x3e, 0x2b, 0x98, 0x8b, 0x99, 0xa6, 0xd3, 0x1c, 0xb0,
    0xfd, 0x8b, 0x74, 0x06, 0xaa, 0xa2, 0xda, 0xaa, 0x5a, 0x9e, 0xd6, 0x6f,
    0xb7, 0x46, 0xe3, 0xa5, 0x8c, 0xc3, 0x61, 0x4c,
};
static const struct drbg_kat_no_reseed kat123_nor_t = {
    2, kat123_nor_entropyin, kat123_nor_nonce, kat123_nor_persstr,
    kat123_nor_addin0, kat123_nor_addin1, kat123_nor_retbytes
};
static const struct drbg_kat kat123_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat123_nor_t
};

static const unsigned char kat124_nor_entropyin[] = {
    0x8a, 0x08, 0x99, 0xc4, 0x65, 0x94, 0x9b, 0x49, 0x99, 0x61, 0x9c, 0x06,
    0x08, 0x69, 0x97, 0xe6,
};
static const unsigned char kat124_nor_nonce[] = {
    0xe1, 0xad, 0x18, 0xe1, 0x76, 0x45, 0xdf, 0x1c,
};
static const unsigned char kat124_nor_persstr[] = {0};
static const unsigned char kat124_nor_addin0[] = {0};
static const unsigned char kat124_nor_addin1[] = {0};
static const unsigned char kat124_nor_retbytes[] = {
    0x39, 0xde, 0x5b, 0x2b, 0x5d, 0xff, 0x4a, 0xad, 0x99, 0xcd, 0xfa, 0xbc,
    0xac, 0x9d, 0x7b, 0x3c, 0x0d, 0x0f, 0x9f, 0xed, 0xb1, 0xc1, 0xf0, 0x37,
    0x04, 0xb3, 0xb5, 0xbb, 0xd8, 0xc8, 0xea, 0x2c, 0xbd, 0x75, 0x9b, 0x4e,
    0x66, 0x8d, 0x12, 0x70, 0x85, 0x50, 0xba, 0xd5, 0xd5, 0xd3, 0x15, 0xac,
    0xec, 0x8c, 0x0a, 0xb9, 0x52, 0xbf, 0x2a, 0xc8, 0x69, 0x58, 0x6a, 0xaa,
    0x27, 0x86, 0x3d, 0x68, 0xd9, 0x18, 0x39, 0x60, 0xd5, 0x08, 0x13, 0x3e,
    0x4a, 0xdb, 0x9c, 0x8a, 0xf3, 0xe4, 0x1f, 0x53,
};
static const struct drbg_kat_no_reseed kat124_nor_t = {
    3, kat124_nor_entropyin, kat124_nor_nonce, kat124_nor_persstr,
    kat124_nor_addin0, kat124_nor_addin1, kat124_nor_retbytes
};
static const struct drbg_kat kat124_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat124_nor_t
};

static const unsigned char kat125_nor_entropyin[] = {
    0x5d, 0xd1, 0xda, 0x78, 0x87, 0x75, 0xb4, 0xa3, 0x06, 0x69, 0xb9, 0x2a,
    0x1c, 0x99, 0x2e, 0xba,
};
static const unsigned char kat125_nor_nonce[] = {
    0xb4, 0x3a, 0xa4, 0x1b, 0x02, 0xeb, 0x42, 0xcf,
};
static const unsigned char kat125_nor_persstr[] = {0};
static const unsigned char kat125_nor_addin0[] = {0};
static const unsigned char kat125_nor_addin1[] = {0};
static const unsigned char kat125_nor_retbytes[] = {
    0x8a, 0x45, 0x62, 0xf7, 0x66, 0x21, 0x61, 0x00, 0x04, 0xf4, 0x01, 0xfe,
    0x72, 0xf5, 0xd1, 0x40, 0x6d, 0xbe, 0x6a, 0xda, 0xeb, 0xd9, 0x26, 0xc2,
    0x06, 0x5f, 0x57, 0x46, 0x6b, 0xd7, 0x88, 0x30, 0x6b, 0x20, 0xd1, 0x3c,
    0x16, 0xd1, 0xe7, 0x62, 0x7c, 0xdb, 0x20, 0x0c, 0x25, 0x81, 0xeb, 0x23,
    0xb0, 0x2d, 0x58, 0x8d, 0xc4, 0xe5, 0x25, 0x1d, 0x61, 0x05, 0x62, 0xa0,
    0x84, 0xc1, 0x89, 0x40, 0xf2, 0xf4, 0xd5, 0xec, 0x30, 0x6c, 0x6e, 0xee,
    0x83, 0xee, 0x25, 0x7b, 0x93, 0xa7, 0x49, 0xd8,
};
static const struct drbg_kat_no_reseed kat125_nor_t = {
    4, kat125_nor_entropyin, kat125_nor_nonce, kat125_nor_persstr,
    kat125_nor_addin0, kat125_nor_addin1, kat125_nor_retbytes
};
static const struct drbg_kat kat125_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat125_nor_t
};

static const unsigned char kat126_nor_entropyin[] = {
    0x7c, 0x46, 0x37, 0xdc, 0x79, 0xe2, 0x9f, 0x7b, 0x85, 0x7f, 0x65, 0x70,
    0x0d, 0x6f, 0xa8, 0x91,
};
static const unsigned char kat126_nor_nonce[] = {
    0x8a, 0xb3, 0x67, 0x89, 0xa6, 0x8a, 0x4a, 0x94,
};
static const unsigned char kat126_nor_persstr[] = {0};
static const unsigned char kat126_nor_addin0[] = {0};
static const unsigned char kat126_nor_addin1[] = {0};
static const unsigned char kat126_nor_retbytes[] = {
    0x4d, 0xfd, 0x7b, 0xde, 0xe8, 0x96, 0x6c, 0x26, 0x03, 0x31, 0xad, 0xc8,
    0xa1, 0x76, 0x3b, 0xd4, 0xd1, 0x5b, 0xa7, 0x13, 0xd5, 0xce, 0x7e, 0x29,
    0x1d, 0x27, 0x43, 0x87, 0xd5, 0x9a, 0x98, 0x02, 0x8b, 0x36, 0x75, 0xc5,
    0xa5, 0x8c, 0x23, 0x7d, 0x8f, 0x40, 0x63, 0x8b, 0xfe, 0x48, 0x92, 0xa1,
    0x6c, 0x6f, 0x76, 0xe9, 0x2e, 0x8b, 0x43, 0x35, 0xf2, 0x8a, 0x93, 0xef,
    0x41, 0x01, 0x8e, 0x60, 0x22, 0xa9, 0x07, 0x2e, 0xef, 0x6c, 0xf0, 0xaa,
    0x83, 0x3d, 0x80, 0xf0, 0x56, 0xa8, 0x89, 0x8a,
};
static const struct drbg_kat_no_reseed kat126_nor_t = {
    5, kat126_nor_entropyin, kat126_nor_nonce, kat126_nor_persstr,
    kat126_nor_addin0, kat126_nor_addin1, kat126_nor_retbytes
};
static const struct drbg_kat kat126_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat126_nor_t
};

static const unsigned char kat127_nor_entropyin[] = {
    0x46, 0xf1, 0x5e, 0x71, 0x1d, 0xdd, 0x17, 0x2e, 0xd9, 0x94, 0x69, 0xa9,
    0xd1, 0x75, 0x80, 0x11,
};
static const unsigned char kat127_nor_nonce[] = {
    0xb2, 0x2c, 0x68, 0x5b, 0xd5, 0x46, 0x35, 0x21,
};
static const unsigned char kat127_nor_persstr[] = {0};
static const unsigned char kat127_nor_addin0[] = {0};
static const unsigned char kat127_nor_addin1[] = {0};
static const unsigned char kat127_nor_retbytes[] = {
    0xb2, 0xb0, 0x0d, 0x3e, 0xf6, 0xa9, 0xec, 0x8a, 0x8c, 0xcd, 0x82, 0x7f,
    0x82, 0x8f, 0xd5, 0x65, 0x56, 0xea, 0x3d, 0xbd, 0xc4, 0x7a, 0x73, 0xa3,
    0x15, 0xf0, 0x72, 0x53, 0xd6, 0x6e, 0x14, 0xf8, 0xfa, 0x02, 0x0e, 0x5b,
    0xf0, 0x25, 0xc7, 0xad, 0xc5, 0xf0, 0xb8, 0x1b, 0x2c, 0xd9, 0x92, 0x93,
    0x0e, 0x43, 0xe4, 0xb4, 0xe2, 0xd8, 0xdf, 0xb9, 0x26, 0xfc, 0x38, 0x7f,
    0x9f, 0xa9, 0x8c, 0x58, 0xf2, 0x9f, 0x26, 0x7d, 0xe1, 0xa8, 0x23, 0x5c,
    0x53, 0x27, 0xae, 0x85, 0x48, 0x6f, 0xa3, 0xba,
};
static const struct drbg_kat_no_reseed kat127_nor_t = {
    6, kat127_nor_entropyin, kat127_nor_nonce, kat127_nor_persstr,
    kat127_nor_addin0, kat127_nor_addin1, kat127_nor_retbytes
};
static const struct drbg_kat kat127_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat127_nor_t
};

static const unsigned char kat128_nor_entropyin[] = {
    0x89, 0x48, 0x66, 0x09, 0xb2, 0x1f, 0x4a, 0x77, 0xd4, 0x49, 0x03, 0x30,
    0x92, 0x38, 0xcd, 0x10,
};
static const unsigned char kat128_nor_nonce[] = {
    0x2c, 0x65, 0xe5, 0x7d, 0xc6, 0x2b, 0x71, 0x82,
};
static const unsigned char kat128_nor_persstr[] = {0};
static const unsigned char kat128_nor_addin0[] = {0};
static const unsigned char kat128_nor_addin1[] = {0};
static const unsigned char kat128_nor_retbytes[] = {
    0x04, 0x4e, 0xb7, 0x22, 0xca, 0xf8, 0x5f, 0x4a, 0xa4, 0x58, 0x64, 0x59,
    0x92, 0x5f, 0x77, 0x68, 0xa5, 0x95, 0x67, 0x40, 0x92, 0xf5, 0x59, 0xf0,
    0x4d, 0x78, 0xef, 0x6d, 0x94, 0xdb, 0x38, 0x2b, 0xb1, 0xeb, 0xbb, 0x7b,
    0x9e, 0x10, 0xba, 0x28, 0x89, 0x6e, 0x84, 0xfa, 0x54, 0xb1, 0x00, 0x71,
    0x5f, 0x6d, 0xe5, 0xd9, 0xc3, 0x13, 0x83, 0x23, 0x38, 0x30, 0x7c, 0xd4,
    0x80, 0xe7, 0xca, 0xa5, 0xb9, 0x58, 0xd3, 0x9f, 0x48, 0x99, 0xef, 0x8f,
    0x1e, 0x09, 0x26, 0xe9, 0xf9, 0xde, 0x96, 0x42,
};
static const struct drbg_kat_no_reseed kat128_nor_t = {
    7, kat128_nor_entropyin, kat128_nor_nonce, kat128_nor_persstr,
    kat128_nor_addin0, kat128_nor_addin1, kat128_nor_retbytes
};
static const struct drbg_kat kat128_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat128_nor_t
};

static const unsigned char kat129_nor_entropyin[] = {
    0xf1, 0x32, 0x23, 0x28, 0xdf, 0xf8, 0xc7, 0x33, 0x0d, 0x74, 0x08, 0x31,
    0x5f, 0x74, 0x76, 0xad,
};
static const unsigned char kat129_nor_nonce[] = {
    0x30, 0x26, 0xa9, 0x60, 0x7b, 0xc8, 0xd5, 0xa7,
};
static const unsigned char kat129_nor_persstr[] = {0};
static const unsigned char kat129_nor_addin0[] = {0};
static const unsigned char kat129_nor_addin1[] = {0};
static const unsigned char kat129_nor_retbytes[] = {
    0xb5, 0xfc, 0x47, 0x2a, 0x4a, 0x35, 0xd5, 0x9a, 0x0f, 0x29, 0x8e, 0xa5,
    0xfc, 0x14, 0x46, 0xd1, 0x50, 0x82, 0x0f, 0x8c, 0x78, 0x08, 0x85, 0x11,
    0xa4, 0x53, 0x97, 0xf4, 0x7f, 0xce, 0x79, 0xbd, 0xf5, 0xfd, 0x62, 0xaf,
    0x96, 0xe7, 0xd9, 0x7a, 0x83, 0xef, 0x50, 0x54, 0x59, 0xaa, 0xdd, 0x81,
    0xc2, 0x44, 0xbb, 0xe4, 0x70, 0x41, 0x96, 0xbd, 0x1f, 0x31, 0x24, 0xf6,
    0xd4, 0x21, 0xf7, 0xa8, 0x2b, 0xcf, 0x8b, 0xe4, 0x0e, 0x54, 0x60, 0xf0,
    0xf7, 0x4e, 0x6c, 0xa8, 0xec, 0x39, 0xd7, 0xa3,
};
static const struct drbg_kat_no_reseed kat129_nor_t = {
    8, kat129_nor_entropyin, kat129_nor_nonce, kat129_nor_persstr,
    kat129_nor_addin0, kat129_nor_addin1, kat129_nor_retbytes
};
static const struct drbg_kat kat129_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat129_nor_t
};

static const unsigned char kat130_nor_entropyin[] = {
    0x0c, 0xad, 0xd9, 0xc2, 0x73, 0x60, 0xad, 0x9c, 0xf3, 0xde, 0x96, 0xb6,
    0x20, 0x56, 0xf6, 0xc4,
};
static const unsigned char kat130_nor_nonce[] = {
    0xb2, 0xeb, 0x6a, 0x83, 0x95, 0x74, 0xc0, 0x19,
};
static const unsigned char kat130_nor_persstr[] = {0};
static const unsigned char kat130_nor_addin0[] = {0};
static const unsigned char kat130_nor_addin1[] = {0};
static const unsigned char kat130_nor_retbytes[] = {
    0xd2, 0x8a, 0x38, 0xeb, 0xc8, 0xae, 0xbf, 0xc4, 0x1d, 0x58, 0x11, 0x6d,
    0xf3, 0x6a, 0x35, 0x33, 0xc4, 0x8c, 0x87, 0x7b, 0xb8, 0x99, 0x2c, 0x30,
    0xea, 0x17, 0xb4, 0x6b, 0xa2, 0x94, 0x59, 0x67, 0xc8, 0x17, 0x8a, 0x50,
    0x0f, 0x27, 0x11, 0x97, 0xc6, 0x7e, 0x87, 0x66, 0xbb, 0x55, 0x2d, 0xf5,
    0x97, 0x0f, 0x07, 0xa7, 0x6a, 0x4d, 0x8e, 0x57, 0xfd, 0x64, 0x67, 0xb4,
    0x14, 0x84, 0x39, 0xdc, 0x16, 0x3c, 0xf7, 0x00, 0x78, 0x1c, 0x7b, 0xd8,
    0x1c, 0x1d, 0x1b, 0x2f, 0xc3, 0x84, 0x85, 0x10,
};
static const struct drbg_kat_no_reseed kat130_nor_t = {
    9, kat130_nor_entropyin, kat130_nor_nonce, kat130_nor_persstr,
    kat130_nor_addin0, kat130_nor_addin1, kat130_nor_retbytes
};
static const struct drbg_kat kat130_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat130_nor_t
};

static const unsigned char kat131_nor_entropyin[] = {
    0x12, 0xc2, 0xb4, 0x95, 0xcf, 0xa5, 0x5a, 0x44, 0x82, 0xdc, 0xee, 0x03,
    0x59, 0x23, 0xe8, 0x5e,
};
static const unsigned char kat131_nor_nonce[] = {
    0xcb, 0x5b, 0x88, 0x62, 0x2d, 0x93, 0x35, 0x89,
};
static const unsigned char kat131_nor_persstr[] = {0};
static const unsigned char kat131_nor_addin0[] = {0};
static const unsigned char kat131_nor_addin1[] = {0};
static const unsigned char kat131_nor_retbytes[] = {
    0x70, 0x75, 0x31, 0x2c, 0x4d, 0xc2, 0x1e, 0x54, 0xa2, 0xa8, 0x33, 0x46,
    0xee, 0x8d, 0x37, 0x10, 0xa2, 0xec, 0xeb, 0x34, 0xc0, 0x2d, 0xcd, 0xab,
    0x38, 0x17, 0xf8, 0xa6, 0x82, 0xcc, 0x0b, 0x1b, 0xf7, 0xb1, 0xc7, 0x77,
    0xcb, 0xd2, 0xaf, 0x1e, 0x45, 0x4f, 0xf4, 0x71, 0x12, 0x9a, 0x94, 0x05,
    0x9a, 0x25, 0x3c, 0x82, 0x95, 0x5b, 0x5a, 0xd9, 0x19, 0x63, 0x72, 0xd1,
    0xc9, 0x34, 0x16, 0x50, 0x0a, 0x65, 0xf9, 0x62, 0x15, 0x1a, 0x32, 0xc6,
    0x3b, 0xc6, 0xbc, 0x6b, 0x84, 0x43, 0xac, 0x9f,
};
static const struct drbg_kat_no_reseed kat131_nor_t = {
    10, kat131_nor_entropyin, kat131_nor_nonce, kat131_nor_persstr,
    kat131_nor_addin0, kat131_nor_addin1, kat131_nor_retbytes
};
static const struct drbg_kat kat131_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat131_nor_t
};

static const unsigned char kat132_nor_entropyin[] = {
    0x2f, 0xea, 0x96, 0x91, 0xd8, 0x25, 0x87, 0x63, 0xd5, 0xd0, 0xe8, 0x69,
    0x54, 0xcf, 0xd8, 0xda,
};
static const unsigned char kat132_nor_nonce[] = {
    0x3e, 0xf7, 0x1a, 0x30, 0x17, 0xee, 0xf2, 0x5a,
};
static const unsigned char kat132_nor_persstr[] = {0};
static const unsigned char kat132_nor_addin0[] = {0};
static const unsigned char kat132_nor_addin1[] = {0};
static const unsigned char kat132_nor_retbytes[] = {
    0x59, 0x56, 0xec, 0x9a, 0x16, 0xbc, 0x61, 0xb8, 0x4b, 0xc0, 0x01, 0x4e,
    0xee, 0x2f, 0x56, 0x15, 0xcc, 0x25, 0x90, 0x08, 0xba, 0xb2, 0x3a, 0xd6,
    0xa0, 0x0f, 0xf7, 0xe3, 0x66, 0x87, 0xe0, 0xf6, 0xee, 0x90, 0xea, 0x4b,
    0x90, 0xb3, 0xc7, 0xa9, 0xe9, 0x0c, 0x8f, 0x9d, 0x84, 0x4c, 0x91, 0x5b,
    0x4c, 0xaa, 0xea, 0x7c, 0x15, 0x8b, 0x0e, 0x91, 0x64, 0xd9, 0x8d, 0x2e,
    0x95, 0x31, 0xf0, 0x2f, 0x5d, 0xcb, 0x36, 0xa8, 0x4b, 0xeb, 0xcd, 0xbd,
    0x7b, 0x7a, 0x91, 0xa7, 0xf8, 0xe1, 0x35, 0x38,
};
static const struct drbg_kat_no_reseed kat132_nor_t = {
    11, kat132_nor_entropyin, kat132_nor_nonce, kat132_nor_persstr,
    kat132_nor_addin0, kat132_nor_addin1, kat132_nor_retbytes
};
static const struct drbg_kat kat132_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat132_nor_t
};

static const unsigned char kat133_nor_entropyin[] = {
    0x62, 0x4a, 0xb7, 0x6f, 0x32, 0x91, 0xda, 0x22, 0xfa, 0x54, 0xd4, 0x90,
    0x31, 0xfe, 0x02, 0xfc,
};
static const unsigned char kat133_nor_nonce[] = {
    0xd4, 0x01, 0xec, 0x34, 0xa7, 0xe4, 0xff, 0x2f,
};
static const unsigned char kat133_nor_persstr[] = {0};
static const unsigned char kat133_nor_addin0[] = {0};
static const unsigned char kat133_nor_addin1[] = {0};
static const unsigned char kat133_nor_retbytes[] = {
    0x8b, 0x86, 0x2d, 0x9b, 0xa4, 0xad, 0xaf, 0xd1, 0x7a, 0x45, 0x8c, 0x58,
    0x5d, 0x87, 0x06, 0xf1, 0x13, 0x2a, 0xee, 0xd8, 0xaa, 0x11, 0x9d, 0xda,
    0xf4, 0x41, 0x2a, 0x4a, 0x57, 0x16, 0xad, 0x44, 0x2c, 0x97, 0x4f, 0x6f,
    0xb3, 0x67, 0xba, 0xe7, 0x6f, 0xe0, 0xf3, 0x3f, 0x84, 0xd3, 0x11, 0x78,
    0x87, 0xac, 0x45, 0x92, 0xdf, 0xbc, 0x85, 0x99, 0x83, 0x9b, 0x8e, 0x43,
    0x6f, 0xb2, 0x26, 0x32, 0xfe, 0x43, 0xa8, 0xaa, 0x77, 0xff, 0xd8, 0x6d,
    0x09, 0xb4, 0xb3, 0x78, 0xa3, 0x9c, 0x61, 0x37,
};
static const struct drbg_kat_no_reseed kat133_nor_t = {
    12, kat133_nor_entropyin, kat133_nor_nonce, kat133_nor_persstr,
    kat133_nor_addin0, kat133_nor_addin1, kat133_nor_retbytes
};
static const struct drbg_kat kat133_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat133_nor_t
};

static const unsigned char kat134_nor_entropyin[] = {
    0x1d, 0xa2, 0xef, 0x22, 0x7c, 0xb7, 0x8d, 0xb7, 0x7f, 0xfa, 0x87, 0x2e,
    0x40, 0x76, 0x24, 0xc3,
};
static const unsigned char kat134_nor_nonce[] = {
    0x3c, 0x17, 0xde, 0x55, 0x28, 0x3f, 0xe6, 0x6a,
};
static const unsigned char kat134_nor_persstr[] = {0};
static const unsigned char kat134_nor_addin0[] = {0};
static const unsigned char kat134_nor_addin1[] = {0};
static const unsigned char kat134_nor_retbytes[] = {
    0x68, 0x2a, 0x99, 0x1f, 0xe8, 0x36, 0xc3, 0x2b, 0x61, 0x13, 0x40, 0xe6,
    0x98, 0xc4, 0x83, 0x01, 0xc4, 0xc0, 0xb1, 0x4a, 0x22, 0x1f, 0x2f, 0x75,
    0xe3, 0x7e, 0x68, 0xb4, 0xdb, 0x4c, 0xf9, 0xa2, 0x6a, 0x46, 0x5a, 0x8b,
    0xbe, 0x55, 0x58, 0x9a, 0x8a, 0x5d, 0x00, 0x60, 0x93, 0x91, 0x7c, 0xe8,
    0x59, 0xf9, 0x3b, 0x0f, 0x01, 0x6d, 0x43, 0xe2, 0xee, 0xb4, 0xed, 0x73,
    0x9a, 0x77, 0xe6, 0x5f, 0xa5, 0xc0, 0x88, 0x06, 0xce, 0x7a, 0x89, 0xd3,
    0x05, 0x40, 0x5a, 0x72, 0x0e, 0x29, 0x6d, 0x25,
};
static const struct drbg_kat_no_reseed kat134_nor_t = {
    13, kat134_nor_entropyin, kat134_nor_nonce, kat134_nor_persstr,
    kat134_nor_addin0, kat134_nor_addin1, kat134_nor_retbytes
};
static const struct drbg_kat kat134_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat134_nor_t
};

static const unsigned char kat135_nor_entropyin[] = {
    0xb7, 0xdd, 0xb8, 0x2f, 0x56, 0x64, 0x83, 0x4b, 0x4f, 0xb1, 0x77, 0x78,
    0xd2, 0x2e, 0x62, 0xf2,
};
static const unsigned char kat135_nor_nonce[] = {
    0x52, 0x46, 0x19, 0x24, 0xbe, 0xca, 0xb1, 0x75,
};
static const unsigned char kat135_nor_persstr[] = {0};
static const unsigned char kat135_nor_addin0[] = {0};
static const unsigned char kat135_nor_addin1[] = {0};
static const unsigned char kat135_nor_retbytes[] = {
    0x87, 0x35, 0xd0, 0x6e, 0x26, 0x81, 0x4e, 0xe5, 0x4b, 0x5d, 0xac, 0xa4,
    0xe1, 0xda, 0x3e, 0x32, 0x1a, 0x5a, 0x19, 0xb0, 0x62, 0xec, 0x0c, 0x3a,
    0xfb, 0xe3, 0xb1, 0x6f, 0x23, 0x33, 0x2a, 0x68, 0x7f, 0xad, 0xb2, 0x9e,
    0x65, 0x20, 0x81, 0x30, 0xc3, 0xd6, 0x67, 0xc0, 0x75, 0x66, 0x0f, 0xf7,
    0x0a, 0xea, 0x96, 0x43, 0x0f, 0xee, 0x25, 0x4c, 0x47, 0x26, 0x86, 0xb8,
    0xe8, 0x2c, 0xa3, 0x59, 0xa5, 0x7b, 0xbd, 0xc3, 0x00, 0x4b, 0xb3, 0xeb,
    0x64, 0x1c, 0x1f, 0x97, 0xe4, 0xb1, 0x9e, 0x02,
};
static const struct drbg_kat_no_reseed kat135_nor_t = {
    14, kat135_nor_entropyin, kat135_nor_nonce, kat135_nor_persstr,
    kat135_nor_addin0, kat135_nor_addin1, kat135_nor_retbytes
};
static const struct drbg_kat kat135_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat135_nor_t
};

static const unsigned char kat136_nor_entropyin[] = {
    0x21, 0xb6, 0x80, 0x8c, 0x0f, 0xc5, 0x9e, 0xcd, 0x92, 0x5e, 0xbe, 0x35,
    0xdd, 0xfe, 0xce, 0x08,
};
static const unsigned char kat136_nor_nonce[] = {
    0x09, 0xdd, 0x03, 0xcc, 0xac, 0x54, 0x85, 0xdf,
};
static const unsigned char kat136_nor_persstr[] = {0};
static const unsigned char kat136_nor_addin0[] = {
    0x22, 0x72, 0xf1, 0x97, 0x6f, 0x14, 0x65, 0xa0, 0x7f, 0xfd, 0x72, 0xae,
    0x25, 0x0e, 0x20, 0x0d,
};
static const unsigned char kat136_nor_addin1[] = {
    0x0b, 0x2e, 0x00, 0x90, 0xa2, 0x1a, 0xc3, 0x72, 0xc9, 0x4c, 0xe8, 0x9a,
    0x2c, 0xfb, 0xc5, 0xf0,
};
static const unsigned char kat136_nor_retbytes[] = {
    0x16, 0x59, 0x15, 0xd1, 0xba, 0x81, 0xeb, 0x3f, 0x0b, 0x02, 0x13, 0xb0,
    0x58, 0xe5, 0x28, 0x26, 0x5d, 0x29, 0x0c, 0x14, 0xd3, 0x0b, 0x53, 0x2a,
    0x7f, 0x3c, 0xf1, 0xc3, 0x94, 0x1a, 0x73, 0xef, 0x5f, 0xe3, 0x7c, 0x87,
    0x2d, 0x72, 0x41, 0xb1, 0x54, 0x45, 0x48, 0xa2, 0x3e, 0x92, 0x30, 0xcc,
    0x7f, 0x21, 0x15, 0xd3, 0xbb, 0x44, 0xa3, 0x0d, 0x1a, 0x34, 0xea, 0x93,
    0xbb, 0x5f, 0xed, 0x30, 0x5f, 0x95, 0xad, 0x7d, 0x4c, 0x3b, 0xe5, 0xef,
    0xed, 0xe5, 0xa8, 0x9e, 0xdf, 0x97, 0x55, 0x7e,
};
static const struct drbg_kat_no_reseed kat136_nor_t = {
    0, kat136_nor_entropyin, kat136_nor_nonce, kat136_nor_persstr,
    kat136_nor_addin0, kat136_nor_addin1, kat136_nor_retbytes
};
static const struct drbg_kat kat136_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat136_nor_t
};

static const unsigned char kat137_nor_entropyin[] = {
    0x38, 0x13, 0xf3, 0x76, 0xf4, 0xfd, 0x1c, 0x6d, 0x73, 0xf3, 0xe1, 0x60,
    0xd4, 0xf9, 0x98, 0xe9,
};
static const unsigned char kat137_nor_nonce[] = {
    0x95, 0x42, 0x85, 0x69, 0x74, 0x2d, 0x38, 0x7a,
};
static const unsigned char kat137_nor_persstr[] = {0};
static const unsigned char kat137_nor_addin0[] = {
    0x64, 0xa7, 0x33, 0xe7, 0x74, 0x59, 0x6f, 0xf4, 0x8d, 0xed, 0x12, 0x9b,
    0x04, 0xd9, 0x28, 0x5a,
};
static const unsigned char kat137_nor_addin1[] = {
    0x5b, 0x12, 0xc6, 0x91, 0x5e, 0x0b, 0x86, 0x99, 0x8b, 0x00, 0xed, 0x29,
    0x0b, 0x90, 0x1a, 0x30,
};
static const unsigned char kat137_nor_retbytes[] = {
    0x62, 0x24, 0xb1, 0xea, 0x5d, 0x09, 0x72, 0x90, 0xa6, 0x03, 0xb7, 0x2f,
    0xdb, 0x2a, 0x94, 0xdc, 0xd8, 0x55, 0xfc, 0x70, 0x2f, 0x9e, 0x7f, 0x0c,
    0x1f, 0x4d, 0x57, 0xfa, 0x1b, 0x64, 0xe2, 0x0f, 0x66, 0x0b, 0xeb, 0x61,
    0x20, 0xf6, 0xd6, 0xee, 0x24, 0x76, 0x5b, 0x50, 0x67, 0x4f, 0x1e, 0x61,
    0x2b, 0xb2, 0x91, 0x59, 0xd0, 0x99, 0x34, 0xba, 0x18, 0x40, 0x33, 0x28,
    0xed, 0xf8, 0x2f, 0x35, 0x30, 0x29, 0x1e, 0x61, 0x72, 0xbc, 0x4b, 0x7e,
    0x76, 0xe7, 0xda, 0x5d, 0xb0, 0xe6, 0xcf, 0xcd,
};
static const struct drbg_kat_no_reseed kat137_nor_t = {
    1, kat137_nor_entropyin, kat137_nor_nonce, kat137_nor_persstr,
    kat137_nor_addin0, kat137_nor_addin1, kat137_nor_retbytes
};
static const struct drbg_kat kat137_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat137_nor_t
};

static const unsigned char kat138_nor_entropyin[] = {
    0xe7, 0x50, 0x04, 0xf4, 0x1a, 0xe0, 0x7d, 0xb8, 0xec, 0xef, 0xca, 0x47,
    0x88, 0x8e, 0xc1, 0x14,
};
static const unsigned char kat138_nor_nonce[] = {
    0x1c, 0x1d, 0x86, 0x98, 0x0d, 0x1d, 0xad, 0x6c,
};
static const unsigned char kat138_nor_persstr[] = {0};
static const unsigned char kat138_nor_addin0[] = {
    0xfe, 0x3a, 0x34, 0x1a, 0x5c, 0x8f, 0xaf, 0xf3, 0x15, 0x59, 0x79, 0x3f,
    0x3d, 0xd9, 0xb4, 0x3f,
};
static const unsigned char kat138_nor_addin1[] = {
    0x95, 0xee, 0x78, 0x2a, 0xc0, 0xac, 0x3e, 0xa4, 0xd2, 0xc5, 0x14, 0x8f,
    0x3b, 0x37, 0x74, 0x7f,
};
static const unsigned char kat138_nor_retbytes[] = {
    0xbf, 0x50, 0x6d, 0xd7, 0x0b, 0x76, 0xc9, 0x54, 0x1a, 0x06, 0x3b, 0x8c,
    0xc4, 0xaa, 0x6f, 0x93, 0xb5, 0x54, 0x9d, 0x71, 0xdf, 0x89, 0x98, 0x23,
    0xc9, 0xb6, 0x9c, 0x47, 0xd3, 0x0a, 0x77, 0xc4, 0xcd, 0x6f, 0x4e, 0x50,
    0x65, 0x4c, 0x49, 0x55, 0xa8, 0xe5, 0x58, 0x1e, 0x43, 0xbe, 0x2c, 0xf4,
    0x6a, 0x52, 0x89, 0x56, 0x0a, 0xcf, 0x00, 0x0b, 0xf5, 0x44, 0x30, 0xa1,
    0x94, 0x9b, 0x52, 0x43, 0xe9, 0xa7, 0x99, 0xdb, 0x3c, 0xb0, 0x86, 0xf0,
    0x8e, 0x6e, 0x1e, 0x34, 0xe5, 0x4d, 0x32, 0x08,
};
static const struct drbg_kat_no_reseed kat138_nor_t = {
    2, kat138_nor_entropyin, kat138_nor_nonce, kat138_nor_persstr,
    kat138_nor_addin0, kat138_nor_addin1, kat138_nor_retbytes
};
static const struct drbg_kat kat138_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat138_nor_t
};

static const unsigned char kat139_nor_entropyin[] = {
    0x78, 0x64, 0x71, 0x92, 0x8f, 0xf5, 0x58, 0xa2, 0x93, 0xf4, 0xa3, 0x66,
    0x86, 0x61, 0x18, 0x72,
};
static const unsigned char kat139_nor_nonce[] = {
    0xdc, 0x41, 0x93, 0x0c, 0x51, 0x63, 0xbc, 0x77,
};
static const unsigned char kat139_nor_persstr[] = {0};
static const unsigned char kat139_nor_addin0[] = {
    0xfc, 0xfc, 0x99, 0xf0, 0x5f, 0xe9, 0xa0, 0x58, 0xc0, 0x47, 0xdc, 0x03,
    0xc0, 0x17, 0x1c, 0x6f,
};
static const unsigned char kat139_nor_addin1[] = {
    0xeb, 0xa3, 0xd9, 0x77, 0x42, 0x43, 0xa2, 0x73, 0xee, 0xd0, 0x91, 0xd5,
    0x44, 0x4a, 0x7b, 0xa2,
};
static const unsigned char kat139_nor_retbytes[] = {
    0x5e, 0xb1, 0x33, 0xf8, 0xf1, 0xef, 0x9d, 0x76, 0x4c, 0xe4, 0x50, 0xd6,
    0xd2, 0xbe, 0xda, 0xb3, 0xdb, 0xc8, 0x2f, 0x5a, 0x95, 0x6b, 0xe5, 0x7a,
    0xc4, 0x08, 0xd6, 0xba, 0x92, 0x51, 0xe6, 0x42, 0xbd, 0xa0, 0xbb, 0xcd,
    0x92, 0x0e, 0xfe, 0xe3, 0x2b, 0xbe, 0x77, 0xc3, 0xab, 0xe8, 0x45, 0xb4,
    0xaa, 0x0c, 0xb2, 0xdd, 0x45, 0x0e, 0x97, 0x10, 0x8c, 0xa2, 0xa9, 0xa0,
    0xe8, 0xfa, 0x9b, 0x3c, 0x4a, 0xdd, 0xc2, 0x0b, 0xdb, 0x2a, 0xb5, 0xd7,
    0x6a, 0x5a, 0x4e, 0xee, 0x3e, 0x01, 0x57, 0x93,
};
static const struct drbg_kat_no_reseed kat139_nor_t = {
    3, kat139_nor_entropyin, kat139_nor_nonce, kat139_nor_persstr,
    kat139_nor_addin0, kat139_nor_addin1, kat139_nor_retbytes
};
static const struct drbg_kat kat139_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat139_nor_t
};

static const unsigned char kat140_nor_entropyin[] = {
    0x51, 0x35, 0x9d, 0xd9, 0x2b, 0xf3, 0xb8, 0xc4, 0x32, 0xd3, 0x28, 0xaa,
    0x4a, 0x6c, 0xe8, 0xda,
};
static const unsigned char kat140_nor_nonce[] = {
    0xf8, 0x44, 0xb3, 0x4b, 0xab, 0xf2, 0xbc, 0x9b,
};
static const unsigned char kat140_nor_persstr[] = {0};
static const unsigned char kat140_nor_addin0[] = {
    0x59, 0x3b, 0x02, 0x5d, 0x32, 0x78, 0xc3, 0x2b, 0x26, 0x97, 0x07, 0x61,
    0x95, 0x88, 0x8e, 0xbf,
};
static const unsigned char kat140_nor_addin1[] = {
    0xae, 0x23, 0xf5, 0x1e, 0xbb, 0x06, 0xf5, 0x17, 0xcd, 0x5c, 0xcd, 0x2b,
    0x86, 0xca, 0xfd, 0x28,
};
static const unsigned char kat140_nor_retbytes[] = {
    0x1d, 0x9f, 0xe2, 0x8a, 0x9d, 0x69, 0x5d, 0x14, 0x20, 0x0d, 0x19, 0x36,
    0x36, 0x93, 0x3c, 0x94, 0xac, 0x11, 0x8f, 0x2e, 0xc3, 0x3c, 0x16, 0xa8,
    0xa6, 0x71, 0xf9, 0xc1, 0xef, 0x47, 0x4b, 0x5b, 0xcf, 0xb1, 0x8e, 0x99,
    0xd2, 0x9b, 0xa6, 0xd5, 0x5c, 0x1a, 0x07, 0xea, 0x42, 0x95, 0x8b, 0x5f,
    0xf6, 0x46, 0x85, 0x17, 0x2c, 0xd7, 0xde, 0x24, 0xfa, 0xf0, 0x65, 0x10,
    0xd5, 0xb9, 0xf7, 0xc8, 0x4c, 0xf8, 0x95, 0x5a, 0xf6, 0x70, 0x7b, 0x93,
    0x21, 0x69, 0x23, 0x36, 0xdf, 0xed, 0xf2, 0xec,
};
static const struct drbg_kat_no_reseed kat140_nor_t = {
    4, kat140_nor_entropyin, kat140_nor_nonce, kat140_nor_persstr,
    kat140_nor_addin0, kat140_nor_addin1, kat140_nor_retbytes
};
static const struct drbg_kat kat140_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat140_nor_t
};

static const unsigned char kat141_nor_entropyin[] = {
    0x56, 0x72, 0xab, 0x20, 0xf6, 0x67, 0xe5, 0x91, 0x8a, 0x80, 0x87, 0x68,
    0x1c, 0x18, 0x7d, 0xac,
};
static const unsigned char kat141_nor_nonce[] = {
    0xa9, 0xb4, 0x2c, 0xfb, 0x74, 0xcc, 0xce, 0xaa,
};
static const unsigned char kat141_nor_persstr[] = {0};
static const unsigned char kat141_nor_addin0[] = {
    0x6f, 0x4d, 0x34, 0xdd, 0x4c, 0xc4, 0x1d, 0x01, 0x5d, 0x00, 0x26, 0x30,
    0xa5, 0x7b, 0x5d, 0xa5,
};
static const unsigned char kat141_nor_addin1[] = {
    0xd5, 0x71, 0xf6, 0xf4, 0xc1, 0x20, 0x0b, 0x61, 0x94, 0x83, 0xb6, 0x91,
    0x66, 0xa5, 0x9c, 0xdc,
};
static const unsigned char kat141_nor_retbytes[] = {
    0x6c, 0x43, 0x69, 0x29, 0x0d, 0x51, 0x7d, 0x40, 0x97, 0x26, 0x0f, 0x88,
    0x61, 0x8c, 0x36, 0x6a, 0x85, 0x53, 0x51, 0xbf, 0x5e, 0x78, 0xab, 0xcb,
    0xb2, 0x33, 0x21, 0xf5, 0xf0, 0xf2, 0xda, 0x59, 0x2b, 0x21, 0x02, 0x4f,
    0xc7, 0xe9, 0xb2, 0x7d, 0x76, 0x8e, 0x84, 0x1e, 0x8d, 0x77, 0x8d, 0x43,
    0x67, 0xa0, 0xfa, 0xca, 0x95, 0x9a, 0x2c, 0x9b, 0x96, 0xbd, 0x2f, 0x25,
    0x6f, 0xbd, 0xd3, 0x45, 0xab, 0x9c, 0xdd, 0x5b, 0x03, 0x5b, 0xdd, 0x7a,
    0xc3, 0xc5, 0xd3, 0x56, 0xbc, 0x1e, 0xd8, 0xe4,
};
static const struct drbg_kat_no_reseed kat141_nor_t = {
    5, kat141_nor_entropyin, kat141_nor_nonce, kat141_nor_persstr,
    kat141_nor_addin0, kat141_nor_addin1, kat141_nor_retbytes
};
static const struct drbg_kat kat141_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat141_nor_t
};

static const unsigned char kat142_nor_entropyin[] = {
    0xb6, 0x5b, 0xc4, 0xcf, 0xa1, 0x15, 0xbc, 0xdd, 0x15, 0x73, 0x0e, 0xf5,
    0xf7, 0xb3, 0xd7, 0xb1,
};
static const unsigned char kat142_nor_nonce[] = {
    0x57, 0x7f, 0x9b, 0xe9, 0x0f, 0x62, 0x5e, 0xde,
};
static const unsigned char kat142_nor_persstr[] = {0};
static const unsigned char kat142_nor_addin0[] = {
    0x9e, 0x76, 0xda, 0xe1, 0xd2, 0xf6, 0x8f, 0x69, 0x93, 0xe5, 0x31, 0xe6,
    0x13, 0xf6, 0x22, 0x63,
};
static const unsigned char kat142_nor_addin1[] = {
    0x16, 0xa5, 0xfc, 0x59, 0x9f, 0x9d, 0x94, 0xe2, 0xd3, 0x71, 0x7c, 0xf7,
    0x75, 0x51, 0xd4, 0x7e,
};
static const unsigned char kat142_nor_retbytes[] = {
    0x33, 0xe0, 0x9a, 0x71, 0x0a, 0xb4, 0x10, 0x34, 0xd1, 0x6b, 0xb1, 0x36,
    0x9e, 0x2b, 0xc1, 0x57, 0xf3, 0x33, 0xa3, 0xba, 0x6d, 0xfc, 0x73, 0x00,
    0xf6, 0xfe, 0x60, 0x0c, 0x52, 0x45, 0xb0, 0x43, 0xc5, 0x91, 0x8d, 0xba,
    0x6b, 0xd3, 0xbc, 0xa6, 0xaf, 0xcf, 0xff, 0x33, 0x25, 0xad, 0xab, 0xe5,
    0x50, 0x88, 0x4f, 0x80, 0xb0, 0x72, 0x39, 0x68, 0x4b, 0xd2, 0xd8, 0xf9,
    0xd5, 0x58, 0xf5, 0x8b, 0x52, 0x08, 0x28, 0x06, 0x7e, 0xe9, 0x0f, 0x47,
    0xb3, 0x09, 0x54, 0xa2, 0x67, 0x6c, 0x5c, 0xec,
};
static const struct drbg_kat_no_reseed kat142_nor_t = {
    6, kat142_nor_entropyin, kat142_nor_nonce, kat142_nor_persstr,
    kat142_nor_addin0, kat142_nor_addin1, kat142_nor_retbytes
};
static const struct drbg_kat kat142_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat142_nor_t
};

static const unsigned char kat143_nor_entropyin[] = {
    0x56, 0x39, 0xff, 0xf8, 0xa2, 0x3a, 0xe7, 0xc5, 0x1b, 0xce, 0x07, 0x44,
    0xcc, 0x69, 0xe5, 0x34,
};
static const unsigned char kat143_nor_nonce[] = {
    0xc6, 0xd5, 0x63, 0x2f, 0xd6, 0x89, 0x9d, 0xa2,
};
static const unsigned char kat143_nor_persstr[] = {0};
static const unsigned char kat143_nor_addin0[] = {
    0xab, 0xce, 0x9c, 0x7f, 0xb0, 0x0d, 0x7e, 0x48, 0x01, 0x65, 0x1f, 0x89,
    0xd5, 0xbd, 0x47, 0xea,
};
static const unsigned char kat143_nor_addin1[] = {
    0x1d, 0xa0, 0x83, 0x32, 0x02, 0xc8, 0x2d, 0xa9, 0x71, 0xe8, 0x19, 0x8e,
    0xe7, 0xaa, 0x0b, 0x90,
};
static const unsigned char kat143_nor_retbytes[] = {
    0x7c, 0x0a, 0x23, 0x05, 0xe5, 0x23, 0x3e, 0xde, 0xed, 0xa4, 0x21, 0x90,
    0x6d, 0x85, 0x28, 0x42, 0x6c, 0xcc, 0x45, 0x5c, 0xcd, 0xc2, 0x8f, 0x30,
    0xdf, 0x3b, 0x31, 0xad, 0x8b, 0xec, 0x99, 0x0b, 0xfc, 0x28, 0x3f, 0xe5,
    0xff, 0xba, 0xba, 0x93, 0xb8, 0x80, 0xf9, 0xa0, 0x17, 0x24, 0x68, 0x4c,
    0xe8, 0xe7, 0x24, 0x2a, 0x26, 0xc3, 0xe9, 0xb1, 0xa9, 0xd2, 0x3a, 0x14,
    0xa4, 0x81, 0xe5, 0xd3, 0xc7, 0x56, 0x12, 0xc6, 0xbe, 0x27, 0x63, 0x36,
    0x1a, 0x74, 0x9e, 0x8e, 0xce, 0x57, 0xf7, 0xf5,
};
static const struct drbg_kat_no_reseed kat143_nor_t = {
    7, kat143_nor_entropyin, kat143_nor_nonce, kat143_nor_persstr,
    kat143_nor_addin0, kat143_nor_addin1, kat143_nor_retbytes
};
static const struct drbg_kat kat143_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat143_nor_t
};

static const unsigned char kat144_nor_entropyin[] = {
    0xb0, 0x39, 0xd4, 0xad, 0xad, 0x57, 0x68, 0x0a, 0xd2, 0xa0, 0xf7, 0x43,
    0xa5, 0x06, 0x33, 0xa0,
};
static const unsigned char kat144_nor_nonce[] = {
    0x0a, 0x47, 0x2b, 0xdd, 0x9f, 0x6e, 0x2b, 0xea,
};
static const unsigned char kat144_nor_persstr[] = {0};
static const unsigned char kat144_nor_addin0[] = {
    0xc5, 0x39, 0x24, 0x1e, 0xe0, 0x59, 0x44, 0xed, 0x08, 0x74, 0xea, 0xf3,
    0xaa, 0xbf, 0x34, 0x2a,
};
static const unsigned char kat144_nor_addin1[] = {
    0xe9, 0x7f, 0x92, 0xe5, 0x6c, 0xde, 0x35, 0x0d, 0x19, 0x01, 0x20, 0x69,
    0x03, 0x60, 0x5f, 0x66,
};
static const unsigned char kat144_nor_retbytes[] = {
    0x22, 0x83, 0x61, 0xa8, 0xb6, 0x11, 0x3e, 0xed, 0x3d, 0x84, 0xed, 0x46,
    0xa0, 0x57, 0x31, 0x34, 0xc7, 0x0d, 0x2d, 0x5a, 0x18, 0x7a, 0x35, 0xb4,
    0x59, 0x9f, 0x2b, 0xf1, 0x82, 0x03, 0x09, 0xe7, 0x21, 0x1d, 0xea, 0x34,
    0xd3, 0x3a, 0x62, 0xdf, 0xa7, 0x42, 0xa2, 0x16, 0x75, 0x2a, 0x9e, 0x57,
    0xf3, 0x3a, 0x60, 0x4b, 0xe3, 0xe8, 0x8e, 0xa0, 0xf1, 0x5f, 0x5d, 0x91,
    0x6d, 0x72, 0x66, 0x46, 0x73, 0xd8, 0x86, 0x10, 0x4a, 0x7b, 0x6c, 0x39,
    0x04, 0xaf, 0x6d, 0x65, 0x63, 0xe0, 0x57, 0x61,
};
static const struct drbg_kat_no_reseed kat144_nor_t = {
    8, kat144_nor_entropyin, kat144_nor_nonce, kat144_nor_persstr,
    kat144_nor_addin0, kat144_nor_addin1, kat144_nor_retbytes
};
static const struct drbg_kat kat144_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat144_nor_t
};

static const unsigned char kat145_nor_entropyin[] = {
    0x71, 0xe7, 0xd5, 0xb1, 0x84, 0x57, 0x07, 0xeb, 0xbe, 0xd4, 0x71, 0xbc,
    0x66, 0x3b, 0x85, 0x17,
};
static const unsigned char kat145_nor_nonce[] = {
    0x42, 0x50, 0x3f, 0xc9, 0x49, 0x68, 0x64, 0xa8,
};
static const unsigned char kat145_nor_persstr[] = {0};
static const unsigned char kat145_nor_addin0[] = {
    0x59, 0xea, 0x5d, 0xc5, 0x5e, 0x22, 0xe0, 0x34, 0xf1, 0xed, 0x46, 0xd1,
    0x73, 0x9f, 0x1d, 0x33,
};
static const unsigned char kat145_nor_addin1[] = {
    0xd9, 0x7a, 0xf0, 0x12, 0x77, 0x04, 0x33, 0x95, 0xdb, 0xef, 0x3a, 0x0e,
    0xac, 0xd6, 0x16, 0x3d,
};
static const unsigned char kat145_nor_retbytes[] = {
    0xd6, 0xc5, 0xca, 0xd4, 0xaf, 0x95, 0x48, 0x34, 0x28, 0x49, 0x7f, 0x0f,
    0xf0, 0xd8, 0xa3, 0x0f, 0x67, 0x37, 0x87, 0xf4, 0x4f, 0xfc, 0xb9, 0xbf,
    0x55, 0xc1, 0xed, 0x39, 0xfa, 0x41, 0xe3, 0xf5, 0x38, 0x49, 0xda, 0x76,
    0xfa, 0xa5, 0x37, 0x7e, 0xaa, 0x83, 0x2c, 0x83, 0x5e, 0xea, 0x20, 0x02,
    0x26, 0xf5, 0xf3, 0xf0, 0x57, 0x5d, 0x67, 0x51, 0x48, 0x20, 0x02, 0x43,
    0x39, 0xe2, 0x48, 0xb0, 0x91, 0x11, 0x19, 0x5c, 0x0b, 0xb7, 0x14, 0x8f,
    0xac, 0xf3, 0x10, 0xb5, 0x93, 0x94, 0x72, 0x56,
};
static const struct drbg_kat_no_reseed kat145_nor_t = {
    9, kat145_nor_entropyin, kat145_nor_nonce, kat145_nor_persstr,
    kat145_nor_addin0, kat145_nor_addin1, kat145_nor_retbytes
};
static const struct drbg_kat kat145_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat145_nor_t
};

static const unsigned char kat146_nor_entropyin[] = {
    0xef, 0x32, 0x52, 0x65, 0xd5, 0xaf, 0x25, 0xd2, 0x55, 0x78, 0x4e, 0x10,
    0x9b, 0x20, 0xb7, 0x61,
};
static const unsigned char kat146_nor_nonce[] = {
    0x52, 0xf6, 0xfa, 0xa7, 0xf3, 0x7c, 0x7d, 0xf2,
};
static const unsigned char kat146_nor_persstr[] = {0};
static const unsigned char kat146_nor_addin0[] = {
    0x77, 0x09, 0xe2, 0x20, 0xf0, 0x67, 0xba, 0x16, 0x1c, 0xd3, 0x61, 0x63,
    0x95, 0x80, 0xc4, 0x2e,
};
static const unsigned char kat146_nor_addin1[] = {
    0xed, 0x6a, 0xcb, 0x90, 0x2f, 0x1d, 0x02, 0x66, 0xc1, 0x4f, 0x29, 0x93,
    0xbd, 0xb7, 0x7f, 0xf9,
};
static const unsigned char kat146_nor_retbytes[] = {
    0x3b, 0x8b, 0xc3, 0xe1, 0xca, 0xb6, 0x7f, 0x41, 0x84, 0x1f, 0x74, 0xb0,
    0xd2, 0x0b, 0x87, 0xb9, 0x54, 0x83, 0xbe, 0x53, 0xda, 0x22, 0x9f, 0xe3,
    0x4b, 0x47, 0x15, 0x1d, 0x2a, 0xc2, 0x80, 0x8c, 0xeb, 0x6e, 0x50, 0xda,
    0x6c, 0xff, 0xfb, 0x14, 0xb5, 0x5d, 0x0f, 0x3b, 0x84, 0x66, 0x76, 0x61,
    0x21, 0x30, 0xa7, 0x01, 0x19, 0x15, 0xb9, 0x46, 0xdf, 0x67, 0x18, 0x95,
    0x14, 0xb6, 0x9c, 0x11, 0x51, 0x4c, 0x06, 0xce, 0x27, 0x1d, 0xc3, 0x70,
    0xf1, 0x36, 0xf6, 0xc5, 0x08, 0x9f, 0x8c, 0x6d,
};
static const struct drbg_kat_no_reseed kat146_nor_t = {
    10, kat146_nor_entropyin, kat146_nor_nonce, kat146_nor_persstr,
    kat146_nor_addin0, kat146_nor_addin1, kat146_nor_retbytes
};
static const struct drbg_kat kat146_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat146_nor_t
};

static const unsigned char kat147_nor_entropyin[] = {
    0x80, 0xda, 0x5b, 0x7e, 0x53, 0x16, 0xc9, 0x9a, 0xe6, 0xa2, 0x1e, 0xcf,
    0xe2, 0xe0, 0x13, 0xb0,
};
static const unsigned char kat147_nor_nonce[] = {
    0x28, 0xdc, 0xd7, 0x15, 0x91, 0xa7, 0x64, 0xb5,
};
static const unsigned char kat147_nor_persstr[] = {0};
static const unsigned char kat147_nor_addin0[] = {
    0x39, 0xa1, 0x65, 0x63, 0x0c, 0x5d, 0xbf, 0x4e, 0xcc, 0x53, 0xf8, 0xdd,
    0x82, 0x5d, 0x98, 0x2a,
};
static const unsigned char kat147_nor_addin1[] = {
    0x57, 0x6a, 0xfb, 0xba, 0x4f, 0x94, 0x08, 0x05, 0x09, 0x38, 0xb8, 0xd2,
    0x04, 0xa8, 0xb1, 0xf6,
};
static const unsigned char kat147_nor_retbytes[] = {
    0x9d, 0x0b, 0x9b, 0x54, 0xb9, 0xa2, 0xcb, 0x61, 0x86, 0xd4, 0x3b, 0x12,
    0x82, 0xfd, 0x21, 0x4a, 0x00, 0xcb, 0x41, 0x39, 0x04, 0x02, 0x3b, 0x48,
    0x20, 0x08, 0x4f, 0x36, 0xa0, 0xf0, 0xe5, 0x08, 0x88, 0xcb, 0x8b, 0x33,
    0xf6, 0x73, 0x15, 0x55, 0x59, 0x5e, 0xa5, 0xb2, 0x05, 0x21, 0xf1, 0x8d,
    0x1e, 0x94, 0xaa, 0x8e, 0xab, 0xaf, 0xcf, 0x16, 0x86, 0xcd, 0xc8, 0x50,
    0x96, 0x59, 0x11, 0x16, 0x6f, 0x94, 0x11, 0x50, 0x13, 0xe7, 0x8d, 0x85,
    0x80, 0x4a, 0x63, 0x4c, 0xf0, 0x7a, 0x3d, 0x55,
};
static const struct drbg_kat_no_reseed kat147_nor_t = {
    11, kat147_nor_entropyin, kat147_nor_nonce, kat147_nor_persstr,
    kat147_nor_addin0, kat147_nor_addin1, kat147_nor_retbytes
};
static const struct drbg_kat kat147_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat147_nor_t
};

static const unsigned char kat148_nor_entropyin[] = {
    0xd7, 0x49, 0x21, 0xd6, 0xdf, 0xa6, 0x9d, 0x7a, 0x73, 0x86, 0x10, 0x16,
    0xcf, 0x15, 0x5c, 0xbb,
};
static const unsigned char kat148_nor_nonce[] = {
    0x3e, 0x49, 0x48, 0x39, 0x02, 0x08, 0x60, 0x04,
};
static const unsigned char kat148_nor_persstr[] = {0};
static const unsigned char kat148_nor_addin0[] = {
    0x48, 0x39, 0xcd, 0x93, 0x86, 0xe6, 0x82, 0x98, 0xaa, 0x27, 0x4b, 0x83,
    0xf5, 0xae, 0x91, 0x34,
};
static const unsigned char kat148_nor_addin1[] = {
    0xc0, 0x8d, 0x03, 0xf6, 0x5e, 0xea, 0xbd, 0xe5, 0xc9, 0xbf, 0x72, 0x30,
    0xa5, 0xa3, 0xbf, 0x91,
};
static const unsigned char kat148_nor_retbytes[] = {
    0x17, 0x2b, 0x29, 0xd4, 0x6e, 0x9a, 0xcd, 0x26, 0xc0, 0xa3, 0x82, 0x71,
    0xad, 0xaf, 0x92, 0x6e, 0x4e, 0xce, 0x48, 0x03, 0xf2, 0x13, 0xab, 0xd6,
    0xa1, 0x06, 0x5b, 0x8f, 0xe3, 0x14, 0x27, 0xf1, 0xf5, 0xbc, 0xac, 0x78,
    0xf2, 0xf2, 0x15, 0x6e, 0x78, 0x11, 0xe9, 0xc5, 0xc9, 0x84, 0xf9, 0x7a,
    0xef, 0xfa, 0xa4, 0x51, 0x9d, 0xae, 0xd0, 0x67, 0xff, 0xa1, 0x97, 0x59,
    0x45, 0x89, 0x74, 0x60, 0x54, 0x8f, 0x78, 0xb6, 0x73, 0x70, 0xd3, 0x32,
    0xaf, 0x3a, 0xf1, 0xf9, 0xa1, 0x2d, 0xdc, 0x22,
};
static const struct drbg_kat_no_reseed kat148_nor_t = {
    12, kat148_nor_entropyin, kat148_nor_nonce, kat148_nor_persstr,
    kat148_nor_addin0, kat148_nor_addin1, kat148_nor_retbytes
};
static const struct drbg_kat kat148_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat148_nor_t
};

static const unsigned char kat149_nor_entropyin[] = {
    0xd4, 0xa2, 0xf2, 0xde, 0xd0, 0x1a, 0xe7, 0xa5, 0xc3, 0x3b, 0x09, 0xf6,
    0x38, 0xf9, 0xae, 0x5c,
};
static const unsigned char kat149_nor_nonce[] = {
    0xee, 0xab, 0x2a, 0xc9, 0x71, 0xbc, 0x8f, 0x6b,
};
static const unsigned char kat149_nor_persstr[] = {0};
static const unsigned char kat149_nor_addin0[] = {
    0xc3, 0xe7, 0x27, 0xb3, 0xce, 0x8c, 0xf2, 0x20, 0x0d, 0xfc, 0x06, 0x7d,
    0x13, 0xc8, 0x02, 0x18,
};
static const unsigned char kat149_nor_addin1[] = {
    0x6d, 0x4c, 0x88, 0x9b, 0x91, 0x60, 0x38, 0x90, 0x45, 0x82, 0x77, 0xb9,
    0xfc, 0xc5, 0x7a, 0xa8,
};
static const unsigned char kat149_nor_retbytes[] = {
    0x9d, 0xef, 0xb2, 0x3c, 0x7c, 0xbf, 0x32, 0xe2, 0xf8, 0x95, 0xd8, 0xbe,
    0x3c, 0xcd, 0x9b, 0x0c, 0x6f, 0x2d, 0x6c, 0xdd, 0x3d, 0xb0, 0xaa, 0xd9,
    0xa3, 0xa5, 0xf2, 0xf6, 0x18, 0x9a, 0x44, 0x17, 0x58, 0xbf, 0x1a, 0x62,
    0x23, 0x13, 0xa9, 0xad, 0x7d, 0xa7, 0xd2, 0x95, 0x36, 0x4e, 0xf8, 0x57,
    0x90, 0x1e, 0x22, 0x29, 0xe7, 0x31, 0x5b, 0x06, 0xc7, 0x55, 0x98, 0xc9,
    0xce, 0x6d, 0xb2, 0x6a, 0x96, 0x6c, 0x3d, 0xfa, 0x49, 0xdf, 0x45, 0x51,
    0x4b, 0xdf, 0x46, 0x12, 0x5c, 0x62, 0x28, 0x66,
};
static const struct drbg_kat_no_reseed kat149_nor_t = {
    13, kat149_nor_entropyin, kat149_nor_nonce, kat149_nor_persstr,
    kat149_nor_addin0, kat149_nor_addin1, kat149_nor_retbytes
};
static const struct drbg_kat kat149_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat149_nor_t
};

static const unsigned char kat150_nor_entropyin[] = {
    0xbf, 0x76, 0x41, 0x51, 0x13, 0x65, 0x97, 0xee, 0xf9, 0xc1, 0xa7, 0xcc,
    0xff, 0x0f, 0x34, 0x5f,
};
static const unsigned char kat150_nor_nonce[] = {
    0x29, 0xf9, 0x05, 0x9f, 0x90, 0x81, 0x6c, 0x57,
};
static const unsigned char kat150_nor_persstr[] = {0};
static const unsigned char kat150_nor_addin0[] = {
    0x77, 0x25, 0xef, 0x70, 0x59, 0x2c, 0x36, 0x2d, 0x70, 0xb0, 0x88, 0xed,
    0x63, 0x9f, 0x9d, 0x9b,
};
static const unsigned char kat150_nor_addin1[] = {
    0x5a, 0xb2, 0xe0, 0x06, 0x7c, 0x3b, 0x38, 0x4e, 0x55, 0xa7, 0x84, 0x92,
    0xf0, 0xf6, 0xed, 0x44,
};
static const unsigned char kat150_nor_retbytes[] = {
    0xca, 0x09, 0x5d, 0xa3, 0x9d, 0x9c, 0x21, 0xd7, 0xda, 0x07, 0x3d, 0x9c,
    0x95, 0xd2, 0xe4, 0x15, 0x50, 0x3b, 0x33, 0xc3, 0x27, 0xd7, 0x39, 0xf1,
    0x83, 0x8b, 0xbe, 0xa4, 0xfc, 0x6f, 0x02, 0x54, 0xfd, 0xaf, 0x8e, 0xf6,
    0x15, 0x2e, 0x92, 0x63, 0xf4, 0x6b, 0x86, 0x4f, 0x39, 0xc7, 0x10, 0x4d,
    0x1d, 0x33, 0x7d, 0x99, 0xfe, 0xe5, 0x88, 0x06, 0x11, 0x52, 0xe6, 0x23,
    0xd7, 0xe0, 0x0a, 0x27, 0xe0, 0x3b, 0x5d, 0x16, 0xfe, 0x6e, 0x54, 0x34,
    0x53, 0xa3, 0x1d, 0x4d, 0xaf, 0xed, 0xa3, 0xb5,
};
static const struct drbg_kat_no_reseed kat150_nor_t = {
    14, kat150_nor_entropyin, kat150_nor_nonce, kat150_nor_persstr,
    kat150_nor_addin0, kat150_nor_addin1, kat150_nor_retbytes
};
static const struct drbg_kat kat150_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat150_nor_t
};

static const unsigned char kat151_nor_entropyin[] = {
    0x91, 0xd9, 0x9c, 0x51, 0x1c, 0x2f, 0x19, 0x67, 0xd3, 0x1d, 0x20, 0xb3,
    0x13, 0xe5, 0x38, 0x2d,
};
static const unsigned char kat151_nor_nonce[] = {
    0x1e, 0xaf, 0x7f, 0x1d, 0x0a, 0xa3, 0x6a, 0x19,
};
static const unsigned char kat151_nor_persstr[] = {
    0x31, 0xe6, 0xb0, 0x4d, 0x73, 0xb1, 0xa9, 0x70, 0xfd, 0x3d, 0xcd, 0x8f,
    0x08, 0x9e, 0x44, 0x02,
};
static const unsigned char kat151_nor_addin0[] = {0};
static const unsigned char kat151_nor_addin1[] = {0};
static const unsigned char kat151_nor_retbytes[] = {
    0x1c, 0x95, 0xfb, 0x98, 0x5c, 0xf1, 0x7c, 0xb9, 0xcf, 0x0b, 0xcd, 0x53,
    0x95, 0x99, 0x72, 0xc8, 0xfd, 0x4b, 0xbc, 0x72, 0x5b, 0x2e, 0xcc, 0x8e,
    0xe8, 0x43, 0xd5, 0x5e, 0x49, 0x4b, 0xc7, 0x68, 0x4d, 0x17, 0x65, 0x93,
    0x1c, 0x17, 0x3a, 0x83, 0x8d, 0xc7, 0xb4, 0x34, 0x4c, 0xdc, 0x14, 0x58,
    0x64, 0x09, 0x06, 0x6d, 0x4f, 0x1d, 0x54, 0xac, 0x21, 0x4e, 0xcb, 0xe5,
    0xbd, 0x13, 0x9c, 0x65, 0xab, 0xe1, 0x99, 0x52, 0x5f, 0xa8, 0xff, 0xbe,
    0xf5, 0x93, 0x04, 0xe4, 0x7b, 0xaa, 0xc9, 0xa9,
};
static const struct drbg_kat_no_reseed kat151_nor_t = {
    0, kat151_nor_entropyin, kat151_nor_nonce, kat151_nor_persstr,
    kat151_nor_addin0, kat151_nor_addin1, kat151_nor_retbytes
};
static const struct drbg_kat kat151_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat151_nor_t
};

static const unsigned char kat152_nor_entropyin[] = {
    0xe6, 0x77, 0x28, 0x1e, 0x87, 0x1c, 0x28, 0xd6, 0x31, 0xb2, 0x36, 0x11,
    0x07, 0x24, 0x0c, 0x22,
};
static const unsigned char kat152_nor_nonce[] = {
    0x87, 0x46, 0x89, 0x67, 0xe4, 0x55, 0xd2, 0xda,
};
static const unsigned char kat152_nor_persstr[] = {
    0x11, 0x4b, 0xa9, 0x2f, 0x79, 0x45, 0x18, 0x01, 0x5c, 0x06, 0x7f, 0x7c,
    0x35, 0x8c, 0x40, 0x41,
};
static const unsigned char kat152_nor_addin0[] = {0};
static const unsigned char kat152_nor_addin1[] = {0};
static const unsigned char kat152_nor_retbytes[] = {
    0x59, 0x58, 0x4f, 0xf0, 0x0c, 0x44, 0xa0, 0xdb, 0xd1, 0x5c, 0x1f, 0x35,
    0xe0, 0x50, 0xc1, 0x84, 0xe8, 0xd5, 0x87, 0xe8, 0x85, 0x9c, 0xb0, 0x61,
    0xc8, 0x40, 0x9c, 0x45, 0x4b, 0x95, 0x39, 0x24, 0xf5, 0xe6, 0xee, 0x1f,
    0xee, 0x4d, 0x76, 0x34, 0x62, 0xdd, 0x6a, 0x29, 0xa6, 0xcb, 0xe4, 0xa3,
    0x73, 0x97, 0x4c, 0x01, 0xd6, 0x88, 0xee, 0x57, 0x28, 0xcf, 0x71, 0x2b,
    0xcd, 0xc8, 0x2c, 0xb0, 0xad, 0x15, 0x41, 0x92, 0x29, 0xb8, 0xcd, 0x11,
    0x4a, 0xc7, 0x85, 0x1e, 0x37, 0xa3, 0xd7, 0xb2,
};
static const struct drbg_kat_no_reseed kat152_nor_t = {
    1, kat152_nor_entropyin, kat152_nor_nonce, kat152_nor_persstr,
    kat152_nor_addin0, kat152_nor_addin1, kat152_nor_retbytes
};
static const struct drbg_kat kat152_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat152_nor_t
};

static const unsigned char kat153_nor_entropyin[] = {
    0xbd, 0x48, 0x31, 0xa8, 0x52, 0x79, 0xfd, 0x76, 0x7c, 0xaf, 0xf4, 0x4b,
    0xa2, 0x60, 0x53, 0xae,
};
static const unsigned char kat153_nor_nonce[] = {
    0x9e, 0x56, 0xa8, 0x08, 0x71, 0x18, 0x27, 0xf6,
};
static const unsigned char kat153_nor_persstr[] = {
    0x69, 0xe0, 0xfe, 0x7b, 0x80, 0x4c, 0x65, 0xbb, 0x8c, 0x7b, 0xa8, 0x9a,
    0x8b, 0x4c, 0x7c, 0x6a,
};
static const unsigned char kat153_nor_addin0[] = {0};
static const unsigned char kat153_nor_addin1[] = {0};
static const unsigned char kat153_nor_retbytes[] = {
    0x7d, 0x1f, 0xe8, 0xc9, 0xde, 0x79, 0x61, 0x1d, 0x57, 0xb5, 0x12, 0x97,
    0xcf, 0x1b, 0x23, 0x04, 0x59, 0x15, 0x04, 0xa0, 0x46, 0x94, 0xe6, 0x0e,
    0x21, 0xfd, 0xd7, 0x0f, 0x3d, 0x78, 0x2d, 0x17, 0xae, 0xa3, 0x51, 0x73,
    0xef, 0xa3, 0x0e, 0xf2, 0xf2, 0x35, 0x89, 0x4d, 0x69, 0x6f, 0x1a, 0x15,
    0x0b, 0xfe, 0x03, 0x96, 0x27, 0x1f, 0xd8, 0xcb, 0xa0, 0x62, 0x82, 0x35,
    0x80, 0xa9, 0xe1, 0xde, 0x8e, 0x32, 0xd9, 0x62, 0xbd, 0xcb, 0x4b, 0x4f,
    0x47, 0xb9, 0x42, 0xfc, 0x39, 0x84, 0x9a, 0xbf,
};
static const struct drbg_kat_no_reseed kat153_nor_t = {
    2, kat153_nor_entropyin, kat153_nor_nonce, kat153_nor_persstr,
    kat153_nor_addin0, kat153_nor_addin1, kat153_nor_retbytes
};
static const struct drbg_kat kat153_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat153_nor_t
};

static const unsigned char kat154_nor_entropyin[] = {
    0x46, 0xbd, 0xab, 0x81, 0x1f, 0x76, 0xb7, 0x52, 0x55, 0x1a, 0x2b, 0x11,
    0xb7, 0x08, 0x96, 0x3e,
};
static const unsigned char kat154_nor_nonce[] = {
    0x8b, 0xe4, 0xfc, 0x15, 0x9f, 0xdb, 0x75, 0x15,
};
static const unsigned char kat154_nor_persstr[] = {
    0xbc, 0x4f, 0x4d, 0x95, 0x92, 0x7b, 0x1a, 0xa6, 0x66, 0x79, 0x1c, 0x4a,
    0xa8, 0x18, 0x70, 0x0f,
};
static const unsigned char kat154_nor_addin0[] = {0};
static const unsigned char kat154_nor_addin1[] = {0};
static const unsigned char kat154_nor_retbytes[] = {
    0xc5, 0x35, 0x85, 0x9c, 0xef, 0x07, 0x49, 0x16, 0xf7, 0xa3, 0x78, 0x55,
    0x37, 0x28, 0x5b, 0xe3, 0x85, 0x89, 0x38, 0x22, 0xa9, 0x53, 0xfd, 0x41,
    0xab, 0xa6, 0x00, 0xf5, 0x6d, 0x80, 0xe6, 0xef, 0xe2, 0x95, 0x2f, 0x92,
    0xec, 0xe5, 0xcb, 0xec, 0x15, 0xb2, 0xb0, 0x3a, 0x22, 0x90, 0x80, 0xde,
    0x98, 0xc6, 0xeb, 0xda, 0x74, 0x1f, 0x19, 0xd3, 0x55, 0x70, 0x15, 0xa9,
    0x0e, 0xe0, 0xb2, 0xa0, 0xde, 0x9c, 0x0e, 0xb6, 0x4c, 0x8b, 0x8a, 0xf6,
    0xdc, 0xcf, 0x8f, 0x74, 0xe7, 0xca, 0x1c, 0x1f,
};
static const struct drbg_kat_no_reseed kat154_nor_t = {
    3, kat154_nor_entropyin, kat154_nor_nonce, kat154_nor_persstr,
    kat154_nor_addin0, kat154_nor_addin1, kat154_nor_retbytes
};
static const struct drbg_kat kat154_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat154_nor_t
};

static const unsigned char kat155_nor_entropyin[] = {
    0x67, 0x01, 0x13, 0x8a, 0x46, 0xb1, 0x48, 0xa2, 0x68, 0x88, 0x79, 0x98,
    0x9a, 0xb4, 0xda, 0x96,
};
static const unsigned char kat155_nor_nonce[] = {
    0x1b, 0xbd, 0x5c, 0x7f, 0x09, 0x55, 0x34, 0xcc,
};
static const unsigned char kat155_nor_persstr[] = {
    0xce, 0xd0, 0x4a, 0x25, 0xa8, 0xbd, 0xed, 0xbd, 0xbb, 0x28, 0x53, 0x43,
    0x76, 0xbb, 0x24, 0x14,
};
static const unsigned char kat155_nor_addin0[] = {0};
static const unsigned char kat155_nor_addin1[] = {0};
static const unsigned char kat155_nor_retbytes[] = {
    0xb4, 0xc2, 0xf0, 0xe9, 0x89, 0xbd, 0x85, 0x8e, 0xd1, 0x60, 0x37, 0xf8,
    0x34, 0x45, 0x57, 0xb0, 0xcb, 0x35, 0xd0, 0x44, 0x19, 0x1d, 0x68, 0x0c,
    0xb1, 0xaf, 0xcd, 0x2d, 0x85, 0x63, 0x68, 0x6a, 0x92, 0xfd, 0xff, 0x5e,
    0x13, 0x31, 0xfd, 0x3f, 0xc8, 0x7d, 0x14, 0xf3, 0xba, 0x10, 0xeb, 0x14,
    0xee, 0x6b, 0x16, 0xd6, 0x48, 0x18, 0x13, 0x92, 0x33, 0x5c, 0x0e, 0xe4,
    0xc2, 0xab, 0x27, 0x58, 0x81, 0xbb, 0xa3, 0x6c, 0x55, 0x3d, 0x34, 0x81,
    0x0b, 0x8c, 0x68, 0x9a, 0xad, 0x01, 0x73, 0x43,
};
static const struct drbg_kat_no_reseed kat155_nor_t = {
    4, kat155_nor_entropyin, kat155_nor_nonce, kat155_nor_persstr,
    kat155_nor_addin0, kat155_nor_addin1, kat155_nor_retbytes
};
static const struct drbg_kat kat155_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat155_nor_t
};

static const unsigned char kat156_nor_entropyin[] = {
    0x9d, 0xf8, 0x20, 0x29, 0x7b, 0xc0, 0x01, 0x6d, 0x24, 0x6a, 0x93, 0xcb,
    0x64, 0x19, 0x20, 0x73,
};
static const unsigned char kat156_nor_nonce[] = {
    0xf2, 0xdc, 0xff, 0x78, 0xc9, 0x0a, 0x15, 0xa7,
};
static const unsigned char kat156_nor_persstr[] = {
    0x70, 0x59, 0x43, 0x3f, 0x58, 0x64, 0x6f, 0x57, 0x94, 0xcc, 0xf1, 0x71,
    0xca, 0x18, 0x8a, 0x95,
};
static const unsigned char kat156_nor_addin0[] = {0};
static const unsigned char kat156_nor_addin1[] = {0};
static const unsigned char kat156_nor_retbytes[] = {
    0x06, 0xa5, 0xc6, 0x1c, 0x66, 0x45, 0x03, 0x46, 0x9a, 0xf0, 0xff, 0x67,
    0x14, 0xae, 0x87, 0xe8, 0xa7, 0x69, 0x96, 0xc3, 0x5d, 0x80, 0xd4, 0x5b,
    0x02, 0x08, 0xbb, 0x26, 0x73, 0x5b, 0x9e, 0x46, 0x30, 0xea, 0xcf, 0x50,
    0xcf, 0x62, 0xff, 0x35, 0x30, 0x21, 0x4e, 0x8d, 0xed, 0xbe, 0x42, 0x40,
    0x84, 0x86, 0x38, 0xeb, 0x41, 0x78, 0x3f, 0x8d, 0x1d, 0x3e, 0xfa, 0x63,
    0x99, 0xbc, 0x1d, 0x95, 0x54, 0xf5, 0xb5, 0xa3, 0x86, 0x08, 0x24, 0x98,
    0xa3, 0x3f, 0x8e, 0xa4, 0xfa, 0x53, 0x09, 0xf1,
};
static const struct drbg_kat_no_reseed kat156_nor_t = {
    5, kat156_nor_entropyin, kat156_nor_nonce, kat156_nor_persstr,
    kat156_nor_addin0, kat156_nor_addin1, kat156_nor_retbytes
};
static const struct drbg_kat kat156_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat156_nor_t
};

static const unsigned char kat157_nor_entropyin[] = {
    0x24, 0x4c, 0xc3, 0xba, 0xa0, 0x00, 0xaf, 0xa7, 0xd5, 0x47, 0x51, 0x17,
    0x70, 0x16, 0x5f, 0x70,
};
static const unsigned char kat157_nor_nonce[] = {
    0xc5, 0x83, 0x94, 0x2d, 0x88, 0x88, 0x28, 0xff,
};
static const unsigned char kat157_nor_persstr[] = {
    0x53, 0xab, 0x8f, 0x3e, 0x39, 0x26, 0x96, 0xff, 0xb6, 0xb0, 0x6f, 0x45,
    0xaf, 0x12, 0x56, 0x56,
};
static const unsigned char kat157_nor_addin0[] = {0};
static const unsigned char kat157_nor_addin1[] = {0};
static const unsigned char kat157_nor_retbytes[] = {
    0x67, 0xd1, 0x73, 0x5a, 0x4c, 0xd3, 0x7b, 0xe6, 0x7e, 0x68, 0x2e, 0xe5,
    0xa3, 0xc9, 0x9a, 0xba, 0x13, 0x8b, 0x74, 0x88, 0x7b, 0x90, 0xc8, 0x29,
    0x1e, 0x9e, 0x4e, 0x71, 0x56, 0xc7, 0x86, 0xef, 0x56, 0x26, 0xec, 0xb2,
    0x42, 0x0e, 0x47, 0xd6, 0x4b, 0x6a, 0x04, 0x4f, 0xac, 0x95, 0x1d, 0xdb,
    0x30, 0x2d, 0x18, 0x4c, 0x5a, 0xe0, 0x1f, 0x03, 0xc4, 0x2e, 0xcf, 0x63,
    0x3a, 0x1e, 0x53, 0x9f, 0xcd, 0xb4, 0xf9, 0x26, 0xa4, 0x72, 0xc4, 0x8d,
    0xd8, 0xce, 0x30, 0x91, 0x4c, 0xc2, 0x29, 0xd4,
};
static const struct drbg_kat_no_reseed kat157_nor_t = {
    6, kat157_nor_entropyin, kat157_nor_nonce, kat157_nor_persstr,
    kat157_nor_addin0, kat157_nor_addin1, kat157_nor_retbytes
};
static const struct drbg_kat kat157_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat157_nor_t
};

static const unsigned char kat158_nor_entropyin[] = {
    0xd8, 0x99, 0x7d, 0x36, 0xed, 0x9d, 0x8d, 0x50, 0xa1, 0x90, 0xed, 0x4f,
    0xf8, 0x16, 0x0a, 0xcf,
};
static const unsigned char kat158_nor_nonce[] = {
    0xd9, 0xb4, 0xad, 0xd5, 0xce, 0x5a, 0x28, 0xb9,
};
static const unsigned char kat158_nor_persstr[] = {
    0xfd, 0x9f, 0x0c, 0xfd, 0xb8, 0x0e, 0xe5, 0x81, 0xe8, 0x36, 0x76, 0xcf,
    0xc4, 0xfa, 0x09, 0xd7,
};
static const unsigned char kat158_nor_addin0[] = {0};
static const unsigned char kat158_nor_addin1[] = {0};
static const unsigned char kat158_nor_retbytes[] = {
    0x68, 0x50, 0xdb, 0x84, 0xa8, 0x68, 0xeb, 0x1f, 0x0d, 0xe7, 0x5a, 0x38,
    0x81, 0xa4, 0xec, 0x79, 0xa1, 0x15, 0x6f, 0x30, 0xb8, 0x61, 0x8c, 0xa1,
    0x6e, 0x55, 0x17, 0x5a, 0xb6, 0xf8, 0x40, 0x93, 0x2d, 0x21, 0xa3, 0x40,
    0xd9, 0x46, 0x5f, 0xa1, 0x2a, 0x58, 0x7c, 0xff, 0x1f, 0x87, 0x86, 0x27,
    0xaa, 0xd2, 0xe9, 0xc8, 0x27, 0x84, 0xf6, 0x50, 0xdf, 0xca, 0xbd, 0x12,
    0xb3, 0x6c, 0xdb, 0x1b, 0x10, 0xa2, 0xd1, 0x75, 0x1d, 0x97, 0x56, 0xb5,
    0xe2, 0x89, 0xe8, 0xaa, 0x7d, 0x23, 0x02, 0xeb,
};
static const struct drbg_kat_no_reseed kat158_nor_t = {
    7, kat158_nor_entropyin, kat158_nor_nonce, kat158_nor_persstr,
    kat158_nor_addin0, kat158_nor_addin1, kat158_nor_retbytes
};
static const struct drbg_kat kat158_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat158_nor_t
};

static const unsigned char kat159_nor_entropyin[] = {
    0x02, 0x85, 0xda, 0x43, 0x1e, 0xd1, 0xc6, 0x18, 0x07, 0x8e, 0x1b, 0x4e,
    0xd9, 0xef, 0xfc, 0xa4,
};
static const unsigned char kat159_nor_nonce[] = {
    0xf1, 0x0a, 0x22, 0xcf, 0x4a, 0xdb, 0x6c, 0x3f,
};
static const unsigned char kat159_nor_persstr[] = {
    0x07, 0x3b, 0x53, 0x7b, 0x35, 0xc6, 0xa4, 0x97, 0xe4, 0x78, 0x51, 0x6e,
    0xd7, 0x97, 0x83, 0xee,
};
static const unsigned char kat159_nor_addin0[] = {0};
static const unsigned char kat159_nor_addin1[] = {0};
static const unsigned char kat159_nor_retbytes[] = {
    0x12, 0xf8, 0xf3, 0x57, 0x63, 0x3b, 0x5c, 0x76, 0x96, 0x65, 0xc3, 0x33,
    0xb9, 0x75, 0xa8, 0x1e, 0xd9, 0x7f, 0x94, 0x0f, 0x50, 0x09, 0x4b, 0xda,
    0xb1, 0xe2, 0x78, 0x12, 0x63, 0x56, 0xf9, 0xf9, 0xda, 0xb2, 0x1a, 0x2f,
    0xea, 0x61, 0xa0, 0xf4, 0x91, 0x14, 0x39, 0x78, 0x89, 0x42, 0x38, 0xc0,
    0x4c, 0xf4, 0x95, 0x6a, 0xe8, 0xff, 0xab, 0xad, 0xb6, 0xab, 0xf0, 0xac,
    0x4e, 0xfe, 0xf7, 0x6d, 0x5a, 0x58, 0x1e, 0xd7, 0x97, 0x94, 0x8c, 0x66,
    0x72, 0x00, 0x8a, 0x89, 0x73, 0xe9, 0x75, 0x67,
};
static const struct drbg_kat_no_reseed kat159_nor_t = {
    8, kat159_nor_entropyin, kat159_nor_nonce, kat159_nor_persstr,
    kat159_nor_addin0, kat159_nor_addin1, kat159_nor_retbytes
};
static const struct drbg_kat kat159_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat159_nor_t
};

static const unsigned char kat160_nor_entropyin[] = {
    0xd6, 0xb7, 0x35, 0xfd, 0xa7, 0x41, 0x9f, 0x43, 0x23, 0x1d, 0x90, 0x51,
    0xd7, 0x35, 0x41, 0xf6,
};
static const unsigned char kat160_nor_nonce[] = {
    0x7e, 0x09, 0x5b, 0x38, 0xbe, 0x8e, 0xc5, 0x49,
};
static const unsigned char kat160_nor_persstr[] = {
    0x27, 0x87, 0x2a, 0xf7, 0x30, 0xf3, 0xab, 0x90, 0xa3, 0xa5, 0x20, 0xe3,
    0x24, 0xc9, 0x08, 0xa0,
};
static const unsigned char kat160_nor_addin0[] = {0};
static const unsigned char kat160_nor_addin1[] = {0};
static const unsigned char kat160_nor_retbytes[] = {
    0xf6, 0x84, 0x24, 0x8e, 0x03, 0x9a, 0x99, 0x19, 0x77, 0x3e, 0x17, 0x80,
    0x86, 0xde, 0xb5, 0x58, 0x10, 0x52, 0xee, 0x66, 0xec, 0x06, 0x23, 0x6a,
    0x49, 0x72, 0xe4, 0x45, 0x9a, 0xe1, 0x6f, 0x7d, 0x5d, 0xe6, 0x6c, 0x74,
    0x23, 0x18, 0x9f, 0x12, 0xe1, 0xc9, 0xa1, 0x05, 0xd1, 0xfb, 0x15, 0x00,
    0x01, 0x0e, 0xb2, 0x80, 0xf3, 0x46, 0xb4, 0x5d, 0x96, 0xac, 0x90, 0x38,
    0x9f, 0xf5, 0x17, 0x4c, 0xc0, 0x09, 0x11, 0xec, 0x3a, 0x89, 0x72, 0xed,
    0x02, 0x8f, 0x74, 0xeb, 0xa8, 0x34, 0x77, 0x32,
};
static const struct drbg_kat_no_reseed kat160_nor_t = {
    9, kat160_nor_entropyin, kat160_nor_nonce, kat160_nor_persstr,
    kat160_nor_addin0, kat160_nor_addin1, kat160_nor_retbytes
};
static const struct drbg_kat kat160_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat160_nor_t
};

static const unsigned char kat161_nor_entropyin[] = {
    0x87, 0x06, 0xde, 0xce, 0x8a, 0x59, 0x5f, 0x0c, 0x48, 0x85, 0xc1, 0x46,
    0x95, 0xff, 0x29, 0xd9,
};
static const unsigned char kat161_nor_nonce[] = {
    0x96, 0x58, 0x7c, 0x6f, 0x34, 0xd2, 0x9e, 0x6f,
};
static const unsigned char kat161_nor_persstr[] = {
    0xbd, 0x77, 0x0a, 0x8f, 0x31, 0xf7, 0x84, 0x82, 0xb2, 0xca, 0x2e, 0x1e,
    0x4b, 0x24, 0xb3, 0x3b,
};
static const unsigned char kat161_nor_addin0[] = {0};
static const unsigned char kat161_nor_addin1[] = {0};
static const unsigned char kat161_nor_retbytes[] = {
    0x4d, 0x30, 0x52, 0x3f, 0xc5, 0x74, 0xd6, 0x35, 0x2b, 0xad, 0xa3, 0xd2,
    0x56, 0x71, 0x78, 0xc7, 0x02, 0x80, 0x5a, 0xde, 0xef, 0x69, 0xbd, 0x68,
    0x20, 0xed, 0x97, 0x14, 0x6b, 0xb3, 0x0d, 0x09, 0xc1, 0x6f, 0xff, 0x50,
    0x20, 0x89, 0xa9, 0x0f, 0x9c, 0x99, 0x36, 0x86, 0xc7, 0x2c, 0x40, 0x18,
    0x25, 0xa8, 0x3f, 0x09, 0x4e, 0x81, 0x52, 0xaf, 0x22, 0x1d, 0x71, 0xd5,
    0x5e, 0x68, 0x82, 0xa1, 0x0f, 0x4d, 0x17, 0xaa, 0x08, 0xdf, 0xbe, 0x54,
    0xb2, 0x63, 0xd2, 0xba, 0xae, 0x0b, 0x83, 0xd4,
};
static const struct drbg_kat_no_reseed kat161_nor_t = {
    10, kat161_nor_entropyin, kat161_nor_nonce, kat161_nor_persstr,
    kat161_nor_addin0, kat161_nor_addin1, kat161_nor_retbytes
};
static const struct drbg_kat kat161_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat161_nor_t
};

static const unsigned char kat162_nor_entropyin[] = {
    0x32, 0x52, 0x6f, 0x30, 0xf3, 0x61, 0xbc, 0xd4, 0xdf, 0x58, 0x02, 0x72,
    0x44, 0x42, 0xb0, 0x1d,
};
static const unsigned char kat162_nor_nonce[] = {
    0x19, 0x29, 0xdf, 0xb1, 0x7d, 0x50, 0x04, 0x42,
};
static const unsigned char kat162_nor_persstr[] = {
    0xf9, 0x45, 0xc1, 0xe2, 0xea, 0xbc, 0x43, 0x51, 0x93, 0x13, 0x46, 0x28,
    0xbc, 0x8c, 0x03, 0xe1,
};
static const unsigned char kat162_nor_addin0[] = {0};
static const unsigned char kat162_nor_addin1[] = {0};
static const unsigned char kat162_nor_retbytes[] = {
    0xe8, 0x96, 0x26, 0x54, 0x57, 0xd9, 0x25, 0x1f, 0x40, 0x62, 0xa1, 0x4f,
    0x08, 0xd1, 0x6a, 0xe5, 0xc4, 0xff, 0x37, 0x0b, 0x46, 0x75, 0xf2, 0x0f,
    0xed, 0x95, 0x11, 0xea, 0xd9, 0x99, 0xa5, 0xe4, 0xab, 0x70, 0x94, 0xba,
    0x66, 0x78, 0xe6, 0x9a, 0xbc, 0x7a, 0x95, 0x91, 0x66, 0x76, 0x28, 0x0c,
    0x7f, 0x78, 0xcf, 0xbe, 0xd7, 0xdf, 0xce, 0xa4, 0x64, 0x93, 0xf5, 0x76,
    0x34, 0xe9, 0x3d, 0xd1, 0x87, 0x00, 0x60, 0x7e, 0x1e, 0x19, 0xb7, 0x8a,
    0x57, 0xd1, 0xe4, 0x7b, 0xa0, 0x79, 0xe4, 0x05,
};
static const struct drbg_kat_no_reseed kat162_nor_t = {
    11, kat162_nor_entropyin, kat162_nor_nonce, kat162_nor_persstr,
    kat162_nor_addin0, kat162_nor_addin1, kat162_nor_retbytes
};
static const struct drbg_kat kat162_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat162_nor_t
};

static const unsigned char kat163_nor_entropyin[] = {
    0x87, 0x4b, 0x0d, 0xd4, 0x2d, 0x1d, 0xdf, 0xb3, 0xd3, 0x83, 0x52, 0x71,
    0xf5, 0x73, 0x16, 0x29,
};
static const unsigned char kat163_nor_nonce[] = {
    0x87, 0xb6, 0xc8, 0x9e, 0xda, 0x28, 0x47, 0x8a,
};
static const unsigned char kat163_nor_persstr[] = {
    0x7d, 0x5b, 0x80, 0x24, 0x74, 0x11, 0xc0, 0x67, 0x79, 0x7d, 0xfc, 0x10,
    0xc1, 0x66, 0x87, 0x75,
};
static const unsigned char kat163_nor_addin0[] = {0};
static const unsigned char kat163_nor_addin1[] = {0};
static const unsigned char kat163_nor_retbytes[] = {
    0x69, 0x97, 0x23, 0x85, 0x9d, 0x25, 0xab, 0x75, 0x9a, 0x9b, 0x8b, 0x37,
    0x28, 0xa6, 0x7b, 0xf6, 0x39, 0xe0, 0x2b, 0x80, 0x5d, 0x5e, 0xa7, 0xb6,
    0xe6, 0xd6, 0x14, 0xec, 0x29, 0x72, 0x8b, 0x68, 0xd6, 0xd9, 0xed, 0x29,
    0xa9, 0x3a, 0xb1, 0xf1, 0xc6, 0x24, 0xd3, 0xf2, 0xd5, 0xca, 0x18, 0x26,
    0xcd, 0x09, 0xcc, 0x1c, 0x55, 0x55, 0x60, 0x45, 0x37, 0x56, 0x7f, 0xa6,
    0xfd, 0xb1, 0xf6, 0x62, 0x2d, 0x29, 0xf2, 0xd8, 0x1d, 0x1d, 0xd7, 0xc2,
    0x31, 0x79, 0xd7, 0x5c, 0x43, 0xd7, 0x2f, 0x06,
};
static const struct drbg_kat_no_reseed kat163_nor_t = {
    12, kat163_nor_entropyin, kat163_nor_nonce, kat163_nor_persstr,
    kat163_nor_addin0, kat163_nor_addin1, kat163_nor_retbytes
};
static const struct drbg_kat kat163_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat163_nor_t
};

static const unsigned char kat164_nor_entropyin[] = {
    0x51, 0x62, 0x63, 0x6e, 0x7b, 0x1a, 0x5c, 0xf9, 0xb1, 0xc1, 0x4d, 0x72,
    0x3b, 0xb3, 0xee, 0xb9,
};
static const unsigned char kat164_nor_nonce[] = {
    0x21, 0xe5, 0xb1, 0x77, 0xf1, 0x19, 0xc4, 0x77,
};
static const unsigned char kat164_nor_persstr[] = {
    0x65, 0x13, 0x31, 0x8d, 0x80, 0x21, 0x85, 0x46, 0x38, 0x5c, 0x2a, 0xfc,
    0x56, 0xe4, 0x11, 0x6d,
};
static const unsigned char kat164_nor_addin0[] = {0};
static const unsigned char kat164_nor_addin1[] = {0};
static const unsigned char kat164_nor_retbytes[] = {
    0xc2, 0x95, 0xe6, 0x51, 0x5a, 0xbf, 0x25, 0x98, 0x12, 0x27, 0x52, 0xb2,
    0x4b, 0x4b, 0xb1, 0x49, 0x3d, 0x82, 0x1e, 0x6b, 0xb1, 0x95, 0x61, 0xc2,
    0x69, 0x0d, 0xfb, 0xc0, 0x61, 0x19, 0xc8, 0x65, 0x39, 0xc0, 0xd2, 0x10,
    0x02, 0xd2, 0x44, 0x84, 0x74, 0x76, 0x2d, 0x60, 0x75, 0x1b, 0xa8, 0xa5,
    0x81, 0x65, 0xf6, 0x6f, 0xca, 0xf6, 0x34, 0x2d, 0xcd, 0x4e, 0x46, 0x83,
    0x63, 0x46, 0x2f, 0x94, 0x14, 0x9d, 0xa6, 0x63, 0x6c, 0x54, 0x8f, 0xe5,
    0xb9, 0xf2, 0xaa, 0xd1, 0x38, 0xa5, 0x4a, 0x43,
};
static const struct drbg_kat_no_reseed kat164_nor_t = {
    13, kat164_nor_entropyin, kat164_nor_nonce, kat164_nor_persstr,
    kat164_nor_addin0, kat164_nor_addin1, kat164_nor_retbytes
};
static const struct drbg_kat kat164_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat164_nor_t
};

static const unsigned char kat165_nor_entropyin[] = {
    0xf8, 0xa1, 0x56, 0x6d, 0x14, 0xff, 0xf9, 0x2e, 0x8d, 0xe3, 0x79, 0xd1,
    0x68, 0xff, 0x3c, 0xfa,
};
static const unsigned char kat165_nor_nonce[] = {
    0x4e, 0x83, 0x8a, 0x12, 0x4e, 0x4b, 0x53, 0xdf,
};
static const unsigned char kat165_nor_persstr[] = {
    0x16, 0x3e, 0x39, 0x3b, 0x29, 0x0a, 0x4d, 0x39, 0x0a, 0xb0, 0xbe, 0xb3,
    0x92, 0xf5, 0x2d, 0x26,
};
static const unsigned char kat165_nor_addin0[] = {0};
static const unsigned char kat165_nor_addin1[] = {0};
static const unsigned char kat165_nor_retbytes[] = {
    0x76, 0x23, 0x4a, 0xfc, 0x29, 0x6e, 0xa3, 0x6a, 0x44, 0x25, 0x4f, 0x99,
    0x9a, 0xc3, 0x1f, 0xca, 0x25, 0x8a, 0x24, 0x42, 0x7c, 0xf4, 0xbf, 0xe2,
    0xc5, 0x44, 0x95, 0xfc, 0x41, 0x47, 0x8e, 0xc4, 0xa0, 0x0b, 0x54, 0x06,
    0x59, 0xb3, 0xb9, 0x46, 0x1c, 0xc6, 0x18, 0x8b, 0xc1, 0xf5, 0x7c, 0x19,
    0xae, 0x41, 0x4b, 0xd1, 0x8a, 0xa8, 0x1e, 0xca, 0x7b, 0x9d, 0x76, 0x5a,
    0x78, 0x4f, 0x0e, 0xf2, 0x43, 0x35, 0xe4, 0x6c, 0x2c, 0x77, 0xb8, 0xdc,
    0x91, 0x5f, 0x5d, 0x12, 0xc2, 0x6b, 0xc6, 0x53,
};
static const struct drbg_kat_no_reseed kat165_nor_t = {
    14, kat165_nor_entropyin, kat165_nor_nonce, kat165_nor_persstr,
    kat165_nor_addin0, kat165_nor_addin1, kat165_nor_retbytes
};
static const struct drbg_kat kat165_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat165_nor_t
};

static const unsigned char kat166_nor_entropyin[] = {
    0x5e, 0x05, 0x66, 0xb6, 0x60, 0x63, 0x1b, 0x94, 0xbf, 0xa0, 0x96, 0xb3,
    0x22, 0x5d, 0x59, 0xe3,
};
static const unsigned char kat166_nor_nonce[] = {
    0x81, 0x2b, 0xba, 0x14, 0x29, 0x37, 0x56, 0x2d,
};
static const unsigned char kat166_nor_persstr[] = {
    0x08, 0xe8, 0x3a, 0x8e, 0x3d, 0x50, 0xe5, 0x0d, 0xb2, 0xb0, 0xbe, 0x92,
    0xf2, 0x36, 0xf7, 0x86,
};
static const unsigned char kat166_nor_addin0[] = {
    0x49, 0x6b, 0x30, 0xce, 0xb7, 0xef, 0x9e, 0xbe, 0x9d, 0x44, 0x9d, 0x12,
    0x4d, 0x22, 0x02, 0xa6,
};
static const unsigned char kat166_nor_addin1[] = {
    0x2d, 0xf0, 0xb0, 0x3d, 0xd0, 0xec, 0xec, 0xd2, 0x05, 0x52, 0xfb, 0xfd,
    0x33, 0x18, 0x8d, 0x4f,
};
static const unsigned char kat166_nor_retbytes[] = {
    0x48, 0x1e, 0x35, 0x58, 0xc5, 0x0c, 0xb6, 0x69, 0x16, 0x44, 0x48, 0x00,
    0x1d, 0xaa, 0x1d, 0x56, 0x3c, 0x52, 0xaf, 0xe1, 0x4a, 0x92, 0xb5, 0x8d,
    0xbf, 0xa6, 0x12, 0xf0, 0xec, 0xd9, 0x4e, 0x2e, 0x64, 0x23, 0x40, 0xa3,
    0x90, 0x70, 0x28, 0xfe, 0xcc, 0x21, 0x40, 0x87, 0xab, 0xef, 0xe3, 0xb6,
    0x5c, 0x8f, 0xc2, 0xe0, 0x25, 0x5f, 0x52, 0x0c, 0x85, 0xbf, 0x62, 0xf0,
    0x2a, 0x7e, 0x18, 0x79, 0x69, 0x50, 0xf4, 0xfd, 0xd4, 0xa4, 0x08, 0xbe,
    0xc6, 0x92, 0x4e, 0xa5, 0x02, 0x36, 0x05, 0x90,
};
static const struct drbg_kat_no_reseed kat166_nor_t = {
    0, kat166_nor_entropyin, kat166_nor_nonce, kat166_nor_persstr,
    kat166_nor_addin0, kat166_nor_addin1, kat166_nor_retbytes
};
static const struct drbg_kat kat166_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat166_nor_t
};

static const unsigned char kat167_nor_entropyin[] = {
    0xf9, 0xbd, 0x51, 0x0f, 0x8a, 0xb8, 0x9e, 0xe8, 0x45, 0x10, 0x69, 0xf3,
    0xf9, 0xc0, 0x1c, 0xb0,
};
static const unsigned char kat167_nor_nonce[] = {
    0xa4, 0x22, 0xdf, 0xa0, 0xb0, 0x62, 0x1e, 0xa0,
};
static const unsigned char kat167_nor_persstr[] = {
    0x94, 0xb4, 0xb3, 0xcf, 0x14, 0x53, 0x45, 0x7f, 0xd5, 0x8e, 0xdc, 0xf4,
    0xae, 0xaf, 0x4c, 0x4c,
};
static const unsigned char kat167_nor_addin0[] = {
    0x30, 0x66, 0xbd, 0xe4, 0xe2, 0x6c, 0x14, 0xda, 0x64, 0x9f, 0xc0, 0xcb,
    0x71, 0x4b, 0x7e, 0x20,
};
static const unsigned char kat167_nor_addin1[] = {
    0x40, 0x54, 0x4e, 0x97, 0x2e, 0x9a, 0x7b, 0xc2, 0x27, 0xc8, 0x7f, 0xeb,
    0x19, 0x3a, 0x85, 0xeb,
};
static const unsigned char kat167_nor_retbytes[] = {
    0x44, 0xf9, 0x71, 0x34, 0x7a, 0xbb, 0x13, 0xb4, 0x50, 0x30, 0xef, 0x60,
    0x03, 0xac, 0x15, 0x4e, 0x07, 0x6e, 0x6e, 0xf4, 0x9e, 0xc5, 0x84, 0x9d,
    0x80, 0xf0, 0x92, 0x65, 0x07, 0x01, 0x55, 0x9c, 0x41, 0xcf, 0x6e, 0xeb,
    0x02, 0xe8, 0xd8, 0xe3, 0x19, 0xa5, 0xb7, 0x56, 0x95, 0x01, 0x4d, 0xb8,
    0x95, 0xde, 0x56, 0xdb, 0x11, 0x21, 0x79, 0x8f, 0x8e, 0x68, 0x1e, 0x6f,
    0x4a, 0x3e, 0xc6, 0x6e, 0x85, 0x3f, 0x01, 0x3b, 0x01, 0x78, 0x59, 0xd1,
    0xc7, 0x6d, 0x34, 0x8c, 0x0e, 0xe7, 0x26, 0x65,
};
static const struct drbg_kat_no_reseed kat167_nor_t = {
    1, kat167_nor_entropyin, kat167_nor_nonce, kat167_nor_persstr,
    kat167_nor_addin0, kat167_nor_addin1, kat167_nor_retbytes
};
static const struct drbg_kat kat167_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat167_nor_t
};

static const unsigned char kat168_nor_entropyin[] = {
    0x10, 0x5d, 0x33, 0xd2, 0x36, 0x2b, 0x87, 0xad, 0x3e, 0xe5, 0x1f, 0xd7,
    0xac, 0x74, 0xb7, 0x56,
};
static const unsigned char kat168_nor_nonce[] = {
    0xcf, 0xa2, 0x0f, 0x0e, 0x56, 0x6d, 0xc5, 0xc7,
};
static const unsigned char kat168_nor_persstr[] = {
    0x76, 0x4c, 0x28, 0xee, 0x81, 0x33, 0x6a, 0xed, 0xca, 0x87, 0x2c, 0xe2,
    0x66, 0xd5, 0x63, 0x13,
};
static const unsigned char kat168_nor_addin0[] = {
    0x12, 0xcc, 0xa7, 0x0d, 0xcf, 0x8b, 0xee, 0x7f, 0xcb, 0x19, 0x89, 0x49,
    0x83, 0x9b, 0x14, 0x40,
};
static const unsigned char kat168_nor_addin1[] = {
    0x11, 0x4b, 0xc1, 0x7e, 0x43, 0x91, 0xcc, 0x4f, 0xe8, 0xcd, 0x45, 0x15,
    0xc2, 0xeb, 0x05, 0xa9,
};
static const unsigned char kat168_nor_retbytes[] = {
    0x00, 0xdb, 0x80, 0x05, 0xe9, 0xc9, 0x19, 0x1c, 0x5c, 0xb1, 0xc8, 0x9f,
    0x79, 0xfe, 0x13, 0xa0, 0xe9, 0xb5, 0xb4, 0xe1, 0x0b, 0x4c, 0xc7, 0x72,
    0xc8, 0xad, 0x9f, 0x86, 0xd5, 0x7b, 0xb0, 0xf4, 0x3e, 0xc8, 0xa4, 0xb8,
    0x75, 0xa5, 0xd4, 0x25, 0x75, 0xb9, 0x1d, 0x6e, 0x52, 0x77, 0xb1, 0x82,
    0x75, 0xac, 0x76, 0x93, 0x25, 0xaf, 0x3f, 0xaf, 0xfb, 0xff, 0x6a, 0x9d,
    0x66, 0xf6, 0x85, 0x00, 0xa0, 0x7f, 0xae, 0xe8, 0x22, 0xd0, 0x8e, 0xd8,
    0xde, 0x79, 0x0b, 0xc5, 0x1c, 0xea, 0x80, 0xa9,
};
static const struct drbg_kat_no_reseed kat168_nor_t = {
    2, kat168_nor_entropyin, kat168_nor_nonce, kat168_nor_persstr,
    kat168_nor_addin0, kat168_nor_addin1, kat168_nor_retbytes
};
static const struct drbg_kat kat168_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat168_nor_t
};

static const unsigned char kat169_nor_entropyin[] = {
    0x5f, 0x21, 0xe9, 0x6d, 0x25, 0xa5, 0x4e, 0xc1, 0xdb, 0xa7, 0x3e, 0x67,
    0xb2, 0x20, 0x2d, 0xab,
};
static const unsigned char kat169_nor_nonce[] = {
    0xb8, 0xfd, 0x43, 0x45, 0xdc, 0x8e, 0xb0, 0x83,
};
static const unsigned char kat169_nor_persstr[] = {
    0x40, 0x92, 0xd3, 0x5f, 0xe7, 0x46, 0x09, 0xeb, 0x54, 0x0b, 0xfb, 0xc1,
    0x8f, 0x07, 0x88, 0xc6,
};
static const unsigned char kat169_nor_addin0[] = {
    0x2b, 0xcb, 0xc4, 0xa0, 0x21, 0x88, 0x4d, 0xb4, 0x10, 0xb7, 0xd2, 0x3a,
    0x35, 0x16, 0x29, 0x24,
};
static const unsigned char kat169_nor_addin1[] = {
    0x74, 0x20, 0xff, 0x1f, 0xbc, 0xf8, 0x33, 0xdb, 0x6a, 0xff, 0xfa, 0x4c,
    0x0f, 0x8b, 0xd0, 0xf8,
};
static const unsigned char kat169_nor_retbytes[] = {
    0x6f, 0x43, 0x3e, 0x45, 0x9b, 0x9a, 0xdc, 0xa5, 0xc7, 0x36, 0x93, 0xc5,
    0x86, 0xe9, 0xd3, 0x71, 0x83, 0xb9, 0xf8, 0xc4, 0x46, 0x38, 0x05, 0xcf,
    0x2b, 0x78, 0x4f, 0xcd, 0x8b, 0x9e, 0x67, 0x71, 0xf5, 0x3b, 0x6f, 0x8d,
    0xd8, 0xa9, 0x8e, 0x04, 0x76, 0x17, 0x33, 0x3c, 0xd2, 0x6e, 0xa7, 0xb7,
    0x6b, 0x2c, 0x7e, 0xfd, 0xc1, 0xcc, 0x2c, 0xcb, 0x88, 0x58, 0x77, 0x7f,
    0x64, 0x67, 0xb2, 0x22, 0xc5, 0x1e, 0xab, 0x14, 0xbc, 0x0d, 0x74, 0xc6,
    0x2a, 0xf7, 0x35, 0x82, 0x0e, 0xca, 0x54, 0xce,
};
static const struct drbg_kat_no_reseed kat169_nor_t = {
    3, kat169_nor_entropyin, kat169_nor_nonce, kat169_nor_persstr,
    kat169_nor_addin0, kat169_nor_addin1, kat169_nor_retbytes
};
static const struct drbg_kat kat169_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat169_nor_t
};

static const unsigned char kat170_nor_entropyin[] = {
    0xdb, 0x01, 0x94, 0x2b, 0x1e, 0x18, 0x9b, 0x29, 0x81, 0x40, 0x7d, 0xdf,
    0xb8, 0x4b, 0x30, 0x81,
};
static const unsigned char kat170_nor_nonce[] = {
    0x3f, 0xbe, 0x77, 0x68, 0xdc, 0xd6, 0x48, 0x70,
};
static const unsigned char kat170_nor_persstr[] = {
    0xeb, 0xa9, 0xf4, 0x97, 0x4b, 0x3c, 0xe7, 0xbf, 0x67, 0x57, 0x7f, 0xdb,
    0xb6, 0x91, 0x60, 0xad,
};
static const unsigned char kat170_nor_addin0[] = {
    0xf9, 0xf8, 0x5d, 0xc7, 0x36, 0x2a, 0x00, 0x95, 0x10, 0x95, 0x5f, 0x90,
    0x00, 0x42, 0x35, 0x56,
};
static const unsigned char kat170_nor_addin1[] = {
    0x7b, 0xd4, 0xf4, 0xd1, 0x30, 0xa5, 0xd6, 0x1b, 0xe1, 0x9e, 0xdc, 0xa0,
    0x70, 0x78, 0xe2, 0x8d,
};
static const unsigned char kat170_nor_retbytes[] = {
    0x57, 0xa4, 0xb0, 0xe9, 0x81, 0x76, 0x2f, 0x6b, 0xc6, 0xa9, 0x80, 0xb6,
    0x93, 0xa6, 0xe1, 0xda, 0xc8, 0x5c, 0xe3, 0x44, 0xe9, 0x8f, 0x63, 0xde,
    0x34, 0x5d, 0x34, 0x27, 0xd9, 0x8c, 0x4c, 0xcc, 0x00, 0x89, 0x2c, 0x6b,
    0x3e, 0xf7, 0x35, 0xbc, 0x27, 0x3f, 0x68, 0x57, 0x02, 0x1a, 0x1f, 0x2b,
    0x81, 0xbe, 0x7d, 0x60, 0xc2, 0x2e, 0x60, 0x04, 0x2f, 0xad, 0x87, 0x50,
    0x89, 0x4f, 0x77, 0xfe, 0x64, 0xbf, 0x25, 0x31, 0x2f, 0x15, 0x0f, 0xfe,
    0xe6, 0x12, 0x1b, 0x13, 0x45, 0x1a, 0xeb, 0xed,
};
static const struct drbg_kat_no_reseed kat170_nor_t = {
    4, kat170_nor_entropyin, kat170_nor_nonce, kat170_nor_persstr,
    kat170_nor_addin0, kat170_nor_addin1, kat170_nor_retbytes
};
static const struct drbg_kat kat170_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat170_nor_t
};

static const unsigned char kat171_nor_entropyin[] = {
    0x33, 0x75, 0xf0, 0x34, 0x06, 0x9c, 0x25, 0x69, 0xe6, 0x4d, 0x71, 0x8a,
    0x4d, 0xeb, 0x98, 0x74,
};
static const unsigned char kat171_nor_nonce[] = {
    0x58, 0xcf, 0x63, 0x12, 0x2c, 0x30, 0xc5, 0xf5,
};
static const unsigned char kat171_nor_persstr[] = {
    0x47, 0x65, 0x14, 0x2d, 0xa0, 0x16, 0x13, 0x78, 0x45, 0x83, 0xcf, 0xe5,
    0x9c, 0x5b, 0x9d, 0xdf,
};
static const unsigned char kat171_nor_addin0[] = {
    0x47, 0xdc, 0x67, 0x57, 0xcb, 0xce, 0x93, 0x86, 0x86, 0x78, 0x08, 0xb2,
    0x4f, 0x32, 0x78, 0xfa,
};
static const unsigned char kat171_nor_addin1[] = {
    0x20, 0x49, 0x63, 0x11, 0xbf, 0xad, 0x06, 0xcc, 0xd3, 0x46, 0x0c, 0x93,
    0x11, 0x07, 0xad, 0x87,
};
static const unsigned char kat171_nor_retbytes[] = {
    0xf5, 0x99, 0xcb, 0xbb, 0xf0, 0xd9, 0x3c, 0x9e, 0x30, 0x15, 0x1d, 0xd0,
    0x86, 0x89, 0x83, 0x8d, 0x41, 0x32, 0xe4, 0xd8, 0x3b, 0xd3, 0x80, 0x63,
    0xda, 0x1d, 0x84, 0x57, 0x66, 0xf4, 0xc3, 0x7b, 0x00, 0xbb, 0xd5, 0x60,
    0xb7, 0x83, 0x62, 0x68, 0x7c, 0xdf, 0x7d, 0x67, 0x43, 0x37, 0xd2, 0x03,
    0xa3, 0x34, 0x67, 0x22, 0x6d, 0x35, 0xcb, 0xdb, 0x6f, 0xc9, 0x24, 0xc9,
    0xb9, 0x66, 0xf9, 0xdf, 0x19, 0x96, 0x6f, 0x59, 0xf5, 0xf5, 0x7c, 0x9f,
    0x3b, 0x46, 0xe3, 0x06, 0x5e, 0xc8, 0xb6, 0x52,
};
static const struct drbg_kat_no_reseed kat171_nor_t = {
    5, kat171_nor_entropyin, kat171_nor_nonce, kat171_nor_persstr,
    kat171_nor_addin0, kat171_nor_addin1, kat171_nor_retbytes
};
static const struct drbg_kat kat171_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat171_nor_t
};

static const unsigned char kat172_nor_entropyin[] = {
    0x62, 0x9b, 0x62, 0x13, 0x1d, 0x5a, 0xf0, 0x4f, 0xe2, 0xe1, 0xf3, 0x77,
    0xc6, 0xfe, 0x7d, 0x01,
};
static const unsigned char kat172_nor_nonce[] = {
    0xca, 0xeb, 0x88, 0x81, 0x84, 0x52, 0x6f, 0x97,
};
static const unsigned char kat172_nor_persstr[] = {
    0xe9, 0x6e, 0x5c, 0xa5, 0x45, 0x5d, 0x7a, 0x26, 0x94, 0x1b, 0xd9, 0x93,
    0xe4, 0x80, 0xe6, 0x4e,
};
static const unsigned char kat172_nor_addin0[] = {
    0x0d, 0xd7, 0x0e, 0x0d, 0x02, 0x98, 0x47, 0x67, 0x29, 0x37, 0x58, 0x48,
    0x56, 0x19, 0xa3, 0x18,
};
static const unsigned char kat172_nor_addin1[] = {
    0x1b, 0x34, 0x05, 0xea, 0xa9, 0x84, 0x08, 0x01, 0x4e, 0x6b, 0x14, 0xf0,
    0x80, 0xb4, 0xb7, 0x40,
};
static const unsigned char kat172_nor_retbytes[] = {
    0x9c, 0xf2, 0x02, 0xd4, 0xd4, 0x9e, 0x74, 0x4a, 0xe4, 0xcf, 0xf5, 0x04,
    0x90, 0x49, 0x91, 0xe4, 0xd8, 0xad, 0x96, 0x12, 0xd6, 0x4d, 0xa8, 0xea,
    0x03, 0xba, 0x02, 0x3f, 0x54, 0xf4, 0xcc, 0x49, 0x59, 0x8d, 0x2f, 0x80,
    0x15, 0xb2, 0xc8, 0xbe, 0x84, 0x00, 0xb4, 0xa3, 0xd9, 0xb4, 0x4d, 0xda,
    0x53, 0x2b, 0x3e, 0x72, 0x4e, 0x27, 0xdd, 0x57, 0x44, 0xf8, 0xf3, 0xc7,
    0xd9, 0x9a, 0x33, 0x67, 0x09, 0x99, 0xb1, 0x1d, 0x40, 0x3d, 0x69, 0x4c,
    0x95, 0xcb, 0x7e, 0x06, 0xa8, 0xa0, 0xe0, 0xb4,
};
static const struct drbg_kat_no_reseed kat172_nor_t = {
    6, kat172_nor_entropyin, kat172_nor_nonce, kat172_nor_persstr,
    kat172_nor_addin0, kat172_nor_addin1, kat172_nor_retbytes
};
static const struct drbg_kat kat172_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat172_nor_t
};

static const unsigned char kat173_nor_entropyin[] = {
    0x1e, 0x46, 0xcc, 0xc7, 0xb6, 0x2a, 0x7a, 0x19, 0x95, 0x8d, 0x65, 0xcb,
    0x6d, 0x20, 0xc1, 0x06,
};
static const unsigned char kat173_nor_nonce[] = {
    0x68, 0x0b, 0xb7, 0xfa, 0x43, 0xb0, 0xdb, 0x4a,
};
static const unsigned char kat173_nor_persstr[] = {
    0x2e, 0x38, 0xeb, 0x08, 0x3f, 0xc5, 0xdb, 0x45, 0x98, 0x57, 0x6d, 0xae,
    0x95, 0xa0, 0x93, 0xb2,
};
static const unsigned char kat173_nor_addin0[] = {
    0x95, 0xb3, 0x64, 0x9a, 0xcf, 0xe5, 0x15, 0x8e, 0xf8, 0x09, 0x06, 0xc4,
    0x1a, 0xe7, 0x48, 0x7f,
};
static const unsigned char kat173_nor_addin1[] = {
    0x03, 0xa9, 0x0d, 0xc9, 0x38, 0x65, 0x0d, 0x30, 0xbe, 0x53, 0xcd, 0x72,
    0xc2, 0x69, 0x07, 0x14,
};
static const unsigned char kat173_nor_retbytes[] = {
    0x05, 0x49, 0x5f, 0x39, 0x99, 0x3c, 0x97, 0xe1, 0xf8, 0x4f, 0xae, 0xe2,
    0x05, 0x82, 0x64, 0xb1, 0xc2, 0x5b, 0x3e, 0xf6, 0x1f, 0x59, 0xeb, 0xa1,
    0x83, 0x2f, 0xa6, 0xaf, 0x8c, 0x11, 0xe2, 0x1b, 0xa3, 0xd4, 0xd3, 0x8c,
    0x25, 0x7a, 0x46, 0xef, 0xaa, 0xc7, 0x94, 0x2a, 0x58, 0x2b, 0x17, 0x27,
    0x7f, 0xec, 0x6b, 0x3a, 0x9a, 0x95, 0x4e, 0x3f, 0x32, 0x3c, 0x9a, 0x2a,
    0x9f, 0x8f, 0xa9, 0xf9, 0x0b, 0x1f, 0x47, 0xfb, 0xee, 0xc6, 0xd1, 0x06,
    0x98, 0x33, 0x7c, 0x18, 0x7e, 0x5b, 0x82, 0xc8,
};
static const struct drbg_kat_no_reseed kat173_nor_t = {
    7, kat173_nor_entropyin, kat173_nor_nonce, kat173_nor_persstr,
    kat173_nor_addin0, kat173_nor_addin1, kat173_nor_retbytes
};
static const struct drbg_kat kat173_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat173_nor_t
};

static const unsigned char kat174_nor_entropyin[] = {
    0x21, 0x95, 0x2d, 0xb3, 0x3e, 0xbd, 0xf9, 0xae, 0x6a, 0x78, 0x5f, 0x6a,
    0xf9, 0x80, 0x91, 0x54,
};
static const unsigned char kat174_nor_nonce[] = {
    0x72, 0x2e, 0xe3, 0x82, 0xe4, 0xa0, 0x4a, 0xe5,
};
static const unsigned char kat174_nor_persstr[] = {
    0xb9, 0x91, 0xbf, 0x9e, 0xb7, 0xcc, 0xa7, 0x6f, 0x25, 0xf4, 0x5e, 0x1d,
    0x0e, 0x0e, 0x96, 0xff,
};
static const unsigned char kat174_nor_addin0[] = {
    0xa8, 0xd8, 0xb6, 0xee, 0x7b, 0xe5, 0x2c, 0xb0, 0x41, 0x4c, 0x55, 0xa3,
    0x8f, 0x6a, 0xd7, 0x4c,
};
static const unsigned char kat174_nor_addin1[] = {
    0x7f, 0x5c, 0x64, 0xce, 0x33, 0x73, 0xc1, 0x54, 0x37, 0x6f, 0x4f, 0x18,
    0x9b, 0x11, 0xe6, 0xfe,
};
static const unsigned char kat174_nor_retbytes[] = {
    0x33, 0xbc, 0x2a, 0x75, 0x99, 0xb4, 0xa2, 0xdb, 0x6f, 0x0d, 0xd1, 0x89,
    0x6e, 0x40, 0xdc, 0xf6, 0x12, 0xb6, 0x26, 0x91, 0x95, 0xe8, 0x1b, 0x20,
    0xc4, 0x84, 0x2c, 0xda, 0x27, 0x22, 0x7f, 0xd9, 0xe7, 0x49, 0x11, 0xd5,
    0x50, 0x4f, 0xc3, 0x4c, 0xa0, 0xd8, 0x58, 0x41, 0x2b, 0x50, 0x82, 0xcc,
    0xe2, 0xc0, 0x7d, 0x37, 0x7e, 0xa4, 0xea, 0xa7, 0xe8, 0x78, 0x22, 0x94,
    0x60, 0xac, 0x2f, 0xb5, 0xad, 0xf2, 0x41, 0xb2, 0x92, 0xde, 0xc3, 0x1a,
    0xeb, 0xfa, 0x44, 0x34, 0xee, 0xff, 0x38, 0x08,
};
static const struct drbg_kat_no_reseed kat174_nor_t = {
    8, kat174_nor_entropyin, kat174_nor_nonce, kat174_nor_persstr,
    kat174_nor_addin0, kat174_nor_addin1, kat174_nor_retbytes
};
static const struct drbg_kat kat174_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat174_nor_t
};

static const unsigned char kat175_nor_entropyin[] = {
    0xf0, 0x81, 0xdd, 0x93, 0x2d, 0x9f, 0x5e, 0x53, 0xd9, 0xee, 0xe9, 0x80,
    0x7d, 0xb5, 0x8f, 0xcd,
};
static const unsigned char kat175_nor_nonce[] = {
    0x92, 0xf0, 0x27, 0xb1, 0x5c, 0xa5, 0x86, 0x6f,
};
static const unsigned char kat175_nor_persstr[] = {
    0x3c, 0xab, 0xd9, 0xfd, 0x3d, 0xe7, 0xe1, 0x96, 0xe1, 0x80, 0x3f, 0x36,
    0x19, 0xa3, 0xd0, 0x48,
};
static const unsigned char kat175_nor_addin0[] = {
    0x5a, 0xea, 0x80, 0x4f, 0x52, 0x50, 0x07, 0x5d, 0x54, 0x27, 0xde, 0x0e,
    0x73, 0xdf, 0x42, 0xda,
};
static const unsigned char kat175_nor_addin1[] = {
    0xed, 0xdc, 0x6c, 0x24, 0x28, 0x13, 0x84, 0x1e, 0xd7, 0xac, 0x48, 0x3e,
    0xb6, 0xcb, 0xab, 0x67,
};
static const unsigned char kat175_nor_retbytes[] = {
    0xbe, 0x50, 0x36, 0xd8, 0xf7, 0x9f, 0xc9, 0xc1, 0x02, 0x9c, 0x3f, 0x9e,
    0x42, 0xf1, 0x04, 0x53, 0x43, 0xdf, 0xca, 0x9e, 0xff, 0x6b, 0x74, 0xf6,
    0xac, 0x0b, 0x96, 0x66, 0xb4, 0xdf, 0x71, 0x51, 0xeb, 0xd1, 0x3f, 0xd0,
    0x52, 0xcc, 0x8b, 0xb9, 0xbc, 0x7b, 0x1c, 0xe6, 0xfc, 0xe0, 0xf2, 0x74,
    0x9f, 0x31, 0x3d, 0xf7, 0xb9, 0xd0, 0x6c, 0xeb, 0xf3, 0xf6, 0xaa, 0x93,
    0x62, 0x70, 0x03, 0x2e, 0x8e, 0x0d, 0xec, 0x34, 0x05, 0xee, 0x24, 0x6b,
    0x80, 0x82, 0x68, 0x50, 0xa1, 0x22, 0x89, 0x00,
};
static const struct drbg_kat_no_reseed kat175_nor_t = {
    9, kat175_nor_entropyin, kat175_nor_nonce, kat175_nor_persstr,
    kat175_nor_addin0, kat175_nor_addin1, kat175_nor_retbytes
};
static const struct drbg_kat kat175_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat175_nor_t
};

static const unsigned char kat176_nor_entropyin[] = {
    0x9e, 0xb9, 0x08, 0xf5, 0x09, 0xc7, 0x7c, 0x0a, 0xc9, 0x53, 0x56, 0x04,
    0x93, 0xf4, 0xdc, 0x28,
};
static const unsigned char kat176_nor_nonce[] = {
    0x1a, 0xeb, 0xf6, 0x9b, 0x03, 0xde, 0x5c, 0xf8,
};
static const unsigned char kat176_nor_persstr[] = {
    0xa7, 0xf0, 0x1f, 0x55, 0xfe, 0x0e, 0x6b, 0x6d, 0x73, 0x1f, 0x38, 0x7c,
    0xaf, 0x95, 0xd0, 0x45,
};
static const unsigned char kat176_nor_addin0[] = {
    0x5d, 0xca, 0x6d, 0x75, 0x18, 0x3f, 0x30, 0x92, 0xa1, 0x8c, 0x8d, 0xfd,
    0x95, 0x0f, 0xdf, 0x98,
};
static const unsigned char kat176_nor_addin1[] = {
    0x50, 0xeb, 0x0c, 0xc7, 0x6a, 0xa3, 0x8e, 0x31, 0xb8, 0x7e, 0x7c, 0xd8,
    0xdf, 0x95, 0x0f, 0x08,
};
static const unsigned char kat176_nor_retbytes[] = {
    0xba, 0x52, 0xcb, 0xdb, 0x84, 0x5f, 0x5e, 0xe8, 0x64, 0xfc, 0x1c, 0x17,
    0xc6, 0xbd, 0x0a, 0x68, 0x43, 0x9a, 0x7e, 0xee, 0x70, 0x27, 0xfe, 0xf5,
    0xa8, 0x08, 0x3b, 0x2e, 0x01, 0x46, 0xc6, 0x04, 0x64, 0x24, 0x65, 0x27,
    0xd5, 0x2a, 0xe5, 0x9f, 0xf5, 0x58, 0x0f, 0x33, 0x6b, 0x94, 0x59, 0x4f,
    0x5a, 0xa9, 0x33, 0xc3, 0xf7, 0x57, 0x44, 0xc7, 0x67, 0x84, 0xdf, 0x97,
    0x83, 0xbd, 0xc9, 0x1d, 0x13, 0x0e, 0x29, 0x65, 0x8f, 0xb2, 0x72, 0x04,
    0x5e, 0x95, 0x37, 0x94, 0xae, 0x9b, 0xb3, 0x16,
};
static const struct drbg_kat_no_reseed kat176_nor_t = {
    10, kat176_nor_entropyin, kat176_nor_nonce, kat176_nor_persstr,
    kat176_nor_addin0, kat176_nor_addin1, kat176_nor_retbytes
};
static const struct drbg_kat kat176_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat176_nor_t
};

static const unsigned char kat177_nor_entropyin[] = {
    0xf8, 0xcf, 0x96, 0x2f, 0x6e, 0x58, 0x39, 0x70, 0x02, 0x7d, 0x42, 0x63,
    0xf3, 0xf8, 0xb9, 0x36,
};
static const unsigned char kat177_nor_nonce[] = {
    0x0c, 0xc2, 0x39, 0xa1, 0xfe, 0x97, 0x50, 0xc7,
};
static const unsigned char kat177_nor_persstr[] = {
    0xc9, 0x25, 0x1c, 0x51, 0x9d, 0x55, 0x26, 0xd6, 0x88, 0x2f, 0x1b, 0x23,
    0x5f, 0xb9, 0xb9, 0x44,
};
static const unsigned char kat177_nor_addin0[] = {
    0x97, 0x52, 0xb7, 0xe4, 0x4c, 0x67, 0xda, 0x39, 0x0d, 0x4e, 0xd5, 0xe8,
    0xb9, 0x58, 0x17, 0xcb,
};
static const unsigned char kat177_nor_addin1[] = {
    0xd1, 0x8d, 0x46, 0x05, 0x12, 0x74, 0x7b, 0x63, 0x86, 0x52, 0xfd, 0x4d,
    0x45, 0x7b, 0xbc, 0xe9,
};
static const unsigned char kat177_nor_retbytes[] = {
    0x5a, 0x7e, 0x2a, 0xc6, 0x6b, 0x5f, 0xb1, 0x2b, 0x3e, 0xee, 0x1b, 0x6a,
    0xad, 0x4b, 0x34, 0xe2, 0x43, 0x9c, 0x53, 0x22, 0x9e, 0x6f, 0x2a, 0x10,
    0xa9, 0x20, 0x69, 0xf8, 0x03, 0x45, 0x62, 0xd1, 0x12, 0x5a, 0x70, 0x77,
    0x66, 0x93, 0x6b, 0x6e, 0x9c, 0x65, 0x51, 0xd7, 0xc8, 0xc0, 0x27, 0x75,
    0xa5, 0x72, 0x18, 0x81, 0xf1, 0x2a, 0x28, 0xe1, 0xd6, 0x3c, 0x18, 0xe4,
    0xa7, 0x86, 0x92, 0x32, 0x51, 0x19, 0x89, 0xc7, 0x9c, 0x5a, 0x8d, 0x3c,
    0xb1, 0xd4, 0xcb, 0x36, 0xea, 0x92, 0x1c, 0x78,
};
static const struct drbg_kat_no_reseed kat177_nor_t = {
    11, kat177_nor_entropyin, kat177_nor_nonce, kat177_nor_persstr,
    kat177_nor_addin0, kat177_nor_addin1, kat177_nor_retbytes
};
static const struct drbg_kat kat177_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat177_nor_t
};

static const unsigned char kat178_nor_entropyin[] = {
    0x55, 0xe3, 0x11, 0xa2, 0x3e, 0x87, 0x2b, 0xa9, 0x06, 0xbf, 0x06, 0x03,
    0x88, 0xe6, 0x0a, 0xe5,
};
static const unsigned char kat178_nor_nonce[] = {
    0x59, 0x72, 0xe7, 0xae, 0xb9, 0xf7, 0x53, 0x44,
};
static const unsigned char kat178_nor_persstr[] = {
    0xbe, 0x3f, 0x6e, 0x93, 0x23, 0x39, 0xc5, 0xb9, 0xf7, 0x1b, 0x87, 0xe7,
    0x2f, 0xf4, 0xda, 0xb1,
};
static const unsigned char kat178_nor_addin0[] = {
    0xfa, 0xb4, 0x6e, 0x39, 0x1e, 0x2c, 0x5f, 0x30, 0xb3, 0x72, 0x98, 0x64,
    0x3b, 0xc5, 0x5d, 0x50,
};
static const unsigned char kat178_nor_addin1[] = {
    0xcf, 0x4e, 0x54, 0x83, 0xcf, 0xd7, 0x94, 0xf0, 0x32, 0x69, 0xe2, 0x89,
    0xf2, 0xb4, 0xc0, 0x8c,
};
static const unsigned char kat178_nor_retbytes[] = {
    0x64, 0x8e, 0xc7, 0x77, 0x3f, 0x8c, 0x66, 0x4f, 0x3d, 0x7f, 0xb2, 0x79,
    0xb5, 0x1c, 0x5b, 0x9a, 0x3d, 0x9b, 0x06, 0x7d, 0x6a, 0xa5, 0x4e, 0x93,
    0xc9, 0xbb, 0x9a, 0x89, 0x2d, 0xa9, 0x1f, 0xad, 0x4a, 0x30, 0xe8, 0xe4,
    0xc5, 0x86, 0x6d, 0xda, 0xbd, 0x2b, 0xb4, 0x4b, 0x24, 0x3d, 0x7a, 0x3a,
    0x00, 0xaf, 0xfe, 0xc9, 0xef, 0xb7, 0x25, 0xc9, 0x51, 0x86, 0x2f, 0xf9,
    0x18, 0x43, 0x92, 0xdf, 0xd6, 0x26, 0xe7, 0x11, 0xe0, 0xa3, 0x84, 0x0d,
    0xb8, 0x49, 0xb1, 0xe3, 0xb0, 0x1b, 0x4f, 0xa4,
};
static const struct drbg_kat_no_reseed kat178_nor_t = {
    12, kat178_nor_entropyin, kat178_nor_nonce, kat178_nor_persstr,
    kat178_nor_addin0, kat178_nor_addin1, kat178_nor_retbytes
};
static const struct drbg_kat kat178_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat178_nor_t
};

static const unsigned char kat179_nor_entropyin[] = {
    0x4a, 0x98, 0x9e, 0x6b, 0xc3, 0x7b, 0x91, 0x57, 0xfe, 0xfe, 0x67, 0x89,
    0xaf, 0xf2, 0x30, 0xf3,
};
static const unsigned char kat179_nor_nonce[] = {
    0x7e, 0x5e, 0xe2, 0x46, 0x79, 0x8d, 0xd7, 0xbd,
};
static const unsigned char kat179_nor_persstr[] = {
    0xa7, 0xf5, 0xaf, 0x48, 0xda, 0x67, 0x7d, 0x67, 0x21, 0xd1, 0x1e, 0x28,
    0x4f, 0xb1, 0xff, 0xef,
};
static const unsigned char kat179_nor_addin0[] = {
    0x33, 0x51, 0x00, 0x20, 0x30, 0xee, 0xa1, 0x00, 0xc8, 0x48, 0x40, 0x74,
    0x8b, 0x0a, 0x3d, 0x4d,
};
static const unsigned char kat179_nor_addin1[] = {
    0xac, 0xd3, 0x2c, 0x8d, 0x51, 0x51, 0x09, 0xe9, 0x0d, 0xfe, 0x1b, 0xd9,
    0xe6, 0xb3, 0x26, 0xbd,
};
static const unsigned char kat179_nor_retbytes[] = {
    0xa7, 0x7b, 0xd7, 0xdc, 0x39, 0x86, 0xb4, 0x31, 0x36, 0x60, 0x6a, 0x20,
    0x9d, 0x60, 0x0a, 0xd9, 0x00, 0x68, 0xb5, 0x67, 0xa5, 0xe3, 0xa2, 0xf7,
    0x5e, 0x61, 0xb2, 0x74, 0x8e, 0x26, 0x5e, 0x50, 0x0f, 0xc7, 0xed, 0x49,
    0x2b, 0x8c, 0xdd, 0xe3, 0x9a, 0x6b, 0x0a, 0x87, 0xad, 0x4d, 0xd0, 0x05,
    0x29, 0xfa, 0x7f, 0xeb, 0x46, 0x35, 0xe3, 0x15, 0xb7, 0x11, 0x6e, 0x1c,
    0xe6, 0x25, 0x1a, 0x80, 0x6c, 0x7c, 0x87, 0x1f, 0x7c, 0xcc, 0x44, 0x75,
    0xe3, 0x32, 0x05, 0x4e, 0xf8, 0x12, 0xc6, 0x31,
};
static const struct drbg_kat_no_reseed kat179_nor_t = {
    13, kat179_nor_entropyin, kat179_nor_nonce, kat179_nor_persstr,
    kat179_nor_addin0, kat179_nor_addin1, kat179_nor_retbytes
};
static const struct drbg_kat kat179_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat179_nor_t
};

static const unsigned char kat180_nor_entropyin[] = {
    0x75, 0xdd, 0xc6, 0xb5, 0x8a, 0x06, 0xa0, 0xfe, 0xb2, 0x0b, 0x70, 0x3a,
    0xb6, 0xdd, 0x6d, 0xa0,
};
static const unsigned char kat180_nor_nonce[] = {
    0x52, 0x71, 0x30, 0xcd, 0xe5, 0x21, 0x4f, 0x55,
};
static const unsigned char kat180_nor_persstr[] = {
    0x3a, 0x2f, 0xa9, 0xc3, 0xb1, 0x33, 0x97, 0xed, 0x8e, 0xbd, 0xf5, 0x7d,
    0x1e, 0xfc, 0x1c, 0x97,
};
static const unsigned char kat180_nor_addin0[] = {
    0x27, 0x48, 0x6f, 0x8d, 0xae, 0x1b, 0x36, 0x46, 0x26, 0x39, 0xff, 0x7e,
    0xee, 0x86, 0x9a, 0x29,
};
static const unsigned char kat180_nor_addin1[] = {
    0xd1, 0xbf, 0xc7, 0xea, 0xbd, 0x8e, 0xdd, 0xf6, 0x22, 0x29, 0x70, 0x12,
    0x16, 0x9f, 0x35, 0x1b,
};
static const unsigned char kat180_nor_retbytes[] = {
    0x4c, 0x89, 0x3c, 0x3d, 0x1e, 0xd3, 0xa1, 0x90, 0xfa, 0x88, 0xe1, 0x59,
    0xd6, 0xc9, 0x9f, 0x26, 0xa0, 0x2f, 0xb5, 0xfc, 0xcb, 0x98, 0xbd, 0xef,
    0x9f, 0xe4, 0x3f, 0x1f, 0x49, 0x2f, 0x49, 0x01, 0x09, 0x22, 0x4b, 0xa6,
    0xc3, 0x17, 0xdb, 0x95, 0x69, 0xf6, 0x18, 0x98, 0x44, 0x09, 0xf2, 0xfb,
    0x3d, 0xb0, 0xb1, 0xe2, 0xcd, 0x4b, 0x95, 0x74, 0x6f, 0x15, 0x9c, 0xca,
    0x76, 0xf1, 0x20, 0x4f, 0x6d, 0x2a, 0x4c, 0x45, 0x5c, 0x54, 0x7a, 0x39,
    0xa5, 0xf7, 0x9f, 0xec, 0x95, 0xc8, 0xf4, 0xcd,
};
static const struct drbg_kat_no_reseed kat180_nor_t = {
    14, kat180_nor_entropyin, kat180_nor_nonce, kat180_nor_persstr,
    kat180_nor_addin0, kat180_nor_addin1, kat180_nor_retbytes
};
static const struct drbg_kat kat180_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat180_nor_t
};

static const unsigned char kat181_nor_entropyin[] = {
    0x4c, 0x2f, 0xbe, 0xa1, 0x66, 0xf7, 0x92, 0x0a, 0xf0, 0xf0, 0x52, 0x1e,
    0x22, 0x48, 0x72, 0x66,
};
static const unsigned char kat181_nor_nonce[] = {
    0xef, 0x58, 0xd3, 0x98, 0x07, 0xd5, 0x88, 0xd8,
};
static const unsigned char kat181_nor_persstr[] = {0};
static const unsigned char kat181_nor_addin0[] = {0};
static const unsigned char kat181_nor_addin1[] = {0};
static const unsigned char kat181_nor_retbytes[] = {
    0x54, 0xd8, 0xc2, 0x26, 0x6c, 0x0f, 0xa6, 0x61, 0x3e, 0x76, 0x76, 0x88,
    0x56, 0x88, 0x3d, 0xa0, 0xbd, 0xcd, 0x0d, 0x66, 0x21, 0x40, 0x7e, 0x85,
    0x99, 0x45, 0x1e, 0x28, 0x59, 0xb5, 0xd0, 0xb5, 0x6e, 0x58, 0x80, 0x89,
    0xe3, 0x9e, 0x22, 0x9d, 0xdf, 0x84, 0xf3, 0x26, 0x7b, 0xa5, 0xac, 0xde,
    0x53, 0x9e, 0x5c, 0x0e, 0x4d, 0x33, 0xaa, 0x51, 0xe1, 0x8e, 0x07, 0x58,
    0xe7, 0xb5, 0x2a, 0x24, 0x0d, 0x39, 0x94, 0xa7, 0x7e, 0xdc, 0x4b, 0xd8,
    0xd3, 0xc2, 0x54, 0xbe, 0xdb, 0xbe, 0x27, 0xf1,
};
static const struct drbg_kat_no_reseed kat181_nor_t = {
    0, kat181_nor_entropyin, kat181_nor_nonce, kat181_nor_persstr,
    kat181_nor_addin0, kat181_nor_addin1, kat181_nor_retbytes
};
static const struct drbg_kat kat181_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat181_nor_t
};

static const unsigned char kat182_nor_entropyin[] = {
    0x5d, 0x83, 0x9a, 0xd9, 0xed, 0xa9, 0x31, 0x02, 0x19, 0x01, 0xb2, 0x62,
    0xe4, 0x8f, 0xfc, 0x81,
};
static const unsigned char kat182_nor_nonce[] = {
    0xef, 0xbe, 0xe0, 0xdc, 0xb5, 0x60, 0x10, 0x86,
};
static const unsigned char kat182_nor_persstr[] = {0};
static const unsigned char kat182_nor_addin0[] = {0};
static const unsigned char kat182_nor_addin1[] = {0};
static const unsigned char kat182_nor_retbytes[] = {
    0xde, 0x5d, 0x11, 0x98, 0xd1, 0xda, 0x55, 0x80, 0x60, 0x56, 0xd7, 0x95,
    0x79, 0x40, 0x9f, 0xe7, 0x24, 0x15, 0xd1, 0x81, 0x14, 0x8d, 0xb8, 0x27,
    0x3b, 0x0c, 0xba, 0x1e, 0x9f, 0xf0, 0xa8, 0xf4, 0x55, 0x20, 0x47, 0xde,
    0xd2, 0xf5, 0x65, 0xe6, 0xff, 0x70, 0x35, 0xed, 0x42, 0x04, 0x78, 0xeb,
    0xbd, 0x8e, 0x4f, 0x0e, 0x30, 0x1d, 0xe8, 0xa5, 0xb5, 0x58, 0xf2, 0x37,
    0x3a, 0x57, 0x70, 0x8e, 0x05, 0xf0, 0x01, 0x77, 0x48, 0xc4, 0x0f, 0x7b,
    0xbd, 0xa2, 0xd2, 0x97, 0xbe, 0x23, 0xcf, 0xdb,
};
static const struct drbg_kat_no_reseed kat182_nor_t = {
    1, kat182_nor_entropyin, kat182_nor_nonce, kat182_nor_persstr,
    kat182_nor_addin0, kat182_nor_addin1, kat182_nor_retbytes
};
static const struct drbg_kat kat182_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat182_nor_t
};

static const unsigned char kat183_nor_entropyin[] = {
    0x80, 0x31, 0x71, 0xf0, 0x67, 0x59, 0x50, 0x3e, 0x5a, 0xca, 0x87, 0xee,
    0x10, 0xd3, 0x7f, 0xbe,
};
static const unsigned char kat183_nor_nonce[] = {
    0xdf, 0x19, 0x52, 0x7d, 0xf6, 0xda, 0x46, 0x6b,
};
static const unsigned char kat183_nor_persstr[] = {0};
static const unsigned char kat183_nor_addin0[] = {0};
static const unsigned char kat183_nor_addin1[] = {0};
static const unsigned char kat183_nor_retbytes[] = {
    0x8f, 0x71, 0xd3, 0x5e, 0x91, 0x48, 0x0b, 0x19, 0x31, 0x17, 0x4d, 0x08,
    0x0c, 0x90, 0x5a, 0xbb, 0xa4, 0xe1, 0x17, 0x94, 0x76, 0xa3, 0x5e, 0xaf,
    0xfb, 0x18, 0xdf, 0x53, 0x8b, 0x39, 0x7d, 0x45, 0xde, 0xcc, 0x4a, 0x4c,
    0x9f, 0xbe, 0xe8, 0xe8, 0xb7, 0x49, 0x11, 0xe7, 0xe3, 0x79, 0xd6, 0x7c,
    0xad, 0x77, 0x87, 0x36, 0x71, 0x24, 0x8c, 0x65, 0xc2, 0x1c, 0x7f, 0x93,
    0xd9, 0xc2, 0x11, 0x62, 0x84, 0xe7, 0x95, 0xe9, 0x41, 0xc8, 0xef, 0xaf,
    0x2d, 0xe0, 0xd1, 0x47, 0x4e, 0x81, 0xe7, 0x0d,
};
static const struct drbg_kat_no_reseed kat183_nor_t = {
    2, kat183_nor_entropyin, kat183_nor_nonce, kat183_nor_persstr,
    kat183_nor_addin0, kat183_nor_addin1, kat183_nor_retbytes
};
static const struct drbg_kat kat183_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat183_nor_t
};

static const unsigned char kat184_nor_entropyin[] = {
    0xbe, 0xb3, 0x3d, 0x42, 0xe9, 0xbe, 0x57, 0x3b, 0xd9, 0xe1, 0xc9, 0xe1,
    0xd6, 0xaf, 0xf4, 0x4d,
};
static const unsigned char kat184_nor_nonce[] = {
    0xe4, 0x2f, 0x6b, 0x81, 0x77, 0xfc, 0x41, 0x7d,
};
static const unsigned char kat184_nor_persstr[] = {0};
static const unsigned char kat184_nor_addin0[] = {0};
static const unsigned char kat184_nor_addin1[] = {0};
static const unsigned char kat184_nor_retbytes[] = {
    0xdf, 0xca, 0xc4, 0x39, 0x73, 0x8b, 0x84, 0xd0, 0x38, 0x18, 0x56, 0x8b,
    0x2b, 0xb9, 0xd7, 0x1a, 0x0f, 0x52, 0xf0, 0xb9, 0x62, 0xd9, 0x3d, 0x91,
    0x3e, 0x2b, 0x90, 0x9a, 0xb4, 0x64, 0xa3, 0x78, 0xe0, 0x03, 0x64, 0x0c,
    0xa3, 0xad, 0xdb, 0x46, 0xf1, 0xf4, 0x60, 0x75, 0x43, 0xf2, 0x98, 0x5f,
    0x56, 0x1f, 0xc4, 0xde, 0xf4, 0x08, 0xb4, 0x7f, 0xca, 0xf8, 0x38, 0xe2,
    0x54, 0x68, 0xfc, 0x97, 0xdb, 0xcf, 0x05, 0x7e, 0xdd, 0x51, 0xd8, 0x8c,
    0x34, 0x2c, 0x81, 0xd6, 0xce, 0x92, 0xb0, 0x20,
};
static const struct drbg_kat_no_reseed kat184_nor_t = {
    3, kat184_nor_entropyin, kat184_nor_nonce, kat184_nor_persstr,
    kat184_nor_addin0, kat184_nor_addin1, kat184_nor_retbytes
};
static const struct drbg_kat kat184_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat184_nor_t
};

static const unsigned char kat185_nor_entropyin[] = {
    0x6b, 0xd4, 0x9c, 0xb7, 0xc8, 0xac, 0x1d, 0x6a, 0x04, 0xdb, 0x64, 0x2d,
    0x8b, 0x46, 0x2e, 0xe7,
};
static const unsigned char kat185_nor_nonce[] = {
    0x73, 0xe7, 0xca, 0xdf, 0x29, 0xbe, 0x5f, 0x35,
};
static const unsigned char kat185_nor_persstr[] = {0};
static const unsigned char kat185_nor_addin0[] = {0};
static const unsigned char kat185_nor_addin1[] = {0};
static const unsigned char kat185_nor_retbytes[] = {
    0xaa, 0x51, 0x57, 0xbd, 0xe2, 0xc8, 0x33, 0x2f, 0x7e, 0xde, 0x49, 0xb6,
    0xfd, 0x59, 0x0d, 0x90, 0xdc, 0xb6, 0x37, 0xaa, 0x2c, 0x62, 0x27, 0x2d,
    0x72, 0x1f, 0x0f, 0x80, 0xba, 0x8f, 0xa8, 0x3f, 0x61, 0x6b, 0x90, 0x1d,
    0xff, 0xe3, 0xb9, 0xe2, 0x36, 0x00, 0x3a, 0xd0, 0x1a, 0x95, 0x9d, 0x95,
    0xbf, 0xfb, 0x1e, 0x24, 0xf1, 0x8c, 0xb4, 0x44, 0x5d, 0x4b, 0x64, 0xae,
    0x2b, 0x5d, 0x1e, 0x78, 0xfd, 0x0b, 0x58, 0x30, 0xfa, 0x4a, 0xec, 0x73,
    0xee, 0x18, 0x9f, 0x17, 0xbe, 0xd7, 0x9b, 0x80,
};
static const struct drbg_kat_no_reseed kat185_nor_t = {
    4, kat185_nor_entropyin, kat185_nor_nonce, kat185_nor_persstr,
    kat185_nor_addin0, kat185_nor_addin1, kat185_nor_retbytes
};
static const struct drbg_kat kat185_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat185_nor_t
};

static const unsigned char kat186_nor_entropyin[] = {
    0x93, 0x06, 0xeb, 0x88, 0xe8, 0xa6, 0x27, 0x44, 0xff, 0x5d, 0xd0, 0x1d,
    0x92, 0xe5, 0xfc, 0x4e,
};
static const unsigned char kat186_nor_nonce[] = {
    0x58, 0x2d, 0xe6, 0x2d, 0x3c, 0x45, 0xe9, 0xc6,
};
static const unsigned char kat186_nor_persstr[] = {0};
static const unsigned char kat186_nor_addin0[] = {0};
static const unsigned char kat186_nor_addin1[] = {0};
static const unsigned char kat186_nor_retbytes[] = {
    0xb7, 0x37, 0x14, 0xfb, 0x42, 0xe9, 0x16, 0x20, 0xd1, 0xb8, 0x1a, 0x80,
    0xa4, 0x16, 0xf5, 0x23, 0x9e, 0xd3, 0x85, 0x6e, 0x4f, 0x73, 0x20, 0xfc,
    0x57, 0x26, 0xa6, 0x32, 0x07, 0x70, 0x77, 0x10, 0x80, 0x3e, 0xa3, 0xfe,
    0x11, 0x81, 0x3e, 0x6b, 0xb7, 0xd5, 0xc2, 0xb9, 0x56, 0x5e, 0xaf, 0x05,
    0x9d, 0x8f, 0x33, 0x84, 0xf6, 0xdd, 0x1d, 0x83, 0xf6, 0x91, 0x7f, 0xf3,
    0xef, 0xcf, 0x92, 0x90, 0xe3, 0x47, 0x52, 0x92, 0x3e, 0x54, 0x77, 0x4a,
    0x10, 0x3d, 0x0a, 0x51, 0x35, 0xef, 0x2f, 0xdf,
};
static const struct drbg_kat_no_reseed kat186_nor_t = {
    5, kat186_nor_entropyin, kat186_nor_nonce, kat186_nor_persstr,
    kat186_nor_addin0, kat186_nor_addin1, kat186_nor_retbytes
};
static const struct drbg_kat kat186_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat186_nor_t
};

static const unsigned char kat187_nor_entropyin[] = {
    0x02, 0x83, 0x59, 0x2b, 0x01, 0xb7, 0xd9, 0x32, 0xdd, 0x79, 0xbe, 0x81,
    0x5a, 0x32, 0x58, 0x41,
};
static const unsigned char kat187_nor_nonce[] = {
    0x4f, 0xcf, 0x02, 0xc7, 0x76, 0x4c, 0x5a, 0x3b,
};
static const unsigned char kat187_nor_persstr[] = {0};
static const unsigned char kat187_nor_addin0[] = {0};
static const unsigned char kat187_nor_addin1[] = {0};
static const unsigned char kat187_nor_retbytes[] = {
    0x24, 0x70, 0xde, 0x1f, 0xf9, 0x0f, 0xd8, 0x4c, 0x4e, 0x17, 0xea, 0xc1,
    0xdc, 0xff, 0x65, 0x63, 0x47, 0x99, 0x71, 0x6a, 0xa0, 0xd0, 0x27, 0xa3,
    0xa9, 0xab, 0xdf, 0x92, 0xae, 0x95, 0x6a, 0x18, 0xab, 0xe8, 0xb2, 0x37,
    0x42, 0xbb, 0xf8, 0x31, 0x7a, 0xec, 0xe6, 0xa6, 0xd1, 0xad, 0x2e, 0x00,
    0x41, 0xfb, 0x11, 0x2b, 0x7c, 0x85, 0x71, 0x50, 0xf3, 0x37, 0xa1, 0x59,
    0x27, 0x09, 0x5f, 0x0a, 0x9e, 0x8c, 0xaa, 0x42, 0xb3, 0x47, 0xc6, 0x5e,
    0x75, 0x0b, 0x07, 0x37, 0xb0, 0xa7, 0x02, 0x41,
};
static const struct drbg_kat_no_reseed kat187_nor_t = {
    6, kat187_nor_entropyin, kat187_nor_nonce, kat187_nor_persstr,
    kat187_nor_addin0, kat187_nor_addin1, kat187_nor_retbytes
};
static const struct drbg_kat kat187_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat187_nor_t
};

static const unsigned char kat188_nor_entropyin[] = {
    0xd1, 0xb8, 0x1f, 0xb4, 0x14, 0xd0, 0xc0, 0x70, 0xee, 0x84, 0x69, 0xb1,
    0x90, 0xc3, 0x32, 0x79,
};
static const unsigned char kat188_nor_nonce[] = {
    0x43, 0xf7, 0x2e, 0x66, 0x35, 0x8b, 0x29, 0xe1,
};
static const unsigned char kat188_nor_persstr[] = {0};
static const unsigned char kat188_nor_addin0[] = {0};
static const unsigned char kat188_nor_addin1[] = {0};
static const unsigned char kat188_nor_retbytes[] = {
    0xca, 0xad, 0x46, 0x38, 0xfb, 0x82, 0xe6, 0x19, 0x66, 0x5a, 0x30, 0x0c,
    0xfa, 0x2f, 0x88, 0x23, 0x4e, 0xb6, 0x5d, 0x41, 0x4b, 0x9d, 0x1e, 0xc9,
    0xf9, 0xcb, 0xee, 0x94, 0x29, 0x62, 0x63, 0x93, 0x36, 0x6c, 0x81, 0x48,
    0xbe, 0x55, 0xb2, 0x38, 0xc0, 0x8f, 0xf4, 0x89, 0x35, 0x73, 0x8c, 0xa6,
    0xe8, 0x40, 0xcc, 0xe0, 0x24, 0xa5, 0x80, 0x5d, 0xe9, 0x21, 0x46, 0x40,
    0x89, 0x93, 0x54, 0x73, 0x80, 0xcf, 0x53, 0xbe, 0x7c, 0x6e, 0x45, 0x5d,
    0xac, 0xed, 0x82, 0x0f, 0xc8, 0xfb, 0x34, 0xe4,
};
static const struct drbg_kat_no_reseed kat188_nor_t = {
    7, kat188_nor_entropyin, kat188_nor_nonce, kat188_nor_persstr,
    kat188_nor_addin0, kat188_nor_addin1, kat188_nor_retbytes
};
static const struct drbg_kat kat188_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat188_nor_t
};

static const unsigned char kat189_nor_entropyin[] = {
    0x61, 0x23, 0xec, 0x40, 0xc4, 0x2f, 0x3b, 0xed, 0x9c, 0xd1, 0x78, 0xd4,
    0x00, 0x4e, 0xe9, 0x1f,
};
static const unsigned char kat189_nor_nonce[] = {
    0x02, 0x64, 0x71, 0x2f, 0x52, 0xed, 0xb2, 0x3d,
};
static const unsigned char kat189_nor_persstr[] = {0};
static const unsigned char kat189_nor_addin0[] = {0};
static const unsigned char kat189_nor_addin1[] = {0};
static const unsigned char kat189_nor_retbytes[] = {
    0xd5, 0x8b, 0x3f, 0x09, 0x1a, 0x96, 0x7c, 0xa1, 0x3b, 0x55, 0x46, 0x3e,
    0x85, 0x29, 0x04, 0x5d, 0x60, 0xa6, 0x0b, 0x96, 0x02, 0xf4, 0x3e, 0xe7,
    0xbe, 0xe0, 0x75, 0x47, 0x1e, 0xef, 0x83, 0x82, 0x3b, 0x2d, 0x54, 0x0b,
    0x3e, 0xce, 0x6d, 0x63, 0x7e, 0x11, 0x53, 0x57, 0x9e, 0x2c, 0x17, 0x9e,
    0xc1, 0x55, 0x25, 0x0a, 0x18, 0xc3, 0xae, 0x46, 0x49, 0x0c, 0x61, 0xe4,
    0x64, 0x31, 0x43, 0x46, 0x4d, 0xce, 0xe9, 0x4c, 0x88, 0xc2, 0xaf, 0xb4,
    0x1b, 0x7d, 0x51, 0x6f, 0x65, 0xf2, 0x15, 0xfc,
};
static const struct drbg_kat_no_reseed kat189_nor_t = {
    8, kat189_nor_entropyin, kat189_nor_nonce, kat189_nor_persstr,
    kat189_nor_addin0, kat189_nor_addin1, kat189_nor_retbytes
};
static const struct drbg_kat kat189_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat189_nor_t
};

static const unsigned char kat190_nor_entropyin[] = {
    0xb1, 0xe6, 0x32, 0x84, 0x94, 0x27, 0xbe, 0xc0, 0x28, 0x97, 0x71, 0x8c,
    0x3d, 0xf8, 0x86, 0x5b,
};
static const unsigned char kat190_nor_nonce[] = {
    0xc8, 0xc1, 0xc6, 0x4b, 0xfd, 0x80, 0xbe, 0x1f,
};
static const unsigned char kat190_nor_persstr[] = {0};
static const unsigned char kat190_nor_addin0[] = {0};
static const unsigned char kat190_nor_addin1[] = {0};
static const unsigned char kat190_nor_retbytes[] = {
    0x49, 0x85, 0x77, 0x77, 0x45, 0xc4, 0xfb, 0x29, 0xd0, 0x19, 0x67, 0x57,
    0x09, 0x37, 0xda, 0x62, 0xa6, 0xc6, 0x44, 0x88, 0x1a, 0x4f, 0xd0, 0x3e,
    0x14, 0x03, 0x99, 0x2a, 0x6a, 0x86, 0xdb, 0xeb, 0x98, 0x44, 0x28, 0x52,
    0x72, 0x88, 0x44, 0x42, 0x9d, 0xad, 0xba, 0xb4, 0x34, 0xa0, 0xb0, 0x35,
    0x94, 0xc8, 0xd6, 0xae, 0x41, 0xfc, 0x2b, 0x4a, 0x9c, 0xed, 0x5f, 0x63,
    0xb4, 0xec, 0xd8, 0xd4, 0x06, 0x4f, 0x80, 0xac, 0xd3, 0xef, 0x50, 0x91,
    0x0c, 0x9d, 0x84, 0xa0, 0x29, 0x0c, 0x26, 0x15,
};
static const struct drbg_kat_no_reseed kat190_nor_t = {
    9, kat190_nor_entropyin, kat190_nor_nonce, kat190_nor_persstr,
    kat190_nor_addin0, kat190_nor_addin1, kat190_nor_retbytes
};
static const struct drbg_kat kat190_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat190_nor_t
};

static const unsigned char kat191_nor_entropyin[] = {
    0xf6, 0xb7, 0x32, 0xcd, 0x0a, 0xa7, 0x20, 0x0a, 0xee, 0x67, 0x84, 0x55,
    0x2d, 0x0c, 0x38, 0xaf,
};
static const unsigned char kat191_nor_nonce[] = {
    0x2d, 0x69, 0xa6, 0x00, 0xa4, 0x26, 0x74, 0xac,
};
static const unsigned char kat191_nor_persstr[] = {0};
static const unsigned char kat191_nor_addin0[] = {0};
static const unsigned char kat191_nor_addin1[] = {0};
static const unsigned char kat191_nor_retbytes[] = {
    0x47, 0xe9, 0x73, 0x8a, 0x0d, 0x0e, 0x09, 0xba, 0x78, 0x6d, 0x5c, 0x9d,
    0x0a, 0x14, 0x47, 0x26, 0x79, 0x43, 0x2a, 0xaa, 0x36, 0x3b, 0xce, 0xb1,
    0x2b, 0xca, 0x2d, 0x85, 0x24, 0x8d, 0x61, 0xcf, 0xb2, 0xe4, 0xb7, 0x12,
    0xea, 0x4c, 0xce, 0xfa, 0x14, 0x82, 0x02, 0x27, 0xdb, 0x3a, 0x31, 0xf0,
    0x63, 0xe9, 0x8c, 0x40, 0xac, 0x21, 0xbb, 0xbf, 0xae, 0x3d, 0x63, 0x26,
    0xc2, 0xfd, 0x59, 0x04, 0x6d, 0x88, 0x64, 0x97, 0x6b, 0xcd, 0x51, 0x78,
    0x27, 0x50, 0x33, 0x62, 0x42, 0x97, 0x48, 0xbb,
};
static const struct drbg_kat_no_reseed kat191_nor_t = {
    10, kat191_nor_entropyin, kat191_nor_nonce, kat191_nor_persstr,
    kat191_nor_addin0, kat191_nor_addin1, kat191_nor_retbytes
};
static const struct drbg_kat kat191_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat191_nor_t
};

static const unsigned char kat192_nor_entropyin[] = {
    0x49, 0xc4, 0x94, 0x01, 0x3b, 0xbe, 0x33, 0x20, 0x85, 0x9f, 0xcc, 0x0c,
    0xf7, 0x3c, 0xd4, 0x72,
};
static const unsigned char kat192_nor_nonce[] = {
    0x83, 0xf1, 0x4a, 0x32, 0x41, 0x9c, 0xda, 0xe2,
};
static const unsigned char kat192_nor_persstr[] = {0};
static const unsigned char kat192_nor_addin0[] = {0};
static const unsigned char kat192_nor_addin1[] = {0};
static const unsigned char kat192_nor_retbytes[] = {
    0xb1, 0xf3, 0xaa, 0x9a, 0x90, 0x44, 0x0b, 0x35, 0xcc, 0xb7, 0x97, 0xf7,
    0xe0, 0xb7, 0xb4, 0xb7, 0x07, 0x15, 0x79, 0x6f, 0x32, 0x04, 0x39, 0x93,
    0x7b, 0x5e, 0x7c, 0xc6, 0x9c, 0xea, 0xc6, 0x69, 0x1d, 0x3c, 0xc2, 0x62,
    0xba, 0xb3, 0x57, 0x08, 0xac, 0x84, 0x5d, 0x9c, 0x44, 0xc5, 0x0a, 0x0d,
    0x45, 0xe2, 0x94, 0x01, 0xe5, 0x2b, 0xe4, 0x22, 0x46, 0x0e, 0x0c, 0x0f,
    0x6b, 0x98, 0x3d, 0xad, 0x6d, 0x5e, 0x61, 0xf7, 0xf6, 0x92, 0x9d, 0x88,
    0x1f, 0x43, 0x03, 0x48, 0xbf, 0x16, 0xde, 0x94,
};
static const struct drbg_kat_no_reseed kat192_nor_t = {
    11, kat192_nor_entropyin, kat192_nor_nonce, kat192_nor_persstr,
    kat192_nor_addin0, kat192_nor_addin1, kat192_nor_retbytes
};
static const struct drbg_kat kat192_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat192_nor_t
};

static const unsigned char kat193_nor_entropyin[] = {
    0x49, 0x8c, 0x61, 0x22, 0x3b, 0xfa, 0xa5, 0x33, 0xda, 0x30, 0x7e, 0x7f,
    0xd7, 0x9b, 0x77, 0x6e,
};
static const unsigned char kat193_nor_nonce[] = {
    0xb3, 0x62, 0xd9, 0x06, 0x63, 0xb1, 0xc9, 0x21,
};
static const unsigned char kat193_nor_persstr[] = {0};
static const unsigned char kat193_nor_addin0[] = {0};
static const unsigned char kat193_nor_addin1[] = {0};
static const unsigned char kat193_nor_retbytes[] = {
    0x80, 0xa2, 0xa0, 0xf2, 0x82, 0x29, 0x85, 0x42, 0x84, 0x71, 0xba, 0xc8,
    0xc5, 0xc4, 0x6c, 0xd5, 0xdb, 0xee, 0x0f, 0x9f, 0xb9, 0x1d, 0x08, 0xb9,
    0xed, 0x69, 0xa5, 0x57, 0x70, 0x01, 0x34, 0x73, 0x87, 0x52, 0x31, 0x0e,
    0xef, 0xc2, 0x3a, 0xce, 0x02, 0x1f, 0xbc, 0xb8, 0x2c, 0xc6, 0x70, 0x96,
    0xb5, 0x8a, 0xae, 0x14, 0x46, 0x36, 0x07, 0xa5, 0x07, 0x70, 0x72, 0xea,
    0x74, 0xaf, 0x4b, 0xfd, 0xe6, 0x8e, 0xe9, 0x1f, 0x89, 0x23, 0x1b, 0xd3,
    0xba, 0xfa, 0xea, 0xaa, 0xd1, 0x40, 0x6b, 0x39,
};
static const struct drbg_kat_no_reseed kat193_nor_t = {
    12, kat193_nor_entropyin, kat193_nor_nonce, kat193_nor_persstr,
    kat193_nor_addin0, kat193_nor_addin1, kat193_nor_retbytes
};
static const struct drbg_kat kat193_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat193_nor_t
};

static const unsigned char kat194_nor_entropyin[] = {
    0x33, 0xf7, 0x24, 0x71, 0x7a, 0xef, 0xf4, 0x85, 0x54, 0x3a, 0xd0, 0xd5,
    0xc7, 0x70, 0x33, 0x56,
};
static const unsigned char kat194_nor_nonce[] = {
    0xac, 0x45, 0x58, 0xa2, 0x2a, 0xae, 0xfc, 0x9d,
};
static const unsigned char kat194_nor_persstr[] = {0};
static const unsigned char kat194_nor_addin0[] = {0};
static const unsigned char kat194_nor_addin1[] = {0};
static const unsigned char kat194_nor_retbytes[] = {
    0x81, 0x0b, 0xe2, 0xed, 0x84, 0xcb, 0x7b, 0xc2, 0x05, 0xf6, 0xba, 0x6d,
    0xd6, 0x91, 0x18, 0x6e, 0x45, 0x4d, 0xa0, 0x73, 0x95, 0x5c, 0xe6, 0xe5,
    0x4f, 0x9b, 0x94, 0x54, 0x45, 0xae, 0x69, 0x8a, 0x3d, 0x47, 0x47, 0xc9,
    0x73, 0x05, 0x2a, 0xa1, 0xcc, 0x7e, 0x21, 0x22, 0x08, 0xec, 0xd4, 0xcc,
    0x58, 0xb2, 0xf3, 0x0f, 0xf6, 0xc4, 0xf2, 0xac, 0xdc, 0xab, 0xc3, 0x72,
    0x7e, 0x86, 0x65, 0xf5, 0x5d, 0xe7, 0x96, 0x2f, 0x6f, 0xcb, 0xbe, 0xb4,
    0x06, 0x86, 0x99, 0x24, 0xc6, 0x16, 0x6a, 0xc2,
};
static const struct drbg_kat_no_reseed kat194_nor_t = {
    13, kat194_nor_entropyin, kat194_nor_nonce, kat194_nor_persstr,
    kat194_nor_addin0, kat194_nor_addin1, kat194_nor_retbytes
};
static const struct drbg_kat kat194_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat194_nor_t
};

static const unsigned char kat195_nor_entropyin[] = {
    0xf4, 0x84, 0xb9, 0x22, 0xf4, 0x92, 0xd1, 0x9b, 0x58, 0x40, 0x7c, 0x24,
    0x2a, 0xb9, 0x0e, 0x76,
};
static const unsigned char kat195_nor_nonce[] = {
    0x89, 0x52, 0xa0, 0xa4, 0xb6, 0x66, 0xb0, 0xc8,
};
static const unsigned char kat195_nor_persstr[] = {0};
static const unsigned char kat195_nor_addin0[] = {0};
static const unsigned char kat195_nor_addin1[] = {0};
static const unsigned char kat195_nor_retbytes[] = {
    0x2d, 0x77, 0x23, 0x5f, 0xa2, 0x73, 0xca, 0xb3, 0xc1, 0xbb, 0x17, 0x6d,
    0x44, 0x81, 0x7c, 0xc2, 0x53, 0x00, 0xb3, 0xf0, 0x17, 0x2a, 0x0b, 0x5a,
    0xaa, 0x66, 0xb2, 0x82, 0xc0, 0x15, 0xd4, 0x26, 0xed, 0xec, 0x5f, 0x1e,
    0xbb, 0xfc, 0x02, 0x69, 0x95, 0x6b, 0x85, 0x99, 0x41, 0x67, 0x99, 0x2a,
    0x71, 0x00, 0x25, 0x86, 0x92, 0x3e, 0xa2, 0x34, 0xbe, 0x6c, 0x5d, 0xf0,
    0x9f, 0x47, 0xd8, 0x91, 0x32, 0xe4, 0x40, 0x82, 0x7b, 0x89, 0xf7, 0xff,
    0x97, 0xe0, 0x32, 0xb3, 0xf7, 0x4f, 0xe3, 0x2f,
};
static const struct drbg_kat_no_reseed kat195_nor_t = {
    14, kat195_nor_entropyin, kat195_nor_nonce, kat195_nor_persstr,
    kat195_nor_addin0, kat195_nor_addin1, kat195_nor_retbytes
};
static const struct drbg_kat kat195_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 0, 80, &kat195_nor_t
};

static const unsigned char kat196_nor_entropyin[] = {
    0x9e, 0x13, 0xa5, 0x7f, 0xb4, 0x3d, 0xdc, 0xd0, 0x69, 0xfc, 0x83, 0x80,
    0xb8, 0x18, 0xf2, 0xae,
};
static const unsigned char kat196_nor_nonce[] = {
    0xfd, 0x09, 0x58, 0x60, 0x67, 0xe5, 0xd9, 0x2c,
};
static const unsigned char kat196_nor_persstr[] = {0};
static const unsigned char kat196_nor_addin0[] = {
    0xb2, 0x25, 0x7b, 0xa7, 0x7d, 0x8a, 0xf7, 0x5d, 0x23, 0xb8, 0x0c, 0x9a,
    0x9c, 0xe7, 0x46, 0x30,
};
static const unsigned char kat196_nor_addin1[] = {
    0x79, 0x77, 0x52, 0x10, 0xd7, 0x49, 0x66, 0x2d, 0x6a, 0x5e, 0xa2, 0xb2,
    0xbb, 0xbf, 0x35, 0x30,
};
static const unsigned char kat196_nor_retbytes[] = {
    0x78, 0xa4, 0x95, 0x79, 0x78, 0x45, 0x0a, 0x7f, 0x8a, 0xd0, 0x0f, 0x3c,
    0x3b, 0x4e, 0x21, 0x87, 0x6f, 0x5a, 0x82, 0x8c, 0xc6, 0x0c, 0xd8, 0xea,
    0xf4, 0x88, 0xea, 0xeb, 0x1b, 0x96, 0x40, 0xbf, 0x49, 0x3a, 0x9c, 0xed,
    0xa1, 0xd3, 0xc7, 0x68, 0x43, 0x4d, 0xeb, 0x20, 0x18, 0x3b, 0x7c, 0x0e,
    0xc8, 0xe7, 0xd7, 0x51, 0xba, 0x90, 0x11, 0xe3, 0x48, 0x95, 0x63, 0xc7,
    0x95, 0xb6, 0x78, 0xbe, 0x5e, 0xa1, 0x36, 0x18, 0xbc, 0xc3, 0xba, 0x54,
    0xfe, 0x59, 0x44, 0x53, 0xc9, 0x5c, 0x88, 0x8b,
};
static const struct drbg_kat_no_reseed kat196_nor_t = {
    0, kat196_nor_entropyin, kat196_nor_nonce, kat196_nor_persstr,
    kat196_nor_addin0, kat196_nor_addin1, kat196_nor_retbytes
};
static const struct drbg_kat kat196_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat196_nor_t
};

static const unsigned char kat197_nor_entropyin[] = {
    0xa4, 0xe8, 0x62, 0x18, 0xfb, 0xeb, 0xac, 0x79, 0xa3, 0x1b, 0x3a, 0xcf,
    0x2a, 0xd3, 0xc6, 0xea,
};
static const unsigned char kat197_nor_nonce[] = {
    0xea, 0x2a, 0xec, 0x42, 0x3f, 0x4d, 0xcf, 0x62,
};
static const unsigned char kat197_nor_persstr[] = {0};
static const unsigned char kat197_nor_addin0[] = {
    0x00, 0x27, 0x7c, 0x54, 0xa4, 0x9f, 0x8d, 0x71, 0xd3, 0x24, 0xc4, 0xf6,
    0xd7, 0x53, 0xab, 0x8b,
};
static const unsigned char kat197_nor_addin1[] = {
    0x8b, 0x56, 0xdd, 0x32, 0xe1, 0x4f, 0xe6, 0x06, 0x89, 0x8f, 0x8f, 0xeb,
    0x4b, 0x5e, 0x05, 0xd6,
};
static const unsigned char kat197_nor_retbytes[] = {
    0xc9, 0x31, 0x06, 0x94, 0xbb, 0xea, 0x39, 0x92, 0x11, 0xec, 0x91, 0xd4,
    0x1c, 0x60, 0xf3, 0x23, 0x13, 0xfc, 0xa2, 0x8a, 0xe5, 0xed, 0xdc, 0x77,
    0x2e, 0x98, 0xd8, 0xf5, 0x92, 0x78, 0x64, 0xad, 0x69, 0xf7, 0x18, 0x00,
    0xd9, 0x33, 0x1e, 0xa4, 0xe5, 0x31, 0xe3, 0xbb, 0x83, 0x56, 0xc6, 0xba,
    0x93, 0x65, 0xac, 0x76, 0xb7, 0x8c, 0x40, 0xff, 0x28, 0xd0, 0xba, 0xd4,
    0xa1, 0x44, 0xb8, 0xd7, 0x4f, 0xe4, 0x56, 0x6e, 0x8d, 0xcf, 0xf0, 0x46,
    0xdc, 0xca, 0xa9, 0x38, 0xae, 0xdb, 0xb2, 0x90,
};
static const struct drbg_kat_no_reseed kat197_nor_t = {
    1, kat197_nor_entropyin, kat197_nor_nonce, kat197_nor_persstr,
    kat197_nor_addin0, kat197_nor_addin1, kat197_nor_retbytes
};
static const struct drbg_kat kat197_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat197_nor_t
};

static const unsigned char kat198_nor_entropyin[] = {
    0xb1, 0x3f, 0x24, 0x09, 0xb5, 0xbc, 0x41, 0x51, 0xca, 0x70, 0x37, 0xf7,
    0xde, 0x83, 0x52, 0x61,
};
static const unsigned char kat198_nor_nonce[] = {
    0x2f, 0x7a, 0xe4, 0x58, 0xde, 0xda, 0x5e, 0x41,
};
static const unsigned char kat198_nor_persstr[] = {0};
static const unsigned char kat198_nor_addin0[] = {
    0xb2, 0x4f, 0x34, 0x9c, 0x35, 0x7a, 0x53, 0xb3, 0x42, 0xeb, 0xe5, 0x31,
    0xe1, 0xa0, 0x40, 0x13,
};
static const unsigned char kat198_nor_addin1[] = {
    0xe8, 0xa5, 0xc6, 0x54, 0x74, 0xa2, 0x7d, 0x2f, 0xd5, 0x0f, 0x6e, 0x6b,
    0xd0, 0x3c, 0xc2, 0x7a,
};
static const unsigned char kat198_nor_retbytes[] = {
    0xb7, 0xd3, 0x33, 0x78, 0x34, 0xf2, 0xe2, 0x23, 0x57, 0x03, 0x00, 0xf1,
    0x6b, 0x9b, 0x5a, 0x1d, 0x1d, 0x7f, 0xda, 0x5e, 0xb5, 0xbe, 0x14, 0xb8,
    0x90, 0x66, 0xdd, 0x9b, 0xe3, 0x9b, 0xab, 0xda, 0xe4, 0x1a, 0xfd, 0xf5,
    0x75, 0x89, 0xe4, 0xc9, 0xec, 0x06, 0x2d, 0xc0, 0xab, 0x0d, 0x62, 0x95,
    0x45, 0x8f, 0xe0, 0x83, 0xdf, 0xa6, 0xc5, 0x51, 0x47, 0xa1, 0x35, 0x65,
    0x62, 0x8d, 0x59, 0x6c, 0x91, 0x3e, 0x28, 0xa8, 0x27, 0x00, 0xd8, 0x9a,
    0x8b, 0xbe, 0x22, 0x6d, 0x33, 0xe0, 0x8e, 0x8b,
};
static const struct drbg_kat_no_reseed kat198_nor_t = {
    2, kat198_nor_entropyin, kat198_nor_nonce, kat198_nor_persstr,
    kat198_nor_addin0, kat198_nor_addin1, kat198_nor_retbytes
};
static const struct drbg_kat kat198_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat198_nor_t
};

static const unsigned char kat199_nor_entropyin[] = {
    0x70, 0xc7, 0x3c, 0x77, 0x62, 0x32, 0xd1, 0x4f, 0x09, 0xdc, 0x1f, 0x84,
    0xaf, 0xad, 0xea, 0xb4,
};
static const unsigned char kat199_nor_nonce[] = {
    0x39, 0xd4, 0x59, 0xb1, 0xa3, 0x61, 0x74, 0x0d,
};
static const unsigned char kat199_nor_persstr[] = {0};
static const unsigned char kat199_nor_addin0[] = {
    0x67, 0x9f, 0x2f, 0xf0, 0xaa, 0xf5, 0x1a, 0x1e, 0x69, 0xac, 0x27, 0x48,
    0x0d, 0x57, 0x69, 0x32,
};
static const unsigned char kat199_nor_addin1[] = {
    0x27, 0xd4, 0x92, 0x0d, 0xfd, 0x26, 0x83, 0xb6, 0x6a, 0x42, 0xad, 0x3d,
    0x32, 0x07, 0x15, 0x47,
};
static const unsigned char kat199_nor_retbytes[] = {
    0xbb, 0x8e, 0x50, 0x08, 0xc1, 0x1c, 0x4d, 0x6e, 0x4c, 0x63, 0x99, 0x6f,
    0x1e, 0x50, 0xed, 0xf4, 0x80, 0xbd, 0x15, 0x5f, 0xd0, 0x46, 0x68, 0x65,
    0x2e, 0xee, 0xd6, 0xa8, 0x71, 0xde, 0x7c, 0x86, 0x05, 0xf2, 0x17, 0xfa,
    0x79, 0x42, 0xe0, 0x3f, 0x3c, 0xc2, 0xb8, 0xbd, 0x1c, 0xa1, 0x8d, 0x6d,
    0xde, 0xf7, 0x3e, 0xd9, 0xec, 0x00, 0x72, 0x03, 0x6a, 0x34, 0x16, 0x10,
    0x2c, 0x60, 0xf8, 0x3d, 0xeb, 0xeb, 0xac, 0xff, 0xe9, 0xfd, 0xeb, 0xe2,
    0xa4, 0x78, 0xc2, 0xc7, 0xca, 0x68, 0xab, 0xb6,
};
static const struct drbg_kat_no_reseed kat199_nor_t = {
    3, kat199_nor_entropyin, kat199_nor_nonce, kat199_nor_persstr,
    kat199_nor_addin0, kat199_nor_addin1, kat199_nor_retbytes
};
static const struct drbg_kat kat199_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat199_nor_t
};

static const unsigned char kat200_nor_entropyin[] = {
    0xb8, 0x38, 0xbb, 0x13, 0x6a, 0xd9, 0xb6, 0xf2, 0xb7, 0x91, 0x3f, 0xf4,
    0x1a, 0xf9, 0xaf, 0x40,
};
static const unsigned char kat200_nor_nonce[] = {
    0x43, 0xcc, 0xad, 0x7a, 0x42, 0x27, 0xad, 0x7c,
};
static const unsigned char kat200_nor_persstr[] = {0};
static const unsigned char kat200_nor_addin0[] = {
    0x09, 0xdb, 0x57, 0x81, 0x70, 0x51, 0xcc, 0x2e, 0x19, 0x80, 0x6d, 0x18,
    0xb0, 0xa9, 0x08, 0x4f,
};
static const unsigned char kat200_nor_addin1[] = {
    0xf8, 0xeb, 0xc3, 0x54, 0x4a, 0x5c, 0xd5, 0x6e, 0x0b, 0x61, 0x36, 0xc9,
    0xc3, 0x42, 0x84, 0x99,
};
static const unsigned char kat200_nor_retbytes[] = {
    0x20, 0x60, 0xe3, 0x92, 0x27, 0x28, 0x70, 0x3c, 0x94, 0x7e, 0x0d, 0x7d,
    0x94, 0x37, 0x4e, 0x97, 0x41, 0xf4, 0x33, 0xca, 0x24, 0x4c, 0x6c, 0xd3,
    0x96, 0xdb, 0x88, 0x4a, 0xa4, 0x1f, 0x6a, 0x75, 0xd8, 0x34, 0xfe, 0x76,
    0x3c, 0x8c, 0x87, 0x3a, 0x96, 0x0f, 0xf3, 0x01, 0x58, 0x6c, 0x87, 0x01,
    0x27, 0x21, 0xd8, 0xbe, 0x1b, 0xc0, 0x22, 0x68, 0x84, 0xc2, 0x35, 0x32,
    0x1b, 0x81, 0x8e, 0xc1, 0x95, 0x94, 0x0d, 0xcb, 0x5c, 0x21, 0xbf, 0xe9,
    0x59, 0x36, 0xec, 0x2f, 0xe4, 0xa5, 0xde, 0xef,
};
static const struct drbg_kat_no_reseed kat200_nor_t = {
    4, kat200_nor_entropyin, kat200_nor_nonce, kat200_nor_persstr,
    kat200_nor_addin0, kat200_nor_addin1, kat200_nor_retbytes
};
static const struct drbg_kat kat200_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat200_nor_t
};

static const unsigned char kat201_nor_entropyin[] = {
    0xe4, 0xf6, 0x9f, 0x55, 0x8d, 0xbb, 0xcf, 0x15, 0x76, 0x8e, 0x21, 0xe2,
    0x76, 0xc7, 0x38, 0x1c,
};
static const unsigned char kat201_nor_nonce[] = {
    0x6d, 0x14, 0xd8, 0xcd, 0xc6, 0xbb, 0xe6, 0xbb,
};
static const unsigned char kat201_nor_persstr[] = {0};
static const unsigned char kat201_nor_addin0[] = {
    0xdb, 0xf7, 0x65, 0xff, 0x20, 0x4e, 0xe3, 0x2c, 0x53, 0x82, 0xa3, 0x9d,
    0xd6, 0xaf, 0xf8, 0x70,
};
static const unsigned char kat201_nor_addin1[] = {
    0x4f, 0xa9, 0x01, 0x68, 0xa5, 0xa0, 0x07, 0x30, 0x90, 0xe4, 0xd4, 0xaf,
    0x43, 0x2c, 0x39, 0xa4,
};
static const unsigned char kat201_nor_retbytes[] = {
    0xe6, 0x54, 0x01, 0xce, 0x0e, 0xc9, 0xf0, 0xd2, 0x47, 0xd5, 0x0c, 0xff,
    0x3a, 0xd0, 0x70, 0xbb, 0x1f, 0x3c, 0x9f, 0x9e, 0x81, 0xe7, 0x23, 0x7c,
    0xa0, 0x53, 0x9b, 0x82, 0xe4, 0xc4, 0xd3, 0x62, 0x98, 0x3b, 0xb5, 0xad,
    0xc3, 0xd2, 0xd9, 0x8d, 0xbc, 0x9a, 0x21, 0xa3, 0x35, 0x1e, 0x17, 0xfd,
    0x78, 0x90, 0xef, 0x8f, 0xef, 0x72, 0x68, 0xf5, 0xc0, 0x03, 0x1f, 0xd4,
    0x9d, 0xf5, 0x63, 0xab, 0x67, 0x71, 0x93, 0x9d, 0xd2, 0xbc, 0xd6, 0xd6,
    0x5c, 0x30, 0x74, 0x85, 0xc0, 0xce, 0x1a, 0xf5,
};
static const struct drbg_kat_no_reseed kat201_nor_t = {
    5, kat201_nor_entropyin, kat201_nor_nonce, kat201_nor_persstr,
    kat201_nor_addin0, kat201_nor_addin1, kat201_nor_retbytes
};
static const struct drbg_kat kat201_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat201_nor_t
};

static const unsigned char kat202_nor_entropyin[] = {
    0xd7, 0xa3, 0x2b, 0x55, 0x97, 0xd9, 0x99, 0x4f, 0xca, 0xdd, 0x9d, 0x94,
    0x56, 0x55, 0x23, 0xa8,
};
static const unsigned char kat202_nor_nonce[] = {
    0x5f, 0x47, 0x25, 0x4c, 0x88, 0xb7, 0x10, 0x55,
};
static const unsigned char kat202_nor_persstr[] = {0};
static const unsigned char kat202_nor_addin0[] = {
    0x1e, 0xbf, 0xd0, 0xd0, 0xa1, 0x3f, 0xfb, 0x06, 0xdd, 0x3b, 0xc3, 0x19,
    0x2f, 0x55, 0xb2, 0xc6,
};
static const unsigned char kat202_nor_addin1[] = {
    0x80, 0x4f, 0xc1, 0x77, 0x2a, 0xb3, 0x93, 0xc7, 0x4d, 0x7c, 0xca, 0x2e,
    0x56, 0x6e, 0xf7, 0x0f,
};
static const unsigned char kat202_nor_retbytes[] = {
    0xeb, 0x26, 0x9c, 0x0f, 0x2d, 0x02, 0xab, 0x44, 0x48, 0xf5, 0x1e, 0x04,
    0x21, 0xdf, 0x97, 0x01, 0xf2, 0x7e, 0x9d, 0xa7, 0x7e, 0x75, 0xb2, 0xe7,
    0x60, 0x3c, 0xf4, 0x21, 0x52, 0xa8, 0xed, 0xc5, 0xfc, 0x29, 0x2f, 0xa9,
    0xd4, 0x04, 0x6b, 0x22, 0xf7, 0x84, 0x2b, 0xfa, 0x80, 0xd4, 0xa0, 0x33,
    0x4c, 0x79, 0xe0, 0xc8, 0x38, 0x9a, 0x9c, 0xcd, 0x94, 0xf3, 0x4b, 0x9c,
    0x6a, 0x18, 0x8b, 0x09, 0x99, 0x6b, 0xd6, 0xd1, 0xe0, 0xeb, 0xe7, 0x25,
    0x39, 0xf6, 0xa2, 0x5a, 0x5c, 0xc7, 0x06, 0xf5,
};
static const struct drbg_kat_no_reseed kat202_nor_t = {
    6, kat202_nor_entropyin, kat202_nor_nonce, kat202_nor_persstr,
    kat202_nor_addin0, kat202_nor_addin1, kat202_nor_retbytes
};
static const struct drbg_kat kat202_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat202_nor_t
};

static const unsigned char kat203_nor_entropyin[] = {
    0xb6, 0x80, 0x1a, 0xb7, 0x28, 0x56, 0x2a, 0xcb, 0xd7, 0x1a, 0x2e, 0x21,
    0x57, 0xea, 0xa5, 0xb1,
};
static const unsigned char kat203_nor_nonce[] = {
    0x2e, 0xf6, 0x9d, 0x68, 0xe4, 0x0b, 0x3d, 0x82,
};
static const unsigned char kat203_nor_persstr[] = {0};
static const unsigned char kat203_nor_addin0[] = {
    0x9b, 0xab, 0x4d, 0xe8, 0x29, 0x5a, 0x5e, 0xe6, 0xe1, 0x5c, 0xc4, 0x96,
    0x2f, 0x08, 0x96, 0x31,
};
static const unsigned char kat203_nor_addin1[] = {
    0xaa, 0xaa, 0x36, 0x77, 0x01, 0x55, 0x28, 0x19, 0xc4, 0xa2, 0x33, 0x16,
    0x65, 0xbb, 0xf0, 0x61,
};
static const unsigned char kat203_nor_retbytes[] = {
    0x09, 0x1e, 0xdf, 0xd8, 0x89, 0x48, 0xe7, 0x49, 0xb6, 0xab, 0xa6, 0xdd,
    0x70, 0x07, 0x7c, 0x88, 0xf1, 0xea, 0x29, 0x7b, 0xce, 0xdc, 0xd9, 0xe1,
    0xf0, 0xda, 0x80, 0xe5, 0x3e, 0xc1, 0x51, 0x7f, 0x08, 0x78, 0x6f, 0x2e,
    0x86, 0x45, 0x17, 0xa2, 0x75, 0x4c, 0x5c, 0xd1, 0x50, 0x08, 0xd6, 0xc0,
    0x3a, 0xb9, 0xff, 0x2d, 0x05, 0x88, 0x80, 0x86, 0x9d, 0x8b, 0x82, 0x47,
    0xbe, 0x79, 0xa0, 0x11, 0x33, 0xc5, 0x56, 0xa5, 0x15, 0xfd, 0xe2, 0xb2,
    0x11, 0x92, 0x69, 0xc7, 0xde, 0x06, 0xb4, 0x73,
};
static const struct drbg_kat_no_reseed kat203_nor_t = {
    7, kat203_nor_entropyin, kat203_nor_nonce, kat203_nor_persstr,
    kat203_nor_addin0, kat203_nor_addin1, kat203_nor_retbytes
};
static const struct drbg_kat kat203_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat203_nor_t
};

static const unsigned char kat204_nor_entropyin[] = {
    0xee, 0x5a, 0xff, 0xe5, 0xe3, 0x15, 0xdb, 0x84, 0xf7, 0x1a, 0x22, 0x0b,
    0x31, 0x58, 0x88, 0xbb,
};
static const unsigned char kat204_nor_nonce[] = {
    0xf5, 0xe5, 0xf7, 0xaa, 0x35, 0xea, 0xe6, 0x29,
};
static const unsigned char kat204_nor_persstr[] = {0};
static const unsigned char kat204_nor_addin0[] = {
    0x6d, 0x3e, 0x8f, 0xd6, 0x87, 0x0d, 0x2f, 0xcb, 0x22, 0x38, 0x1c, 0x19,
    0x58, 0xe3, 0x87, 0x5e,
};
static const unsigned char kat204_nor_addin1[] = {
    0x8a, 0x3a, 0x40, 0x93, 0xfe, 0x50, 0x0f, 0xea, 0xa1, 0xf7, 0xe2, 0xde,
    0x9f, 0xc5, 0x6b, 0xc0,
};
static const unsigned char kat204_nor_retbytes[] = {
    0x27, 0xe7, 0x9a, 0x6d, 0xe3, 0xb9, 0x99, 0x71, 0x18, 0x2e, 0xab, 0x73,
    0xe3, 0x21, 0x68, 0x3e, 0x2a, 0x60, 0x1d, 0xee, 0x46, 0x31, 0x45, 0x31,
    0xef, 0x8a, 0x04, 0x00, 0x03, 0x7b, 0x1a, 0x66, 0x0e, 0x2c, 0xf0, 0x75,
    0x9d, 0xf0, 0xf6, 0xd2, 0xaf, 0xca, 0x9a, 0x0d, 0xec, 0x50, 0x2b, 0x46,
    0x6d, 0x95, 0xaf, 0xec, 0x52, 0xb8, 0x97, 0x41, 0x24, 0x2d, 0xad, 0xa5,
    0x20, 0x28, 0x06, 0xec, 0xe7, 0x39, 0x96, 0xdf, 0x88, 0xf3, 0x5b, 0xf7,
    0xc6, 0x86, 0xa0, 0xf6, 0x97, 0x32, 0xfa, 0x9e,
};
static const struct drbg_kat_no_reseed kat204_nor_t = {
    8, kat204_nor_entropyin, kat204_nor_nonce, kat204_nor_persstr,
    kat204_nor_addin0, kat204_nor_addin1, kat204_nor_retbytes
};
static const struct drbg_kat kat204_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat204_nor_t
};

static const unsigned char kat205_nor_entropyin[] = {
    0x28, 0xf1, 0x5b, 0x1e, 0xcf, 0x1e, 0x8f, 0xbe, 0xb4, 0xeb, 0xb2, 0x43,
    0xe8, 0x6e, 0x51, 0x0e,
};
static const unsigned char kat205_nor_nonce[] = {
    0xa1, 0x49, 0x01, 0x1c, 0xa0, 0x96, 0x2b, 0x79,
};
static const unsigned char kat205_nor_persstr[] = {0};
static const unsigned char kat205_nor_addin0[] = {
    0xc5, 0x44, 0x25, 0xd4, 0x3b, 0x1e, 0x66, 0x35, 0x46, 0x7c, 0x51, 0x58,
    0xab, 0x57, 0x05, 0x96,
};
static const unsigned char kat205_nor_addin1[] = {
    0x79, 0x4c, 0x9b, 0x26, 0x04, 0xba, 0x24, 0x02, 0xa4, 0x76, 0x38, 0xdc,
    0x73, 0x60, 0xac, 0x29,
};
static const unsigned char kat205_nor_retbytes[] = {
    0x48, 0x71, 0x19, 0xe5, 0xdb, 0x7e, 0x5b, 0x6e, 0x57, 0x92, 0xac, 0x25,
    0xeb, 0xc0, 0x55, 0xe9, 0x1b, 0xa8, 0x27, 0xcd, 0x69, 0x5b, 0x5b, 0x0f,
    0x95, 0x91, 0x85, 0xd9, 0xbb, 0x62, 0xe2, 0x7a, 0x8f, 0x6c, 0x81, 0x4a,
    0xfc, 0x39, 0xb2, 0x6f, 0x94, 0x87, 0x87, 0x07, 0xae, 0x2d, 0x67, 0xd0,
    0x18, 0x77, 0xea, 0x26, 0x7f, 0xcc, 0x55, 0xd5, 0xdf, 0x83, 0xb2, 0xd8,
    0x09, 0x9a, 0x09, 0x3e, 0x95, 0x2a, 0x43, 0xc5, 0xd8, 0xde, 0x99, 0x25,
    0xe5, 0xa0, 0xbb, 0xda, 0xfd, 0x81, 0x37, 0x00,
};
static const struct drbg_kat_no_reseed kat205_nor_t = {
    9, kat205_nor_entropyin, kat205_nor_nonce, kat205_nor_persstr,
    kat205_nor_addin0, kat205_nor_addin1, kat205_nor_retbytes
};
static const struct drbg_kat kat205_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat205_nor_t
};

static const unsigned char kat206_nor_entropyin[] = {
    0xae, 0x23, 0xad, 0xa4, 0x5b, 0xa1, 0xc1, 0xc4, 0x91, 0x18, 0x53, 0x11,
    0xaa, 0x99, 0x4b, 0x18,
};
static const unsigned char kat206_nor_nonce[] = {
    0x03, 0xa1, 0x57, 0xc7, 0xa7, 0x4a, 0x20, 0xd2,
};
static const unsigned char kat206_nor_persstr[] = {0};
static const unsigned char kat206_nor_addin0[] = {
    0x09, 0xf3, 0x7d, 0x8e, 0x0c, 0xe2, 0x3d, 0x2e, 0xd9, 0xa8, 0x20, 0xc4,
    0x1c, 0x9c, 0x06, 0xee,
};
static const unsigned char kat206_nor_addin1[] = {
    0xf3, 0x8d, 0xd1, 0x6e, 0x2f, 0xa8, 0xd5, 0x2c, 0x17, 0x08, 0xfc, 0x38,
    0x69, 0xbd, 0x9a, 0x09,
};
static const unsigned char kat206_nor_retbytes[] = {
    0xd4, 0x5f, 0x27, 0xd9, 0x1e, 0x07, 0x7b, 0x9f, 0xa0, 0x60, 0xc6, 0xa7,
    0xd7, 0x07, 0xc0, 0x7c, 0x0e, 0xa3, 0xcb, 0xdb, 0x80, 0x19, 0x4f, 0x08,
    0xfe, 0xa6, 0x4d, 0x31, 0x77, 0x15, 0xce, 0xf7, 0x6d, 0xc8, 0x8f, 0xef,
    0x48, 0x35, 0x6c, 0x1c, 0xf9, 0xf9, 0x9e, 0x56, 0x44, 0xe6, 0xc5, 0x87,
    0x2e, 0x79, 0x37, 0xdc, 0x1e, 0x2c, 0x9f, 0xfe, 0x0d, 0xfa, 0xa8, 0x7e,
    0x42, 0x51, 0x08, 0xd9, 0x7f, 0x65, 0x16, 0x6a, 0xc2, 0xc1, 0x10, 0xed,
    0x0c, 0xd0, 0x7f, 0xac, 0xba, 0x77, 0x36, 0x73,
};
static const struct drbg_kat_no_reseed kat206_nor_t = {
    10, kat206_nor_entropyin, kat206_nor_nonce, kat206_nor_persstr,
    kat206_nor_addin0, kat206_nor_addin1, kat206_nor_retbytes
};
static const struct drbg_kat kat206_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat206_nor_t
};

static const unsigned char kat207_nor_entropyin[] = {
    0xb0, 0x05, 0x0c, 0xbc, 0x41, 0x2b, 0x31, 0x22, 0xd8, 0x91, 0x5b, 0x79,
    0x53, 0x21, 0x61, 0x4c,
};
static const unsigned char kat207_nor_nonce[] = {
    0xa0, 0x98, 0xbf, 0xa0, 0x17, 0xd5, 0xee, 0xb4,
};
static const unsigned char kat207_nor_persstr[] = {0};
static const unsigned char kat207_nor_addin0[] = {
    0x75, 0x81, 0x6d, 0x4a, 0xd2, 0x58, 0x1f, 0x92, 0xbc, 0xe5, 0xe6, 0x79,
    0xbb, 0xcf, 0xff, 0x53,
};
static const unsigned char kat207_nor_addin1[] = {
    0x7d, 0x40, 0xde, 0x16, 0x2b, 0x81, 0x21, 0x31, 0x91, 0x54, 0x08, 0xf9,
    0x79, 0xbf, 0x1a, 0xc8,
};
static const unsigned char kat207_nor_retbytes[] = {
    0x74, 0xcf, 0x8c, 0x5d, 0x35, 0xdc, 0xfc, 0xf4, 0x72, 0xf0, 0x84, 0x2b,
    0x46, 0x5f, 0xdd, 0x13, 0x6f, 0x12, 0xb4, 0x05, 0xee, 0x35, 0x6e, 0xf4,
    0x80, 0x94, 0x3e, 0x70, 0x4c, 0xbb, 0x12, 0xa8, 0xf5, 0x68, 0xd7, 0xe1,
    0xc1, 0x31, 0xe2, 0x91, 0xa4, 0xe8, 0x2a, 0x01, 0x8b, 0x66, 0x7b, 0xb5,
    0xc8, 0x12, 0x74, 0x55, 0x14, 0xc9, 0x02, 0xe7, 0x13, 0xc9, 0x35, 0x26,
    0x53, 0x69, 0x6e, 0xe0, 0x7d, 0xf6, 0xa5, 0x52, 0x59, 0x8d, 0x04, 0x73,
    0x46, 0x0b, 0xb3, 0x8f, 0xe1, 0xd6, 0x02, 0xc8,
};
static const struct drbg_kat_no_reseed kat207_nor_t = {
    11, kat207_nor_entropyin, kat207_nor_nonce, kat207_nor_persstr,
    kat207_nor_addin0, kat207_nor_addin1, kat207_nor_retbytes
};
static const struct drbg_kat kat207_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat207_nor_t
};

static const unsigned char kat208_nor_entropyin[] = {
    0xfa, 0xab, 0x57, 0xf3, 0x12, 0x39, 0xc2, 0x2e, 0x88, 0x08, 0xc2, 0x46,
    0x59, 0x6c, 0x1f, 0x5b,
};
static const unsigned char kat208_nor_nonce[] = {
    0x03, 0x87, 0xf1, 0x63, 0x65, 0x43, 0xa6, 0x61,
};
static const unsigned char kat208_nor_persstr[] = {0};
static const unsigned char kat208_nor_addin0[] = {
    0xd1, 0xb1, 0xea, 0xc5, 0x81, 0x01, 0xb5, 0x60, 0x82, 0xf6, 0xcf, 0x3f,
    0x03, 0xba, 0x3e, 0x6b,
};
static const unsigned char kat208_nor_addin1[] = {
    0x13, 0xb0, 0xb5, 0xba, 0xad, 0x29, 0x9e, 0x76, 0xc7, 0x32, 0xdd, 0x79,
    0x87, 0x8b, 0x7e, 0x4a,
};
static const unsigned char kat208_nor_retbytes[] = {
    0x1a, 0x72, 0x30, 0xd1, 0x4e, 0xda, 0x7f, 0x2f, 0x96, 0xe3, 0x0e, 0x9b,
    0x60, 0x13, 0x7b, 0xa3, 0x3a, 0xa4, 0xc5, 0xbe, 0x04, 0x72, 0x88, 0x04,
    0x37, 0xbc, 0xc6, 0x1c, 0xa8, 0x2d, 0x1e, 0x08, 0x28, 0x91, 0x8b, 0x40,
    0x62, 0xff, 0x7d, 0xee, 0x89, 0x12, 0x8a, 0x48, 0xed, 0x87, 0xac, 0x60,
    0x65, 0x70, 0x51, 0xe0, 0xee, 0xb4, 0xcb, 0xf3, 0x72, 0xa2, 0x4b, 0xef,
    0xff, 0x4d, 0x6e, 0x22, 0xe7, 0x3d, 0x3d, 0xfb, 0x34, 0xd5, 0x9d, 0x21,
    0x3e, 0x05, 0xcc, 0x09, 0x9d, 0x48, 0xbb, 0xda,
};
static const struct drbg_kat_no_reseed kat208_nor_t = {
    12, kat208_nor_entropyin, kat208_nor_nonce, kat208_nor_persstr,
    kat208_nor_addin0, kat208_nor_addin1, kat208_nor_retbytes
};
static const struct drbg_kat kat208_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat208_nor_t
};

static const unsigned char kat209_nor_entropyin[] = {
    0x93, 0xd7, 0x71, 0xd0, 0x7a, 0xe9, 0x8d, 0x78, 0x7f, 0xdb, 0x78, 0x7f,
    0x30, 0xab, 0xd7, 0xd4,
};
static const unsigned char kat209_nor_nonce[] = {
    0x38, 0x0d, 0x74, 0x3d, 0xcf, 0x70, 0xa7, 0x8f,
};
static const unsigned char kat209_nor_persstr[] = {0};
static const unsigned char kat209_nor_addin0[] = {
    0xe8, 0xb4, 0xb8, 0x33, 0xd8, 0x65, 0x73, 0x64, 0xb8, 0xea, 0x4a, 0x67,
    0x04, 0xfb, 0x73, 0x2f,
};
static const unsigned char kat209_nor_addin1[] = {
    0x61, 0x58, 0x53, 0xa9, 0xe6, 0xe5, 0x9b, 0xe8, 0x35, 0x3c, 0xb2, 0x3e,
    0xd9, 0xb0, 0x68, 0xc0,
};
static const unsigned char kat209_nor_retbytes[] = {
    0x2a, 0xea, 0x25, 0x55, 0xd2, 0x58, 0x35, 0xce, 0xb3, 0x42, 0x5f, 0x58,
    0x4b, 0xa1, 0x1f, 0x20, 0xf2, 0x37, 0xe0, 0xa6, 0xaf, 0x2f, 0x55, 0x4b,
    0x38, 0x24, 0x11, 0xec, 0xb0, 0xff, 0xa8, 0xd6, 0xd5, 0x50, 0x4b, 0x65,
    0x44, 0x67, 0xba, 0x34, 0xa0, 0x43, 0x6d, 0xae, 0x4a, 0x9d, 0x48, 0x3f,
    0x6f, 0x58, 0xa5, 0x2d, 0x73, 0x84, 0x45, 0x86, 0xbc, 0xe2, 0xd1, 0x40,
    0x6f, 0x4d, 0x5f, 0x9b, 0x87, 0x8c, 0x4d, 0x14, 0x60, 0xef, 0x9c, 0x97,
    0x4c, 0xac, 0x1c, 0xe0, 0xeb, 0xe8, 0x62, 0x7d,
};
static const struct drbg_kat_no_reseed kat209_nor_t = {
    13, kat209_nor_entropyin, kat209_nor_nonce, kat209_nor_persstr,
    kat209_nor_addin0, kat209_nor_addin1, kat209_nor_retbytes
};
static const struct drbg_kat kat209_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat209_nor_t
};

static const unsigned char kat210_nor_entropyin[] = {
    0x7d, 0x7b, 0x1c, 0x32, 0x5e, 0x98, 0x43, 0xc3, 0xd2, 0x77, 0x0e, 0xfd,
    0x5e, 0x13, 0x4f, 0xe7,
};
static const unsigned char kat210_nor_nonce[] = {
    0x9d, 0xac, 0x46, 0xa2, 0xda, 0x81, 0x03, 0x0f,
};
static const unsigned char kat210_nor_persstr[] = {0};
static const unsigned char kat210_nor_addin0[] = {
    0x9e, 0x3e, 0xa6, 0xea, 0xc1, 0x20, 0xd6, 0x63, 0xe3, 0x30, 0xd2, 0x82,
    0xca, 0x9b, 0x9d, 0x7c,
};
static const unsigned char kat210_nor_addin1[] = {
    0xb8, 0xd7, 0x1f, 0xce, 0x77, 0x79, 0xa9, 0x90, 0x6b, 0x97, 0x90, 0xcd,
    0x1d, 0x4e, 0x48, 0xd5,
};
static const unsigned char kat210_nor_retbytes[] = {
    0x63, 0xd2, 0x8a, 0x30, 0x0a, 0x32, 0x9c, 0xa2, 0x02, 0xb9, 0x84, 0x98,
    0xc9, 0xf4, 0x69, 0x12, 0x62, 0x0b, 0xc8, 0x5c, 0x24, 0x6f, 0x03, 0x4d,
    0xca, 0x41, 0x86, 0xcd, 0x9b, 0x0e, 0x08, 0x10, 0xa3, 0x63, 0x78, 0x58,
    0x78, 0xef, 0xfd, 0xe9, 0x0a, 0xec, 0x8c, 0xb5, 0x84, 0x86, 0x25, 0x24,
    0xee, 0xbf, 0x94, 0x0c, 0x44, 0xfe, 0xd2, 0x1c, 0xb5, 0x80, 0xd4, 0x11,
    0x5f, 0x3e, 0x0d, 0xda, 0x07, 0xe0, 0xe4, 0xa6, 0x66, 0x89, 0xc2, 0xff,
    0x3e, 0x9b, 0x87, 0xed, 0xfa, 0xa4, 0xd0, 0x51,
};
static const struct drbg_kat_no_reseed kat210_nor_t = {
    14, kat210_nor_entropyin, kat210_nor_nonce, kat210_nor_persstr,
    kat210_nor_addin0, kat210_nor_addin1, kat210_nor_retbytes
};
static const struct drbg_kat kat210_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 0, 16, 80, &kat210_nor_t
};

static const unsigned char kat211_nor_entropyin[] = {
    0xe4, 0xba, 0xc9, 0x55, 0xa3, 0xe8, 0x20, 0xe5, 0x89, 0x51, 0x94, 0xe2,
    0xc1, 0x3e, 0xf9, 0xac,
};
static const unsigned char kat211_nor_nonce[] = {
    0x8f, 0x4d, 0x17, 0x7d, 0xa6, 0x5e, 0xd4, 0x8e,
};
static const unsigned char kat211_nor_persstr[] = {
    0x06, 0xc4, 0xc6, 0x01, 0x14, 0x5a, 0x9d, 0x13, 0x7a, 0x49, 0x0c, 0x7b,
    0xcd, 0x22, 0xf4, 0x1e,
};
static const unsigned char kat211_nor_addin0[] = {0};
static const unsigned char kat211_nor_addin1[] = {0};
static const unsigned char kat211_nor_retbytes[] = {
    0xe9, 0x67, 0xd2, 0xfa, 0x24, 0x78, 0x77, 0x53, 0x42, 0xec, 0x5c, 0xa1,
    0x53, 0xe5, 0xcb, 0x13, 0xc6, 0x61, 0x34, 0xd4, 0xe1, 0x7a, 0x89, 0xcf,
    0x62, 0x9d, 0x78, 0x45, 0xf8, 0xce, 0x26, 0x83, 0x32, 0xa0, 0x4f, 0x0d,
    0x7e, 0x22, 0x82, 0x36, 0x8f, 0x40, 0x33, 0x99, 0x4f, 0x20, 0xbc, 0xbc,
    0x7d, 0x43, 0x97, 0xd5, 0xfa, 0xcc, 0x44, 0x88, 0x42, 0xb1, 0xe9, 0x24,
    0x6c, 0x24, 0xa6, 0x8e, 0x02, 0x9f, 0xff, 0x99, 0xe9, 0x08, 0x53, 0xbb,
    0x19, 0xb7, 0xe6, 0xfc, 0x9f, 0x29, 0xe4, 0xb0,
};
static const struct drbg_kat_no_reseed kat211_nor_t = {
    0, kat211_nor_entropyin, kat211_nor_nonce, kat211_nor_persstr,
    kat211_nor_addin0, kat211_nor_addin1, kat211_nor_retbytes
};
static const struct drbg_kat kat211_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat211_nor_t
};

static const unsigned char kat212_nor_entropyin[] = {
    0x2e, 0x95, 0x18, 0xf7, 0x53, 0xf5, 0x9a, 0x4a, 0x53, 0xfd, 0xac, 0x88,
    0xaf, 0x18, 0x7d, 0xac,
};
static const unsigned char kat212_nor_nonce[] = {
    0x9b, 0x92, 0x6c, 0x22, 0x3a, 0xd3, 0xef, 0xb0,
};
static const unsigned char kat212_nor_persstr[] = {
    0xbc, 0xa0, 0x5f, 0xfd, 0xbc, 0x7d, 0x8b, 0x92, 0x8c, 0x79, 0x79, 0xc0,
    0x83, 0xd6, 0x57, 0x84,
};
static const unsigned char kat212_nor_addin0[] = {0};
static const unsigned char kat212_nor_addin1[] = {0};
static const unsigned char kat212_nor_retbytes[] = {
    0x05, 0x5f, 0xbe, 0x1d, 0x5e, 0x27, 0x60, 0x68, 0xf7, 0x88, 0x30, 0x94,
    0xc3, 0xdf, 0x67, 0x9d, 0xa1, 0xbf, 0xcf, 0x38, 0x56, 0x2e, 0xbb, 0xff,
    0x64, 0x45, 0x6c, 0x2e, 0xd4, 0xfb, 0x03, 0x98, 0x5b, 0xdd, 0xfb, 0x35,
    0x21, 0x22, 0xad, 0xbb, 0x0b, 0xdd, 0x92, 0x67, 0x2d, 0x45, 0x4f, 0x7d,
    0xb5, 0x16, 0x20, 0x9c, 0xfa, 0xfe, 0x28, 0x9b, 0xd2, 0x8b, 0xdb, 0xc5,
    0x0b, 0xc2, 0x2f, 0x03, 0x91, 0x2e, 0xa6, 0x94, 0xad, 0x2f, 0xe1, 0xc9,
    0xf2, 0xd5, 0x7e, 0xd3, 0x98, 0x93, 0xe8, 0x01,
};
static const struct drbg_kat_no_reseed kat212_nor_t = {
    1, kat212_nor_entropyin, kat212_nor_nonce, kat212_nor_persstr,
    kat212_nor_addin0, kat212_nor_addin1, kat212_nor_retbytes
};
static const struct drbg_kat kat212_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat212_nor_t
};

static const unsigned char kat213_nor_entropyin[] = {
    0x29, 0x9f, 0xb6, 0x49, 0xae, 0x00, 0x2b, 0x0b, 0xd2, 0x9f, 0x23, 0xfb,
    0xeb, 0x22, 0x1f, 0x08,
};
static const unsigned char kat213_nor_nonce[] = {
    0x61, 0x40, 0xec, 0x1a, 0x58, 0x99, 0xeb, 0xab,
};
static const unsigned char kat213_nor_persstr[] = {
    0xac, 0xd2, 0x16, 0x0d, 0x08, 0x85, 0x32, 0xe0, 0x58, 0x8a, 0x88, 0x8c,
    0x5d, 0xba, 0x86, 0x3b,
};
static const unsigned char kat213_nor_addin0[] = {0};
static const unsigned char kat213_nor_addin1[] = {0};
static const unsigned char kat213_nor_retbytes[] = {
    0x16, 0x30, 0xfc, 0xb4, 0x82, 0x95, 0x6b, 0x0f, 0x9a, 0x1f, 0x2c, 0x24,
    0x8d, 0x06, 0xd8, 0x06, 0x96, 0x69, 0xab, 0x2c, 0x90, 0xaa, 0x39, 0x0d,
    0xa6, 0x36, 0x95, 0x49, 0x45, 0x31, 0x26, 0x07, 0xd6, 0x45, 0xa6, 0xae,
    0x86, 0x4c, 0x3e, 0x3f, 0x82, 0x3e, 0x2a, 0xac, 0x90, 0x00, 0x75, 0x72,
    0x85, 0x48, 0xd7, 0x84, 0xac, 0xa3, 0x51, 0xf1, 0xf0, 0x98, 0xfb, 0x40,
    0x1f, 0xd7, 0xe4, 0x14, 0x1e, 0x2a, 0x37, 0x4c, 0x8f, 0x25, 0x2b, 0xd7,
    0x2a, 0xa8, 0xf5, 0x5d, 0xc0, 0x14, 0xa6, 0x8c,
};
static const struct drbg_kat_no_reseed kat213_nor_t = {
    2, kat213_nor_entropyin, kat213_nor_nonce, kat213_nor_persstr,
    kat213_nor_addin0, kat213_nor_addin1, kat213_nor_retbytes
};
static const struct drbg_kat kat213_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat213_nor_t
};

static const unsigned char kat214_nor_entropyin[] = {
    0xbf, 0x95, 0x08, 0x82, 0x8a, 0x6d, 0xf3, 0x9c, 0xe1, 0x95, 0x7a, 0xc5,
    0x3a, 0x21, 0x6a, 0xe8,
};
static const unsigned char kat214_nor_nonce[] = {
    0x6c, 0xf3, 0x0f, 0x32, 0x88, 0x51, 0x2f, 0x8c,
};
static const unsigned char kat214_nor_persstr[] = {
    0x13, 0x3f, 0x95, 0x81, 0x6e, 0x48, 0xf9, 0xc5, 0x7b, 0x5e, 0xdb, 0xb2,
    0x2c, 0x6f, 0xf9, 0x7e,
};
static const unsigned char kat214_nor_addin0[] = {0};
static const unsigned char kat214_nor_addin1[] = {0};
static const unsigned char kat214_nor_retbytes[] = {
    0x47, 0x83, 0x99, 0xdb, 0xad, 0x14, 0xa5, 0x99, 0x02, 0x7c, 0x8d, 0xc6,
    0xbf, 0xd7, 0x9d, 0x26, 0xed, 0x36, 0x99, 0x7d, 0xe4, 0x88, 0xde, 0x51,
    0xb5, 0x66, 0x71, 0xd1, 0xc1, 0xa3, 0xfe, 0xd8, 0x30, 0x75, 0xfe, 0x3b,
    0x5c, 0x93, 0x0c, 0x5a, 0x0f, 0xeb, 0x42, 0x57, 0x53, 0x17, 0xe8, 0xc6,
    0x49, 0x56, 0x20, 0x7b, 0x62, 0x41, 0x48, 0x96, 0xf5, 0x99, 0x16, 0x4e,
    0x5d, 0xa7, 0xaf, 0xa6, 0x9d, 0x07, 0x87, 0x4b, 0xf1, 0x68, 0x99, 0x2b,
    0xba, 0x0a, 0x9f, 0x98, 0x42, 0xfd, 0x1d, 0x21,
};
static const struct drbg_kat_no_reseed kat214_nor_t = {
    3, kat214_nor_entropyin, kat214_nor_nonce, kat214_nor_persstr,
    kat214_nor_addin0, kat214_nor_addin1, kat214_nor_retbytes
};
static const struct drbg_kat kat214_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat214_nor_t
};

static const unsigned char kat215_nor_entropyin[] = {
    0x15, 0xff, 0x59, 0xf0, 0x7c, 0x18, 0xab, 0x87, 0x83, 0x36, 0xde, 0xe5,
    0x4d, 0xcb, 0x47, 0xa4,
};
static const unsigned char kat215_nor_nonce[] = {
    0x39, 0x63, 0x76, 0xc4, 0x8a, 0x1c, 0xe3, 0x43,
};
static const unsigned char kat215_nor_persstr[] = {
    0x92, 0x4d, 0xda, 0xc1, 0x4f, 0x96, 0x11, 0xca, 0x3f, 0x81, 0x71, 0x20,
    0x21, 0xc7, 0xf7, 0xda,
};
static const unsigned char kat215_nor_addin0[] = {0};
static const unsigned char kat215_nor_addin1[] = {0};
static const unsigned char kat215_nor_retbytes[] = {
    0x76, 0xbe, 0x3c, 0x7f, 0x88, 0x6a, 0xb3, 0x8c, 0x1f, 0x98, 0x2c, 0xd6,
    0x8e, 0x0a, 0x2c, 0x44, 0x85, 0xcf, 0x69, 0x62, 0x7d, 0x67, 0x01, 0x59,
    0x67, 0x3b, 0x9f, 0x31, 0x85, 0xde, 0x19, 0x0f, 0xb8, 0xdf, 0x92, 0xe5,
    0x16, 0x43, 0xc6, 0x61, 0xee, 0x84, 0xfa, 0xf9, 0x1e, 0x4a, 0xe7, 0xec,
    0x26, 0x78, 0x8d, 0xf4, 0x76, 0x72, 0xa1, 0xe3, 0xc4, 0x83, 0xdd, 0x28,
    0xdc, 0xf4, 0x88, 0x41, 0xc9, 0xf9, 0x10, 0xfb, 0x0d, 0x46, 0xdd, 0xda,
    0x6f, 0x16, 0x79, 0x1a, 0xfb, 0xa7, 0x3a, 0x83,
};
static const struct drbg_kat_no_reseed kat215_nor_t = {
    4, kat215_nor_entropyin, kat215_nor_nonce, kat215_nor_persstr,
    kat215_nor_addin0, kat215_nor_addin1, kat215_nor_retbytes
};
static const struct drbg_kat kat215_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat215_nor_t
};

static const unsigned char kat216_nor_entropyin[] = {
    0xdc, 0x5f, 0xd0, 0xfc, 0xab, 0x49, 0x0b, 0x8a, 0xec, 0x8d, 0xb1, 0x21,
    0x54, 0x3e, 0x75, 0x0c,
};
static const unsigned char kat216_nor_nonce[] = {
    0xb0, 0xae, 0x04, 0xcc, 0xe1, 0x4b, 0x1d, 0x5d,
};
static const unsigned char kat216_nor_persstr[] = {
    0x15, 0x9d, 0x87, 0x27, 0x5d, 0xde, 0x2e, 0x35, 0x6f, 0xa8, 0x1d, 0x69,
    0x3c, 0x51, 0x71, 0xd7,
};
static const unsigned char kat216_nor_addin0[] = {0};
static const unsigned char kat216_nor_addin1[] = {0};
static const unsigned char kat216_nor_retbytes[] = {
    0xba, 0xc7, 0xc6, 0xc0, 0xfc, 0xde, 0x77, 0x84, 0xf5, 0x61, 0xf6, 0x6b,
    0x07, 0xa1, 0x24, 0x22, 0xa3, 0xc7, 0x45, 0xd0, 0xef, 0x4d, 0x17, 0x43,
    0x3a, 0xe0, 0xe1, 0x6a, 0x02, 0xe0, 0x8d, 0x8c, 0xdb, 0xe0, 0xed, 0x5a,
    0x96, 0x89, 0x64, 0xff, 0x0a, 0xa9, 0xdd, 0xdd, 0x9a, 0xbd, 0x5c, 0x55,
    0x90, 0x4a, 0xd7, 0xda, 0x95, 0xf9, 0xca, 0x6f, 0x74, 0x5f, 0x0f, 0xfe,
    0x4b, 0xe0, 0xa0, 0xa8, 0x80, 0xdc, 0xee, 0x0a, 0x63, 0x1d, 0x34, 0x92,
    0xc9, 0x9f, 0x6c, 0xc5, 0xe0, 0x54, 0x57, 0xc9,
};
static const struct drbg_kat_no_reseed kat216_nor_t = {
    5, kat216_nor_entropyin, kat216_nor_nonce, kat216_nor_persstr,
    kat216_nor_addin0, kat216_nor_addin1, kat216_nor_retbytes
};
static const struct drbg_kat kat216_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat216_nor_t
};

static const unsigned char kat217_nor_entropyin[] = {
    0xec, 0xa5, 0x9e, 0xe3, 0x14, 0xb9, 0x59, 0x8f, 0x2b, 0x9a, 0x79, 0xb2,
    0x2b, 0xb7, 0x91, 0xd4,
};
static const unsigned char kat217_nor_nonce[] = {
    0x83, 0x82, 0xba, 0x70, 0xec, 0xce, 0x35, 0x9f,
};
static const unsigned char kat217_nor_persstr[] = {
    0x82, 0x1d, 0xe6, 0x3a, 0xbd, 0x7c, 0xcf, 0x3f, 0x74, 0xdc, 0x9e, 0x99,
    0xa5, 0xf6, 0xe3, 0x98,
};
static const unsigned char kat217_nor_addin0[] = {0};
static const unsigned char kat217_nor_addin1[] = {0};
static const unsigned char kat217_nor_retbytes[] = {
    0x1f, 0x35, 0x09, 0x7f, 0x7c, 0x08, 0x06, 0xa6, 0xfa, 0x93, 0x99, 0xda,
    0x1c, 0x45, 0xe8, 0xea, 0xa0, 0x7c, 0x6b, 0x1c, 0x8f, 0xc3, 0xda, 0xd9,
    0x66, 0x67, 0xde, 0x01, 0xa3, 0x13, 0x93, 0x8e, 0x37, 0x64, 0x90, 0xe2,
    0x14, 0x20, 0x7f, 0x79, 0xe1, 0x0c, 0x24, 0x34, 0x38, 0x02, 0x87, 0xb5,
    0xf7, 0x2c, 0x67, 0xb9, 0x14, 0x7a, 0x8f, 0x56, 0x82, 0xcd, 0xaa, 0x04,
    0x1a, 0xbb, 0x7b, 0xf3, 0x24, 0x46, 0x4f, 0x58, 0x9c, 0x06, 0xaf, 0xe1,
    0xaa, 0xdb, 0xd4, 0x74, 0xaa, 0x51, 0x49, 0x7d,
};
static const struct drbg_kat_no_reseed kat217_nor_t = {
    6, kat217_nor_entropyin, kat217_nor_nonce, kat217_nor_persstr,
    kat217_nor_addin0, kat217_nor_addin1, kat217_nor_retbytes
};
static const struct drbg_kat kat217_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat217_nor_t
};

static const unsigned char kat218_nor_entropyin[] = {
    0x24, 0x3d, 0xec, 0x1c, 0x74, 0xb6, 0xda, 0xf7, 0x5c, 0x94, 0x81, 0xde,
    0x3a, 0xcc, 0xb3, 0x94,
};
static const unsigned char kat218_nor_nonce[] = {
    0x63, 0xa8, 0x36, 0x1f, 0x5b, 0x8c, 0x9c, 0x0e,
};
static const unsigned char kat218_nor_persstr[] = {
    0x79, 0xc6, 0x0f, 0x83, 0xc9, 0x26, 0x6b, 0x48, 0xe3, 0x51, 0x64, 0x8c,
    0x99, 0x1c, 0xf4, 0xd5,
};
static const unsigned char kat218_nor_addin0[] = {0};
static const unsigned char kat218_nor_addin1[] = {0};
static const unsigned char kat218_nor_retbytes[] = {
    0xa8, 0x5f, 0x1e, 0x42, 0xf6, 0xb7, 0x3b, 0x34, 0xfb, 0x63, 0xe5, 0x62,
    0x30, 0x47, 0xd2, 0xa1, 0x6e, 0x9e, 0x03, 0xb9, 0xab, 0x8b, 0x2b, 0xd0,
    0x35, 0x66, 0x9b, 0x7e, 0xdf, 0x18, 0xa7, 0x1f, 0x7e, 0xe9, 0x6d, 0xe8,
    0x5f, 0x8b, 0x2c, 0xea, 0xfb, 0xe1, 0x34, 0x7e, 0x91, 0x94, 0xb3, 0xa6,
    0x2d, 0xf8, 0xee, 0x5e, 0x06, 0x9b, 0x8d, 0x1f, 0x61, 0x9c, 0x20, 0x50,
    0xfc, 0xf9, 0xd2, 0xdb, 0x10, 0x10, 0xa4, 0x93, 0xe9, 0xc0, 0xd3, 0x6f,
    0x46, 0xfb, 0x98, 0x40, 0x60, 0x83, 0xf3, 0x35,
};
static const struct drbg_kat_no_reseed kat218_nor_t = {
    7, kat218_nor_entropyin, kat218_nor_nonce, kat218_nor_persstr,
    kat218_nor_addin0, kat218_nor_addin1, kat218_nor_retbytes
};
static const struct drbg_kat kat218_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat218_nor_t
};

static const unsigned char kat219_nor_entropyin[] = {
    0x7f, 0xb9, 0x05, 0xa5, 0x59, 0x8e, 0x31, 0x33, 0xd3, 0xc4, 0x74, 0x89,
    0xb1, 0x1e, 0xaa, 0x17,
};
static const unsigned char kat219_nor_nonce[] = {
    0x91, 0x8c, 0xf0, 0xf9, 0x15, 0x2b, 0x62, 0x94,
};
static const unsigned char kat219_nor_persstr[] = {
    0x28, 0x15, 0x18, 0x77, 0x28, 0x79, 0x0a, 0x95, 0xbb, 0x6d, 0x22, 0xd7,
    0xb0, 0x92, 0xd2, 0xc4,
};
static const unsigned char kat219_nor_addin0[] = {0};
static const unsigned char kat219_nor_addin1[] = {0};
static const unsigned char kat219_nor_retbytes[] = {
    0x15, 0x56, 0xa1, 0x82, 0x04, 0x57, 0x06, 0xd3, 0xc8, 0x60, 0xf2, 0x43,
    0x2e, 0xc0, 0xbb, 0xe9, 0x68, 0x95, 0x5e, 0xbd, 0x52, 0x8c, 0x2e, 0x8f,
    0xbc, 0xc5, 0xc1, 0x04, 0xbe, 0x88, 0x9c, 0x7b, 0x76, 0xff, 0x91, 0x35,
    0x0e, 0xe5, 0xba, 0x69, 0x08, 0xc7, 0xfc, 0x18, 0xf8, 0xb1, 0x1a, 0xfd,
    0x9b, 0xb9, 0x05, 0x72, 0x81, 0xe4, 0x62, 0xf5, 0x35, 0x0f, 0xd6, 0x78,
    0xe5, 0xd8, 0x72, 0xdf, 0x19, 0x91, 0x01, 0x77, 0x17, 0x19, 0x7c, 0x21,
    0xb1, 0x37, 0x4b, 0x79, 0xf6, 0xf7, 0xfe, 0xbb,
};
static const struct drbg_kat_no_reseed kat219_nor_t = {
    8, kat219_nor_entropyin, kat219_nor_nonce, kat219_nor_persstr,
    kat219_nor_addin0, kat219_nor_addin1, kat219_nor_retbytes
};
static const struct drbg_kat kat219_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat219_nor_t
};

static const unsigned char kat220_nor_entropyin[] = {
    0x01, 0xfb, 0x28, 0xbf, 0xa4, 0x03, 0xa7, 0x19, 0xe1, 0xae, 0x11, 0xf6,
    0x1a, 0xae, 0x03, 0x5e,
};
static const unsigned char kat220_nor_nonce[] = {
    0x27, 0x03, 0x6c, 0xd4, 0x15, 0x8f, 0x1b, 0x7b,
};
static const unsigned char kat220_nor_persstr[] = {
    0x6e, 0x18, 0x17, 0x55, 0x16, 0x81, 0x69, 0xba, 0x43, 0xc2, 0x69, 0xaf,
    0x58, 0xec, 0x40, 0xfe,
};
static const unsigned char kat220_nor_addin0[] = {0};
static const unsigned char kat220_nor_addin1[] = {0};
static const unsigned char kat220_nor_retbytes[] = {
    0x1d, 0xe8, 0x3d, 0x60, 0xa0, 0xe3, 0x6f, 0x7f, 0x4f, 0x7e, 0xdc, 0xe0,
    0x30, 0x77, 0xbc, 0x92, 0x68, 0xda, 0x4d, 0x64, 0xed, 0xd2, 0x48, 0x59,
    0xd2, 0x1a, 0x19, 0x0c, 0x06, 0xa0, 0x9a, 0xa3, 0xa5, 0xc7, 0x91, 0x11,
    0x6b, 0xa1, 0x2e, 0x2a, 0x13, 0xff, 0xa5, 0xe1, 0x90, 0xd1, 0x33, 0x8b,
    0x8f, 0x65, 0x5a, 0x8d, 0x98, 0x22, 0xb8, 0x4d, 0x86, 0x50, 0x95, 0x8b,
    0xba, 0x81, 0x9b, 0xf9, 0xb4, 0x12, 0x92, 0xbf, 0x5d, 0x66, 0xb4, 0x9a,
    0xc6, 0xcc, 0xc0, 0x2b, 0x9d, 0xfd, 0x39, 0xa8,
};
static const struct drbg_kat_no_reseed kat220_nor_t = {
    9, kat220_nor_entropyin, kat220_nor_nonce, kat220_nor_persstr,
    kat220_nor_addin0, kat220_nor_addin1, kat220_nor_retbytes
};
static const struct drbg_kat kat220_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat220_nor_t
};

static const unsigned char kat221_nor_entropyin[] = {
    0x96, 0x49, 0x46, 0x5c, 0x60, 0x87, 0x9b, 0x37, 0x25, 0xb9, 0x2a, 0xad,
    0x1f, 0xa4, 0x10, 0x77,
};
static const unsigned char kat221_nor_nonce[] = {
    0x8e, 0x35, 0xb2, 0x87, 0xbc, 0x32, 0xc2, 0x73,
};
static const unsigned char kat221_nor_persstr[] = {
    0x36, 0x1f, 0xf9, 0x96, 0x34, 0xed, 0x95, 0x41, 0xe7, 0x10, 0xb9, 0x51,
    0xb0, 0x10, 0x5c, 0xd3,
};
static const unsigned char kat221_nor_addin0[] = {0};
static const unsigned char kat221_nor_addin1[] = {0};
static const unsigned char kat221_nor_retbytes[] = {
    0x02, 0x11, 0xe8, 0x08, 0x3c, 0x45, 0x76, 0xa8, 0x15, 0xab, 0x2d, 0x68,
    0xf1, 0xc4, 0xae, 0x89, 0x79, 0x19, 0x8d, 0x8e, 0x44, 0xa5, 0xd1, 0x4f,
    0x80, 0x98, 0x76, 0x95, 0xa3, 0x97, 0xcf, 0x71, 0xd2, 0x57, 0xed, 0xec,
    0x3a, 0x32, 0xbc, 0xcb, 0x2e, 0x29, 0x3c, 0x59, 0x17, 0x3f, 0xce, 0x4a,
    0x1f, 0x97, 0x5e, 0x4e, 0x91, 0x11, 0xca, 0xdf, 0xab, 0x18, 0xcf, 0xa0,
    0xbf, 0x5f, 0x39, 0x2e, 0x4e, 0x14, 0x4c, 0x48, 0x27, 0x9d, 0xa5, 0x63,
    0xba, 0xc9, 0xd0, 0x3d, 0xdf, 0x0c, 0x73, 0x57,
};
static const struct drbg_kat_no_reseed kat221_nor_t = {
    10, kat221_nor_entropyin, kat221_nor_nonce, kat221_nor_persstr,
    kat221_nor_addin0, kat221_nor_addin1, kat221_nor_retbytes
};
static const struct drbg_kat kat221_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat221_nor_t
};

static const unsigned char kat222_nor_entropyin[] = {
    0xb3, 0x3d, 0xca, 0x3f, 0xc1, 0x04, 0xae, 0x07, 0x2a, 0x78, 0x1e, 0xcc,
    0x2d, 0xc4, 0xaa, 0x35,
};
static const unsigned char kat222_nor_nonce[] = {
    0x92, 0xd9, 0x74, 0xf0, 0x27, 0x81, 0x11, 0xa6,
};
static const unsigned char kat222_nor_persstr[] = {
    0xf2, 0x71, 0x50, 0x0a, 0x57, 0xf6, 0xd0, 0xfa, 0x65, 0x3e, 0x2a, 0x97,
    0x61, 0xa5, 0x11, 0x8e,
};
static const unsigned char kat222_nor_addin0[] = {0};
static const unsigned char kat222_nor_addin1[] = {0};
static const unsigned char kat222_nor_retbytes[] = {
    0xc4, 0x71, 0x04, 0x46, 0x78, 0xd3, 0xbf, 0xca, 0x70, 0xb7, 0xc0, 0x97,
    0x9d, 0x82, 0xd0, 0xd3, 0xd7, 0xc7, 0xef, 0xac, 0xf8, 0x78, 0xff, 0x6e,
    0x1d, 0x30, 0xf0, 0xf8, 0xff, 0x48, 0xaf, 0xf4, 0xd6, 0x39, 0xfb, 0x18,
    0x9a, 0xcd, 0xab, 0x5b, 0x3c, 0x31, 0x82, 0x6c, 0x96, 0x0f, 0x44, 0x40,
    0x73, 0x14, 0x7d, 0x8a, 0x6c, 0x8e, 0xd5, 0x08, 0x18, 0x8f, 0x48, 0xc9,
    0x5f, 0x40, 0x1d, 0x00, 0x2e, 0xe5, 0xdd, 0x86, 0x6f, 0x0c, 0x4b, 0x05,
    0xd0, 0xac, 0xc4, 0x84, 0x0c, 0x6b, 0xec, 0xf2,
};
static const struct drbg_kat_no_reseed kat222_nor_t = {
    11, kat222_nor_entropyin, kat222_nor_nonce, kat222_nor_persstr,
    kat222_nor_addin0, kat222_nor_addin1, kat222_nor_retbytes
};
static const struct drbg_kat kat222_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat222_nor_t
};

static const unsigned char kat223_nor_entropyin[] = {
    0xbf, 0x2d, 0x50, 0xd2, 0x9b, 0x73, 0x37, 0xe7, 0x1c, 0xe1, 0x29, 0xae,
    0x8f, 0x86, 0x11, 0xf1,
};
static const unsigned char kat223_nor_nonce[] = {
    0x3c, 0xff, 0xbf, 0x94, 0xc8, 0x46, 0x2d, 0xf7,
};
static const unsigned char kat223_nor_persstr[] = {
    0x52, 0x63, 0xf5, 0xb6, 0xea, 0xf9, 0x12, 0x2c, 0x2b, 0x57, 0xc5, 0x63,
    0xaa, 0xc6, 0x58, 0x3c,
};
static const unsigned char kat223_nor_addin0[] = {0};
static const unsigned char kat223_nor_addin1[] = {0};
static const unsigned char kat223_nor_retbytes[] = {
    0x1a, 0xce, 0x79, 0xcb, 0x19, 0x71, 0x9f, 0x58, 0xf3, 0x20, 0xbb, 0x68,
    0xe5, 0xaa, 0x53, 0x1e, 0xbf, 0x59, 0x56, 0xbb, 0x23, 0x07, 0xdc, 0x09,
    0xcf, 0x9a, 0xb5, 0x44, 0xbf, 0xae, 0x36, 0x30, 0x1a, 0x90, 0x84, 0x45,
    0xe4, 0x19, 0x5f, 0x33, 0x4e, 0x62, 0x87, 0xc9, 0x0f, 0xf8, 0xa8, 0xc5,
    0x79, 0xf4, 0x33, 0x7f, 0x88, 0xcc, 0xf9, 0x09, 0x5c, 0xe2, 0x31, 0x68,
    0x70, 0x29, 0x59, 0xa8, 0x21, 0xca, 0x7a, 0x92, 0xef, 0xb0, 0xaa, 0x74,
    0x8c, 0xd2, 0x52, 0x29, 0x8a, 0x59, 0xee, 0x96,
};
static const struct drbg_kat_no_reseed kat223_nor_t = {
    12, kat223_nor_entropyin, kat223_nor_nonce, kat223_nor_persstr,
    kat223_nor_addin0, kat223_nor_addin1, kat223_nor_retbytes
};
static const struct drbg_kat kat223_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat223_nor_t
};

static const unsigned char kat224_nor_entropyin[] = {
    0xa2, 0xc7, 0x60, 0x04, 0xf5, 0x22, 0x92, 0xae, 0x30, 0x74, 0x4b, 0x3b,
    0xb7, 0xb9, 0x74, 0x6a,
};
static const unsigned char kat224_nor_nonce[] = {
    0xa2, 0x6e, 0x98, 0x09, 0x0c, 0x9c, 0x4e, 0x16,
};
static const unsigned char kat224_nor_persstr[] = {
    0xdf, 0xff, 0xad, 0xa1, 0x52, 0xeb, 0xc2, 0xc6, 0xe2, 0xdd, 0x71, 0x21,
    0xd5, 0x84, 0x4e, 0xa0,
};
static const unsigned char kat224_nor_addin0[] = {0};
static const unsigned char kat224_nor_addin1[] = {0};
static const unsigned char kat224_nor_retbytes[] = {
    0x67, 0x5e, 0xd1, 0xb8, 0xa2, 0x8f, 0x8f, 0x00, 0xc3, 0xb6, 0xef, 0xd9,
    0x27, 0x6d, 0xaf, 0x5c, 0xfb, 0xff, 0x95, 0xfd, 0x59, 0x2c, 0xe8, 0xc9,
    0x4c, 0x4b, 0x5d, 0xc9, 0x79, 0x3b, 0x67, 0xeb, 0x2d, 0x9c, 0xa3, 0x7a,
    0xa9, 0xc0, 0xb6, 0x0e, 0xc5, 0xc9, 0xa2, 0xa5, 0x68, 0x39, 0x70, 0x88,
    0x09, 0x59, 0x48, 0x94, 0xcb, 0x7a, 0xdc, 0xf6, 0xc9, 0xce, 0xe7, 0x94,
    0x47, 0x1d, 0xba, 0x01, 0xd6, 0xeb, 0xa2, 0xd1, 0xa6, 0x31, 0x62, 0x4b,
    0x6d, 0x0c, 0x36, 0x80, 0x4f, 0xaf, 0x6a, 0x60,
};
static const struct drbg_kat_no_reseed kat224_nor_t = {
    13, kat224_nor_entropyin, kat224_nor_nonce, kat224_nor_persstr,
    kat224_nor_addin0, kat224_nor_addin1, kat224_nor_retbytes
};
static const struct drbg_kat kat224_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat224_nor_t
};

static const unsigned char kat225_nor_entropyin[] = {
    0x4a, 0x1b, 0x6e, 0x50, 0xd2, 0x85, 0x48, 0x66, 0xe1, 0xc1, 0xdd, 0xb6,
    0x53, 0x77, 0xba, 0xca,
};
static const unsigned char kat225_nor_nonce[] = {
    0x72, 0x39, 0xf9, 0x2b, 0x63, 0xfb, 0x3d, 0xbe,
};
static const unsigned char kat225_nor_persstr[] = {
    0x8d, 0x2e, 0x2c, 0xa3, 0x98, 0x5b, 0xd2, 0x53, 0x8a, 0x71, 0xf0, 0x2c,
    0xc3, 0xeb, 0x55, 0x68,
};
static const unsigned char kat225_nor_addin0[] = {0};
static const unsigned char kat225_nor_addin1[] = {0};
static const unsigned char kat225_nor_retbytes[] = {
    0x0e, 0x4c, 0xb3, 0x28, 0xc0, 0x3f, 0xaa, 0xed, 0xbe, 0xc7, 0x21, 0x57,
    0x25, 0x85, 0x10, 0x69, 0xbc, 0xea, 0xe4, 0x33, 0x2d, 0xe6, 0xa7, 0x0e,
    0x35, 0x21, 0xdd, 0x06, 0x5f, 0x2f, 0x79, 0x23, 0x48, 0x59, 0x69, 0x57,
    0x1e, 0xbd, 0x7f, 0x24, 0xbe, 0x46, 0x0f, 0xd9, 0x01, 0xc6, 0xb3, 0xe3,
    0x56, 0xda, 0x6e, 0xe5, 0x26, 0x2e, 0xf2, 0xd7, 0x6a, 0xd1, 0x4e, 0xb0,
    0xf6, 0x97, 0xf8, 0xfb, 0x92, 0xaf, 0x2f, 0x46, 0x63, 0x01, 0x98, 0xc5,
    0xf7, 0x01, 0x88, 0x60, 0x88, 0x61, 0x47, 0xb3,
};
static const struct drbg_kat_no_reseed kat225_nor_t = {
    14, kat225_nor_entropyin, kat225_nor_nonce, kat225_nor_persstr,
    kat225_nor_addin0, kat225_nor_addin1, kat225_nor_retbytes
};
static const struct drbg_kat kat225_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 0, 80, &kat225_nor_t
};

static const unsigned char kat226_nor_entropyin[] = {
    0x1b, 0x28, 0x27, 0xae, 0x51, 0x82, 0x4e, 0x1f, 0x68, 0xa4, 0x81, 0x16,
    0xdf, 0xb2, 0xe3, 0x43,
};
static const unsigned char kat226_nor_nonce[] = {
    0xb9, 0xea, 0xc1, 0xb9, 0x47, 0xad, 0x0f, 0x4c,
};
static const unsigned char kat226_nor_persstr[] = {
    0x51, 0xfc, 0x5e, 0x48, 0x5d, 0x8c, 0xd0, 0x82, 0x09, 0xc9, 0x89, 0x48,
    0x21, 0xa2, 0xa6, 0xb6,
};
static const unsigned char kat226_nor_addin0[] = {
    0x39, 0xa5, 0x27, 0x48, 0xbd, 0xbc, 0x7f, 0xe0, 0xff, 0x45, 0x45, 0xf0,
    0x17, 0x99, 0x49, 0xe6,
};
static const unsigned char kat226_nor_addin1[] = {
    0x4d, 0x1a, 0x77, 0x4a, 0x70, 0x74, 0x38, 0xbf, 0x46, 0xe6, 0x90, 0x9a,
    0xff, 0x87, 0x41, 0x89,
};
static const unsigned char kat226_nor_retbytes[] = {
    0x90, 0x0d, 0x58, 0x58, 0x48, 0x38, 0x8e, 0x45, 0x2f, 0xde, 0x77, 0x71,
    0x40, 0x8c, 0x52, 0x92, 0x64, 0x5d, 0xf4, 0x62, 0x38, 0x9b, 0xbd, 0xbb,
    0x29, 0xcd, 0x1e, 0x2a, 0xb4, 0xf2, 0x28, 0x74, 0x68, 0xee, 0xfa, 0x08,
    0xb7, 0xb3, 0xbb, 0x69, 0xc2, 0x64, 0x53, 0xbd, 0x47, 0xab, 0xdb, 0x91,
    0xba, 0x58, 0x2a, 0xe2, 0x7b, 0xb9, 0xd5, 0x24, 0x3f, 0x98, 0x3f, 0xef,
    0xc6, 0x2b, 0xc1, 0x86, 0x58, 0x47, 0x99, 0x96, 0x33, 0x14, 0xd0, 0xf5,
    0x64, 0xef, 0x36, 0xbe, 0x63, 0x8b, 0x1a, 0x64,
};
static const struct drbg_kat_no_reseed kat226_nor_t = {
    0, kat226_nor_entropyin, kat226_nor_nonce, kat226_nor_persstr,
    kat226_nor_addin0, kat226_nor_addin1, kat226_nor_retbytes
};
static const struct drbg_kat kat226_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat226_nor_t
};

static const unsigned char kat227_nor_entropyin[] = {
    0x0f, 0x3b, 0x65, 0xbb, 0x13, 0xf1, 0xac, 0xb3, 0x17, 0x2b, 0xe7, 0x2e,
    0xde, 0xa5, 0x24, 0x7c,
};
static const unsigned char kat227_nor_nonce[] = {
    0x35, 0x22, 0xf2, 0xcb, 0xac, 0x34, 0x23, 0x8b,
};
static const unsigned char kat227_nor_persstr[] = {
    0xd8, 0x92, 0xf6, 0x5f, 0x19, 0x49, 0xd2, 0x80, 0xee, 0xcb, 0x29, 0x52,
    0x76, 0xb9, 0xc6, 0x9f,
};
static const unsigned char kat227_nor_addin0[] = {
    0x37, 0x2c, 0x1f, 0x00, 0xa4, 0x62, 0xa5, 0x35, 0xc4, 0x7a, 0x77, 0x1b,
    0x8e, 0x26, 0x53, 0x58,
};
static const unsigned char kat227_nor_addin1[] = {
    0x30, 0x62, 0x26, 0x8c, 0x61, 0x09, 0xe9, 0x2e, 0x96, 0x4a, 0x65, 0x5b,
    0xbe, 0x3f, 0x93, 0x80,
};
static const unsigned char kat227_nor_retbytes[] = {
    0xb7, 0xbb, 0x52, 0x38, 0x4c, 0xa0, 0xe6, 0x07, 0xa9, 0x86, 0xab, 0xac,
    0x17, 0x5e, 0xc2, 0x72, 0xdd, 0xaf, 0x05, 0x30, 0xf2, 0x41, 0xbd, 0xa6,
    0xea, 0x7b, 0x54, 0xb4, 0x24, 0x56, 0x56, 0x9e, 0xcc, 0xc0, 0x25, 0xca,
    0x66, 0x93, 0x6f, 0x2b, 0xac, 0xae, 0xe6, 0xdf, 0x8c, 0x93, 0xf7, 0xc1,
    0x3a, 0x89, 0xbf, 0x30, 0x32, 0xb8, 0x17, 0x46, 0x44, 0x25, 0xf3, 0x24,
    0xba, 0x17, 0x63, 0xad, 0xa7, 0x1d, 0x2d, 0x83, 0x46, 0x6c, 0x4f, 0x69,
    0x31, 0x62, 0x64, 0x82, 0x23, 0x8f, 0xf2, 0x1b,
};
static const struct drbg_kat_no_reseed kat227_nor_t = {
    1, kat227_nor_entropyin, kat227_nor_nonce, kat227_nor_persstr,
    kat227_nor_addin0, kat227_nor_addin1, kat227_nor_retbytes
};
static const struct drbg_kat kat227_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat227_nor_t
};

static const unsigned char kat228_nor_entropyin[] = {
    0xd7, 0x9c, 0x74, 0xc2, 0x3e, 0x2e, 0x01, 0x6f, 0x64, 0xbf, 0xa3, 0xad,
    0x5b, 0x35, 0x19, 0x72,
};
static const unsigned char kat228_nor_nonce[] = {
    0x5e, 0xfc, 0x54, 0xf3, 0x23, 0xf2, 0x08, 0x3c,
};
static const unsigned char kat228_nor_persstr[] = {
    0x08, 0xe3, 0x1c, 0x36, 0xe9, 0x29, 0x6c, 0x6c, 0xc2, 0xa9, 0x6f, 0x2f,
    0xd5, 0x39, 0xae, 0x5f,
};
static const unsigned char kat228_nor_addin0[] = {
    0x80, 0x54, 0x04, 0x91, 0xeb, 0x6a, 0x5b, 0x39, 0xb0, 0x79, 0x72, 0x6a,
    0x2d, 0x2e, 0x6e, 0xd7,
};
static const unsigned char kat228_nor_addin1[] = {
    0x57, 0xd4, 0x92, 0xf0, 0xb0, 0x98, 0xcc, 0xd5, 0x81, 0x0f, 0x9a, 0x6b,
    0xc9, 0xf7, 0x92, 0x13,
};
static const unsigned char kat228_nor_retbytes[] = {
    0x40, 0xa1, 0xa6, 0xf1, 0x3f, 0x9e, 0x2c, 0x2b, 0x20, 0xe4, 0xbe, 0x32,
    0xe5, 0x16, 0x21, 0x55, 0x16, 0x54, 0x03, 0xe9, 0x2a, 0xb8, 0xe4, 0xd5,
    0x51, 0x7b, 0x44, 0xad, 0x6d, 0x24, 0xe0, 0xf7, 0xe5, 0x1a, 0x97, 0x0b,
    0x46, 0x9a, 0xdb, 0x6d, 0xdb, 0x98, 0x70, 0xc1, 0x19, 0x55, 0x74, 0x6e,
    0x28, 0x06, 0x67, 0x18, 0x5c, 0x5a, 0x5e, 0xb1, 0xc7, 0x21, 0x86, 0x3c,
    0x22, 0xb2, 0x60, 0xc2, 0xc0, 0x07, 0x80, 0x02, 0x0e, 0x6a, 0x66, 0xe6,
    0x53, 0x1d, 0xc7, 0x14, 0xe8, 0x83, 0xbc, 0x01,
};
static const struct drbg_kat_no_reseed kat228_nor_t = {
    2, kat228_nor_entropyin, kat228_nor_nonce, kat228_nor_persstr,
    kat228_nor_addin0, kat228_nor_addin1, kat228_nor_retbytes
};
static const struct drbg_kat kat228_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat228_nor_t
};

static const unsigned char kat229_nor_entropyin[] = {
    0xda, 0x8b, 0x8c, 0x8a, 0x28, 0x5d, 0xed, 0xce, 0x25, 0x2e, 0x10, 0xeb,
    0x80, 0x3e, 0x9b, 0x90,
};
static const unsigned char kat229_nor_nonce[] = {
    0x47, 0x7b, 0x9d, 0x7b, 0xb8, 0xf0, 0x76, 0xaf,
};
static const unsigned char kat229_nor_persstr[] = {
    0xb7, 0xa6, 0xed, 0xb7, 0xad, 0x25, 0x10, 0x07, 0xc2, 0x8b, 0x2d, 0x33,
    0x30, 0xe7, 0xc6, 0x6b,
};
static const unsigned char kat229_nor_addin0[] = {
    0x26, 0x71, 0x08, 0x1c, 0x7c, 0xf3, 0xb2, 0x0a, 0x65, 0x44, 0x4c, 0x3e,
    0xd4, 0x95, 0xa8, 0xd4,
};
static const unsigned char kat229_nor_addin1[] = {
    0xe3, 0x1b, 0x86, 0x8c, 0xfe, 0x28, 0x24, 0x13, 0x5d, 0x15, 0x32, 0xf8,
    0x82, 0x31, 0x22, 0xe4,
};
static const unsigned char kat229_nor_retbytes[] = {
    0x60, 0x25, 0x99, 0xed, 0xee, 0xb4, 0xc8, 0x6b, 0xaf, 0x94, 0x79, 0xd4,
    0xb7, 0x45, 0xbd, 0x0f, 0x39, 0x29, 0x9d, 0x2b, 0x80, 0x84, 0xf1, 0xf6,
    0xdf, 0x9f, 0x48, 0xee, 0xcd, 0x08, 0xde, 0xa9, 0x8a, 0x81, 0x37, 0x73,
    0x2f, 0x08, 0xae, 0x83, 0xc6, 0x1b, 0x73, 0xae, 0x5a, 0xf0, 0x95, 0xed,
    0xf7, 0xca, 0xf1, 0x12, 0xd9, 0x89, 0x01, 0x62, 0x24, 0x32, 0xb4, 0xf8,
    0xd8, 0x8f, 0x2e, 0x65, 0x61, 0x24, 0x56, 0x52, 0xef, 0x94, 0xb3, 0x27,
    0xfa, 0xf9, 0xce, 0x09, 0x0f, 0x79, 0x06, 0x8f,
};
static const struct drbg_kat_no_reseed kat229_nor_t = {
    3, kat229_nor_entropyin, kat229_nor_nonce, kat229_nor_persstr,
    kat229_nor_addin0, kat229_nor_addin1, kat229_nor_retbytes
};
static const struct drbg_kat kat229_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat229_nor_t
};

static const unsigned char kat230_nor_entropyin[] = {
    0x4c, 0x49, 0xcf, 0xf5, 0x51, 0xdb, 0x41, 0xe6, 0x01, 0xbd, 0x14, 0x00,
    0x5c, 0x90, 0x90, 0xa6,
};
static const unsigned char kat230_nor_nonce[] = {
    0x7e, 0x9b, 0xf1, 0x33, 0x9d, 0x18, 0xcb, 0x6c,
};
static const unsigned char kat230_nor_persstr[] = {
    0xb8, 0x34, 0xac, 0x33, 0xcd, 0x97, 0xa9, 0xb5, 0x06, 0x59, 0x3e, 0xd5,
    0x97, 0x09, 0xe2, 0x13,
};
static const unsigned char kat230_nor_addin0[] = {
    0x85, 0x64, 0x98, 0x37, 0xdb, 0x26, 0x91, 0x04, 0xef, 0x6a, 0x97, 0x67,
    0x38, 0xa1, 0x71, 0xa8,
};
static const unsigned char kat230_nor_addin1[] = {
    0x59, 0x82, 0x38, 0xe9, 0x01, 0x89, 0xd1, 0xe9, 0x23, 0xae, 0x39, 0xee,
    0x5e, 0x7e, 0xc2, 0xd8,
};
static const unsigned char kat230_nor_retbytes[] = {
    0x4c, 0x28, 0x99, 0x1d, 0x91, 0x98, 0x49, 0x41, 0xd1, 0x26, 0xad, 0xd7,
    0x4b, 0x46, 0x3a, 0x8c, 0x31, 0xc8, 0xe0, 0xb6, 0x6d, 0x12, 0xe6, 0x64,
    0x02, 0x16, 0x71, 0x23, 0x4d, 0x49, 0x46, 0xcd, 0x3e, 0x55, 0xf1, 0x4d,
    0x22, 0x50, 0x66, 0x23, 0x16, 0x5b, 0xa9, 0xd2, 0x2c, 0x6a, 0x7f, 0x11,
    0xff, 0x2a, 0x12, 0xc6, 0x26, 0x74, 0x6f, 0x73, 0x9d, 0x70, 0xc9, 0x71,
    0xd4, 0xd4, 0x7f, 0x22, 0x97, 0x6a, 0x0e, 0x8d, 0x06, 0xb9, 0x36, 0x61,
    0x35, 0x0b, 0x43, 0xc6, 0xb2, 0x75, 0x4a, 0x89,
};
static const struct drbg_kat_no_reseed kat230_nor_t = {
    4, kat230_nor_entropyin, kat230_nor_nonce, kat230_nor_persstr,
    kat230_nor_addin0, kat230_nor_addin1, kat230_nor_retbytes
};
static const struct drbg_kat kat230_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat230_nor_t
};

static const unsigned char kat231_nor_entropyin[] = {
    0x65, 0xa0, 0x8a, 0x98, 0xf1, 0xb6, 0x32, 0xbe, 0xda, 0x76, 0x1c, 0x07,
    0x8d, 0x4a, 0x19, 0x5b,
};
static const unsigned char kat231_nor_nonce[] = {
    0xb9, 0x3a, 0x7d, 0x1a, 0x2f, 0x85, 0x9c, 0xc9,
};
static const unsigned char kat231_nor_persstr[] = {
    0x38, 0xbf, 0x8e, 0x9d, 0xb8, 0x45, 0xfb, 0x9e, 0xe4, 0xa4, 0xa5, 0x0f,
    0x6a, 0x14, 0xeb, 0x9e,
};
static const unsigned char kat231_nor_addin0[] = {
    0x86, 0x5a, 0x17, 0xd8, 0xb9, 0xf7, 0xf9, 0x5a, 0x4b, 0x08, 0xbf, 0x94,
    0xbd, 0xa2, 0x68, 0xa7,
};
static const unsigned char kat231_nor_addin1[] = {
    0x7f, 0x18, 0x18, 0x2a, 0x36, 0xa6, 0xcf, 0x72, 0x40, 0x33, 0x7c, 0x4f,
    0xbb, 0x5e, 0x4e, 0x72,
};
static const unsigned char kat231_nor_retbytes[] = {
    0x30, 0xe2, 0x42, 0x71, 0xf4, 0x13, 0x59, 0x6f, 0x8c, 0xb0, 0xf3, 0xd3,
    0x7c, 0xb5, 0xd3, 0x3f, 0xa5, 0xf1, 0xab, 0xeb, 0x08, 0xe5, 0x33, 0x1c,
    0xe6, 0x90, 0x1b, 0xdb, 0xc1, 0xb8, 0xde, 0x42, 0xac, 0x12, 0xe4, 0x9b,
    0xc0, 0x50, 0xeb, 0xcf, 0xf0, 0x0b, 0x42, 0xb3, 0x94, 0xcd, 0xe6, 0xbe,
    0xf0, 0x99, 0x3f, 0x00, 0x2f, 0x18, 0xc0, 0xe6, 0x8a, 0xed, 0xff, 0x96,
    0x96, 0x70, 0xf3, 0x12, 0xb9, 0xcd, 0xd8, 0x35, 0x9f, 0xaf, 0x3e, 0x0c,
    0x4d, 0xe2, 0x4e, 0xf9, 0xdb, 0x61, 0x3c, 0x98,
};
static const struct drbg_kat_no_reseed kat231_nor_t = {
    5, kat231_nor_entropyin, kat231_nor_nonce, kat231_nor_persstr,
    kat231_nor_addin0, kat231_nor_addin1, kat231_nor_retbytes
};
static const struct drbg_kat kat231_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat231_nor_t
};

static const unsigned char kat232_nor_entropyin[] = {
    0x70, 0x82, 0x70, 0x84, 0xcc, 0x1f, 0x48, 0x31, 0x7e, 0x1c, 0x95, 0xf2,
    0x55, 0x83, 0x7a, 0x3c,
};
static const unsigned char kat232_nor_nonce[] = {
    0x76, 0x5f, 0xba, 0xf6, 0xd0, 0x35, 0xab, 0xb7,
};
static const unsigned char kat232_nor_persstr[] = {
    0xa5, 0xb8, 0x04, 0x29, 0x62, 0x06, 0xcf, 0x3a, 0x78, 0x93, 0x78, 0x08,
    0x65, 0x64, 0x33, 0x30,
};
static const unsigned char kat232_nor_addin0[] = {
    0xbd, 0x67, 0xfb, 0xd0, 0xce, 0xa2, 0x25, 0x28, 0x2d, 0xbb, 0xde, 0x85,
    0xf9, 0xd8, 0x53, 0x24,
};
static const unsigned char kat232_nor_addin1[] = {
    0x55, 0xb5, 0xb4, 0x15, 0xcc, 0xb0, 0x31, 0xf6, 0xca, 0x34, 0xc9, 0xd7,
    0x71, 0x3b, 0xce, 0xd6,
};
static const unsigned char kat232_nor_retbytes[] = {
    0x2b, 0x85, 0x40, 0xe3, 0x7c, 0x31, 0x8a, 0xfa, 0x95, 0xfc, 0x1e, 0x8d,
    0x08, 0xb8, 0xd9, 0xeb, 0xf9, 0xe2, 0xb9, 0x10, 0xf2, 0x2d, 0xde, 0xb5,
    0xfb, 0xae, 0x43, 0x27, 0xd3, 0x7c, 0xd9, 0x87, 0x24, 0xdc, 0x95, 0xbf,
    0x36, 0x7e, 0x6e, 0x90, 0x19, 0x66, 0x39, 0xbd, 0x21, 0x32, 0x3d, 0x11,
    0x12, 0x78, 0xca, 0x16, 0x66, 0x65, 0x8a, 0xc1, 0xe5, 0x5c, 0x7c, 0x93,
    0x0f, 0x85, 0xe7, 0x73, 0xd3, 0x7d, 0xcf, 0xaf, 0x59, 0x11, 0x51, 0xff,
    0xec, 0x27, 0x29, 0xe3, 0x6e, 0xc5, 0xfc, 0x4e,
};
static const struct drbg_kat_no_reseed kat232_nor_t = {
    6, kat232_nor_entropyin, kat232_nor_nonce, kat232_nor_persstr,
    kat232_nor_addin0, kat232_nor_addin1, kat232_nor_retbytes
};
static const struct drbg_kat kat232_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat232_nor_t
};

static const unsigned char kat233_nor_entropyin[] = {
    0x39, 0xe7, 0x94, 0xb3, 0xb9, 0x56, 0x57, 0x4a, 0xdc, 0x9f, 0xa5, 0x40,
    0x01, 0xb2, 0xba, 0xd5,
};
static const unsigned char kat233_nor_nonce[] = {
    0xfc, 0x25, 0x1a, 0x9c, 0xa8, 0x1c, 0x02, 0x9b,
};
static const unsigned char kat233_nor_persstr[] = {
    0x7b, 0x56, 0x4f, 0x15, 0x15, 0x46, 0x1f, 0xa9, 0xef, 0x73, 0x52, 0xa8,
    0x9f, 0xfc, 0xb9, 0x8e,
};
static const unsigned char kat233_nor_addin0[] = {
    0x24, 0x9b, 0x0a, 0xfd, 0xa2, 0x3f, 0x95, 0xf4, 0x5c, 0xb5, 0x11, 0x16,
    0x92, 0x33, 0x59, 0xe7,
};
static const unsigned char kat233_nor_addin1[] = {
    0x18, 0xb4, 0x25, 0x58, 0x45, 0xcd, 0x8b, 0x2f, 0x3e, 0xcd, 0x84, 0xf2,
    0x70, 0x8d, 0x6a, 0x91,
};
static const unsigned char kat233_nor_retbytes[] = {
    0xd5, 0x36, 0x7e, 0x18, 0x62, 0x59, 0xb8, 0x90, 0x0a, 0x4c, 0xc2, 0xd0,
    0x8d, 0xe8, 0xbb, 0x86, 0x50, 0x3d, 0x23, 0xd8, 0x26, 0x42, 0x52, 0xdd,
    0x73, 0xff, 0x9d, 0x21, 0xce, 0x9c, 0xb5, 0xb4, 0x65, 0x67, 0x62, 0x69,
    0x80, 0x8b, 0x47, 0x7b, 0x4b, 0x24, 0xa7, 0x23, 0x1e, 0x5c, 0x8e, 0x31,
    0xbf, 0x3a, 0x5d, 0x1d, 0x96, 0x44, 0x4e, 0x8f, 0xe1, 0x28, 0xcc, 0x71,
    0x40, 0x21, 0x3d, 0xa6, 0x70, 0x62, 0x66, 0x0b, 0xa4, 0x90, 0xb8, 0xe6,
    0xab, 0xee, 0xeb, 0x33, 0x81, 0x01, 0x8f, 0x0b,
};
static const struct drbg_kat_no_reseed kat233_nor_t = {
    7, kat233_nor_entropyin, kat233_nor_nonce, kat233_nor_persstr,
    kat233_nor_addin0, kat233_nor_addin1, kat233_nor_retbytes
};
static const struct drbg_kat kat233_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat233_nor_t
};

static const unsigned char kat234_nor_entropyin[] = {
    0x0e, 0x54, 0xfd, 0x93, 0x17, 0x32, 0x45, 0x39, 0xb5, 0xf8, 0x7d, 0xda,
    0x8c, 0x5f, 0xee, 0xe2,
};
static const unsigned char kat234_nor_nonce[] = {
    0xf6, 0x6c, 0xeb, 0xa0, 0x13, 0x4d, 0x13, 0x92,
};
static const unsigned char kat234_nor_persstr[] = {
    0x83, 0xca, 0x30, 0x99, 0xc4, 0xfd, 0x95, 0x6d, 0x69, 0xde, 0x1d, 0xa9,
    0xdd, 0x2d, 0xae, 0x70,
};
static const unsigned char kat234_nor_addin0[] = {
    0x60, 0x44, 0x60, 0x20, 0x1a, 0xdb, 0xa7, 0x56, 0x0a, 0x9b, 0x7f, 0xcb,
    0xb2, 0xda, 0xb3, 0x45,
};
static const unsigned char kat234_nor_addin1[] = {
    0x91, 0x1e, 0xe9, 0xcf, 0xba, 0xb9, 0xd8, 0xab, 0x26, 0x13, 0x15, 0x93,
    0xb9, 0xa9, 0x74, 0xb8,
};
static const unsigned char kat234_nor_retbytes[] = {
    0x55, 0x98, 0x42, 0x4e, 0x63, 0xe7, 0x44, 0x29, 0x43, 0x1a, 0xd9, 0xa0,
    0x7d, 0xf0, 0x27, 0x04, 0x75, 0x4f, 0xfb, 0x9b, 0x30, 0x4d, 0x28, 0x93,
    0xd0, 0x2e, 0xc6, 0xb4, 0x5f, 0x33, 0x5c, 0xf0, 0x66, 0x3e, 0x86, 0x52,
    0x03, 0xcf, 0x2f, 0x10, 0x6c, 0x1a, 0x94, 0x72, 0xe6, 0x0c, 0x0c, 0x06,
    0x31, 0xa9, 0x6f, 0xd3, 0x85, 0x62, 0x55, 0xac, 0x01, 0xdd, 0xb9, 0x4c,
    0x7d, 0x0a, 0x23, 0x5d, 0xa6, 0x04, 0xe4, 0x6c, 0x19, 0xde, 0xcf, 0xd0,
    0x08, 0x34, 0xa4, 0x63, 0xea, 0x27, 0x06, 0x57,
};
static const struct drbg_kat_no_reseed kat234_nor_t = {
    8, kat234_nor_entropyin, kat234_nor_nonce, kat234_nor_persstr,
    kat234_nor_addin0, kat234_nor_addin1, kat234_nor_retbytes
};
static const struct drbg_kat kat234_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat234_nor_t
};

static const unsigned char kat235_nor_entropyin[] = {
    0x6b, 0x95, 0xce, 0x60, 0xfb, 0xde, 0x0b, 0x7a, 0x1c, 0x08, 0x4e, 0xe9,
    0xc4, 0xc0, 0xef, 0x3a,
};
static const unsigned char kat235_nor_nonce[] = {
    0xbc, 0x55, 0xd0, 0xd7, 0x3d, 0xf2, 0x01, 0x6c,
};
static const unsigned char kat235_nor_persstr[] = {
    0x54, 0xd5, 0x95, 0x2c, 0x46, 0xfc, 0xcb, 0x8b, 0xdd, 0xa8, 0xde, 0xc9,
    0xfe, 0x0f, 0x44, 0x20,
};
static const unsigned char kat235_nor_addin0[] = {
    0xee, 0xfa, 0x1d, 0x45, 0x0f, 0x09, 0x44, 0x5e, 0x37, 0x2f, 0x30, 0xc8,
    0xe7, 0xd0, 0xf3, 0x06,
};
static const unsigned char kat235_nor_addin1[] = {
    0x16, 0x0a, 0xde, 0x5d, 0x9d, 0x4d, 0xfd, 0xce, 0x0e, 0x04, 0x45, 0x67,
    0xa8, 0xba, 0xc0, 0x02,
};
static const unsigned char kat235_nor_retbytes[] = {
    0x2a, 0x7d, 0x8d, 0x1d, 0x24, 0x8e, 0xdd, 0x67, 0xb8, 0x0b, 0x63, 0x6c,
    0x22, 0xb5, 0xca, 0x69, 0xe9, 0x04, 0xce, 0xfc, 0xa3, 0x40, 0xbd, 0x85,
    0x86, 0xd1, 0x79, 0x55, 0xc8, 0x2d, 0x77, 0x29, 0xbb, 0x24, 0x3b, 0x2c,
    0x17, 0x15, 0x07, 0x4d, 0x85, 0x65, 0x97, 0x40, 0x6b, 0x86, 0xb9, 0x25,
    0x81, 0x87, 0x24, 0x23, 0x70, 0x41, 0x12, 0x7e, 0x67, 0x28, 0xfe, 0x30,
    0x80, 0xa4, 0x3f, 0x69, 0x55, 0x01, 0x9e, 0x10, 0xa0, 0xe9, 0xb4, 0xb6,
    0x5e, 0x1d, 0xe6, 0x55, 0xd2, 0x62, 0x11, 0x60,
};
static const struct drbg_kat_no_reseed kat235_nor_t = {
    9, kat235_nor_entropyin, kat235_nor_nonce, kat235_nor_persstr,
    kat235_nor_addin0, kat235_nor_addin1, kat235_nor_retbytes
};
static const struct drbg_kat kat235_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat235_nor_t
};

static const unsigned char kat236_nor_entropyin[] = {
    0x83, 0x99, 0x83, 0x66, 0xce, 0x3e, 0x12, 0x20, 0x2f, 0x20, 0xaf, 0x4c,
    0x35, 0x56, 0x79, 0xf5,
};
static const unsigned char kat236_nor_nonce[] = {
    0x4d, 0xb8, 0x3f, 0x88, 0x5f, 0xcd, 0x68, 0x19,
};
static const unsigned char kat236_nor_persstr[] = {
    0x6e, 0x69, 0xe4, 0xad, 0x38, 0x71, 0x9f, 0x5a, 0x36, 0xf6, 0x24, 0xe3,
    0x76, 0x6f, 0xfe, 0xa0,
};
static const unsigned char kat236_nor_addin0[] = {
    0x4e, 0xd8, 0xec, 0xc9, 0xa4, 0x9f, 0xde, 0xe2, 0x86, 0x35, 0x5b, 0x49,
    0x4a, 0xf8, 0x6d, 0x9b,
};
static const unsigned char kat236_nor_addin1[] = {
    0xc1, 0xe7, 0x9f, 0xea, 0xcb, 0x75, 0x75, 0x24, 0x43, 0x42, 0xe0, 0x18,
    0x3d, 0x25, 0x45, 0x6c,
};
static const unsigned char kat236_nor_retbytes[] = {
    0xba, 0x1b, 0x0c, 0x90, 0x8e, 0x30, 0x15, 0xbb, 0xd8, 0x05, 0xdf, 0xfc,
    0x80, 0x3b, 0xfa, 0x93, 0xf1, 0x30, 0x6d, 0x30, 0xdc, 0x7f, 0x05, 0xd2,
    0xd4, 0xe7, 0x5d, 0x30, 0x63, 0x37, 0x1b, 0x94, 0x7e, 0xd7, 0xec, 0x98,
    0xc2, 0x4c, 0x9e, 0x2d, 0xe4, 0x3b, 0x37, 0x22, 0x0f, 0x85, 0xaf, 0x7d,
    0x4a, 0xd2, 0x63, 0x88, 0x94, 0x86, 0x93, 0x35, 0x8d, 0xa2, 0x3b, 0xde,
    0xe6, 0xf2, 0x25, 0xe8, 0x77, 0x69, 0x05, 0xda, 0x1c, 0x24, 0xb0, 0xf9,
    0x61, 0x25, 0xc0, 0x81, 0xa9, 0x0c, 0xe7, 0xcb,
};
static const struct drbg_kat_no_reseed kat236_nor_t = {
    10, kat236_nor_entropyin, kat236_nor_nonce, kat236_nor_persstr,
    kat236_nor_addin0, kat236_nor_addin1, kat236_nor_retbytes
};
static const struct drbg_kat kat236_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat236_nor_t
};

static const unsigned char kat237_nor_entropyin[] = {
    0x2a, 0x40, 0x08, 0x08, 0xe0, 0x3c, 0xed, 0x38, 0x42, 0xba, 0x1e, 0xb3,
    0xf1, 0x57, 0x5f, 0xe6,
};
static const unsigned char kat237_nor_nonce[] = {
    0xc6, 0xb9, 0xb8, 0xc3, 0x33, 0x06, 0x3c, 0x4e,
};
static const unsigned char kat237_nor_persstr[] = {
    0x44, 0x24, 0xf5, 0x3b, 0x70, 0xa4, 0xf4, 0x87, 0x30, 0xf8, 0x12, 0xb3,
    0xcf, 0xcf, 0x53, 0x9c,
};
static const unsigned char kat237_nor_addin0[] = {
    0x66, 0x5c, 0x17, 0x18, 0xa4, 0x88, 0x18, 0x6e, 0x3f, 0x7d, 0x0b, 0xb6,
    0x96, 0x76, 0x2a, 0xa6,
};
static const unsigned char kat237_nor_addin1[] = {
    0xb6, 0xd1, 0x3e, 0xbd, 0x72, 0x09, 0x43, 0x7e, 0xa5, 0xfa, 0xba, 0x33,
    0x81, 0x4b, 0x2c, 0x74,
};
static const unsigned char kat237_nor_retbytes[] = {
    0x2d, 0xc7, 0xa9, 0x54, 0xff, 0x43, 0x9e, 0xc8, 0x10, 0x8b, 0x06, 0x49,
    0xac, 0xd3, 0xc5, 0xd8, 0xb2, 0xe2, 0x8c, 0x88, 0x04, 0xcc, 0x9d, 0x24,
    0x9a, 0xca, 0xfe, 0xdc, 0x0e, 0xf8, 0x1a, 0xab, 0x1c, 0xae, 0xc3, 0x56,
    0xcc, 0x50, 0xc4, 0x3b, 0x70, 0xc1, 0xfe, 0x4d, 0xc1, 0x17, 0x29, 0x0b,
    0xb0, 0x64, 0x85, 0x88, 0xbf, 0xe8, 0x4c, 0x9b, 0x83, 0x96, 0x40, 0xc1,
    0x50, 0x34, 0x38, 0x76, 0x5c, 0xed, 0xb0, 0xe5, 0x89, 0xcd, 0x13, 0xfc,
    0x8c, 0xd8, 0x0b, 0xd9, 0x6a, 0xd1, 0x9c, 0x6f,
};
static const struct drbg_kat_no_reseed kat237_nor_t = {
    11, kat237_nor_entropyin, kat237_nor_nonce, kat237_nor_persstr,
    kat237_nor_addin0, kat237_nor_addin1, kat237_nor_retbytes
};
static const struct drbg_kat kat237_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat237_nor_t
};

static const unsigned char kat238_nor_entropyin[] = {
    0x6f, 0x7a, 0x7d, 0x94, 0xdc, 0xe0, 0x25, 0x90, 0xe7, 0x90, 0xb9, 0x71,
    0xaf, 0x9d, 0x42, 0x95,
};
static const unsigned char kat238_nor_nonce[] = {
    0xd7, 0x42, 0xb2, 0xe7, 0xe6, 0x10, 0x00, 0x44,
};
static const unsigned char kat238_nor_persstr[] = {
    0xb9, 0xdb, 0x42, 0x70, 0x26, 0x23, 0xc7, 0x50, 0xb8, 0xa5, 0x6f, 0x87,
    0xe9, 0xb8, 0x6c, 0xe3,
};
static const unsigned char kat238_nor_addin0[] = {
    0xd8, 0xaa, 0x16, 0x60, 0x02, 0x63, 0x46, 0xb6, 0x4c, 0x73, 0x04, 0x2c,
    0xcd, 0x7b, 0xe9, 0x11,
};
static const unsigned char kat238_nor_addin1[] = {
    0xb8, 0x42, 0xd8, 0x10, 0x5b, 0x29, 0xdc, 0xdd, 0x49, 0x3b, 0x37, 0x1b,
    0xfe, 0x5f, 0xd9, 0xd9,
};
static const unsigned char kat238_nor_retbytes[] = {
    0x5e, 0xb0, 0x08, 0xc4, 0xe9, 0xbd, 0xab, 0x61, 0x16, 0x28, 0xff, 0x58,
    0x74, 0x11, 0xa8, 0x26, 0x2c, 0x27, 0x58, 0x95, 0x81, 0x03, 0xbd, 0x26,
    0x6a, 0xa1, 0xaf, 0x38, 0x6e, 0xbc, 0x70, 0x48, 0x24, 0x81, 0xdc, 0x8d,
    0xd6, 0xa2, 0x53, 0x22, 0x2e, 0x34, 0x44, 0x4b, 0xe4, 0x5a, 0x6a, 0xa2,
    0x31, 0x11, 0x99, 0x87, 0x37, 0x9b, 0xd7, 0xff, 0xc4, 0x3f, 0x7a, 0x4a,
    0xb7, 0x93, 0xd1, 0xfe, 0x90, 0x85, 0x52, 0x64, 0xf1, 0xf9, 0x47, 0x33,
    0xe4, 0x24, 0xdd, 0x97, 0x38, 0x7a, 0x18, 0x0a,
};
static const struct drbg_kat_no_reseed kat238_nor_t = {
    12, kat238_nor_entropyin, kat238_nor_nonce, kat238_nor_persstr,
    kat238_nor_addin0, kat238_nor_addin1, kat238_nor_retbytes
};
static const struct drbg_kat kat238_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat238_nor_t
};

static const unsigned char kat239_nor_entropyin[] = {
    0x5c, 0x6a, 0xed, 0x6f, 0x74, 0xcc, 0x30, 0x2e, 0x86, 0xc5, 0x04, 0xf0,
    0x58, 0x6c, 0xdc, 0xbd,
};
static const unsigned char kat239_nor_nonce[] = {
    0xe7, 0xcc, 0x8c, 0xd4, 0x4f, 0x6d, 0xac, 0xe7,
};
static const unsigned char kat239_nor_persstr[] = {
    0x77, 0x11, 0x41, 0xd9, 0x42, 0x65, 0x8f, 0xe4, 0xd6, 0x58, 0xf4, 0xa8,
    0xaf, 0x0d, 0xc4, 0x71,
};
static const unsigned char kat239_nor_addin0[] = {
    0x5d, 0x7a, 0x91, 0xe1, 0x8f, 0x88, 0xbd, 0x50, 0x9a, 0x6a, 0x0d, 0x59,
    0x0e, 0x71, 0xc7, 0x61,
};
static const unsigned char kat239_nor_addin1[] = {
    0xd4, 0x6b, 0x57, 0xfa, 0x51, 0xa4, 0x21, 0xe2, 0x5a, 0xed, 0xf1, 0x11,
    0x15, 0x7d, 0xb8, 0xb9,
};
static const unsigned char kat239_nor_retbytes[] = {
    0xad, 0x1a, 0xda, 0x8c, 0xe7, 0x90, 0x45, 0xcb, 0x24, 0x74, 0x83, 0x78,
    0xb4, 0xc4, 0x62, 0x9d, 0xd9, 0x08, 0x05, 0x83, 0x77, 0x50, 0x24, 0x36,
    0xe6, 0xde, 0x51, 0xb8, 0xdb, 0xbd, 0x70, 0xd7, 0x21, 0xe8, 0x04, 0x95,
    0x5a, 0x1e, 0xcb, 0x02, 0xf8, 0xb1, 0x1d, 0x1d, 0x44, 0xd9, 0xf5, 0x01,
    0x18, 0x08, 0xc5, 0x6a, 0x6a, 0xfe, 0x94, 0xc7, 0x7c, 0x15, 0x31, 0x9d,
    0x8e, 0x1f, 0x48, 0xc5, 0x02, 0x34, 0xd1, 0x32, 0x28, 0xfb, 0xc8, 0x4e,
    0x13, 0xb4, 0x44, 0x0e, 0xac, 0xf4, 0xcd, 0x21,
};
static const struct drbg_kat_no_reseed kat239_nor_t = {
    13, kat239_nor_entropyin, kat239_nor_nonce, kat239_nor_persstr,
    kat239_nor_addin0, kat239_nor_addin1, kat239_nor_retbytes
};
static const struct drbg_kat kat239_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat239_nor_t
};

static const unsigned char kat240_nor_entropyin[] = {
    0xb3, 0x0e, 0xd3, 0x21, 0xb4, 0x60, 0x9c, 0x2c, 0x1b, 0x51, 0x23, 0x59,
    0x63, 0x95, 0x8e, 0xa5,
};
static const unsigned char kat240_nor_nonce[] = {
    0x5b, 0xf9, 0xbc, 0x2f, 0xee, 0x57, 0x5e, 0xb4,
};
static const unsigned char kat240_nor_persstr[] = {
    0x8f, 0xbd, 0x86, 0xda, 0x07, 0x24, 0x35, 0x5a, 0x40, 0x59, 0xf6, 0xac,
    0x78, 0xfe, 0x63, 0x42,
};
static const unsigned char kat240_nor_addin0[] = {
    0xfc, 0x54, 0xb5, 0x33, 0x9b, 0x37, 0xeb, 0x68, 0x89, 0xcf, 0xd7, 0xc1,
    0x85, 0x07, 0x0b, 0xd0,
};
static const unsigned char kat240_nor_addin1[] = {
    0xf6, 0xa7, 0x83, 0xd6, 0xd4, 0x2e, 0x5a, 0xd5, 0xab, 0xb0, 0xa9, 0x96,
    0xbd, 0xdf, 0xa0, 0x4c,
};
static const unsigned char kat240_nor_retbytes[] = {
    0x68, 0x3f, 0xaa, 0x73, 0x2c, 0x45, 0x51, 0x60, 0x4c, 0x88, 0x65, 0xb5,
    0xf7, 0x77, 0x57, 0x1c, 0x7d, 0x3c, 0xf1, 0xa6, 0x01, 0x24, 0xc5, 0x9b,
    0x91, 0x28, 0x3d, 0xa0, 0xcd, 0xa9, 0xb2, 0x17, 0x61, 0xd1, 0xc1, 0x7c,
    0x81, 0x85, 0x69, 0x58, 0xc6, 0xd5, 0x90, 0x43, 0x6c, 0x73, 0x59, 0x4b,
    0xb3, 0x6f, 0x46, 0xc2, 0xf8, 0x92, 0x37, 0xd8, 0xc7, 0xa7, 0xdd, 0xd2,
    0xc5, 0x83, 0x94, 0xc9, 0x83, 0xf8, 0xf6, 0xc0, 0x00, 0xd7, 0x75, 0x66,
    0xf2, 0xa1, 0xd8, 0x9b, 0xac, 0x05, 0x4b, 0xdb,
};
static const struct drbg_kat_no_reseed kat240_nor_t = {
    14, kat240_nor_entropyin, kat240_nor_nonce, kat240_nor_persstr,
    kat240_nor_addin0, kat240_nor_addin1, kat240_nor_retbytes
};
static const struct drbg_kat kat240_nor = {
    NO_RESEED, NA, NID_sha1, 16, 8, 16, 16, 80, &kat240_nor_t
};

static const unsigned char kat480_nor_entropyin[] = {
    0x63, 0xb6, 0x77, 0x16, 0x61, 0xab, 0x68, 0x98, 0x09, 0xcc, 0xc5, 0xe5,
    0x59, 0x1b, 0xd1, 0x74, 0x39, 0x21, 0xd6, 0xed, 0x1e, 0x1d, 0xdd, 0x01,
};
static const unsigned char kat480_nor_nonce[] = {
    0xb6, 0x34, 0xe9, 0x89, 0xec, 0x3b, 0xd5, 0xb5, 0x97, 0x77, 0xb2, 0x7d,
};
static const unsigned char kat480_nor_persstr[] = {
    0x82, 0x2d, 0xe3, 0x95, 0x2f, 0xf3, 0x6c, 0xf2, 0xd4, 0xe4, 0x02, 0x7c,
    0x12, 0xc4, 0xf7, 0xcd, 0x66, 0x5c, 0x08, 0xd9, 0x29, 0x59, 0x1b, 0x21,
};
static const unsigned char kat480_nor_addin0[] = {
    0xe5, 0xc6, 0x33, 0xca, 0x50, 0xdc, 0xd8, 0x3e, 0x0a, 0x34, 0xd3, 0x97,
    0xdf, 0x53, 0xf6, 0xd7, 0xa6, 0xf7, 0x17, 0x0a, 0x3f, 0x81, 0xf0, 0xe6,
};
static const unsigned char kat480_nor_addin1[] = {
    0x5f, 0x0b, 0xeb, 0x5a, 0x2d, 0x29, 0x68, 0xe8, 0x3b, 0xa8, 0x7c, 0x92,
    0xbf, 0xa4, 0x20, 0xfd, 0x6e, 0x85, 0x26, 0xfb, 0xbf, 0xde, 0xa1, 0x28,
};
static const unsigned char kat480_nor_retbytes[] = {
    0x8b, 0xec, 0x11, 0xdf, 0x10, 0x22, 0xaa, 0x50, 0xd9, 0x5d, 0xae, 0xaf,
    0x23, 0xd7, 0x8d, 0x6e, 0xe4, 0x5c, 0x43, 0xc5, 0x76, 0x8b, 0x90, 0x18,
    0x1e, 0x10, 0x6c, 0x7d, 0xf8, 0xff, 0x33, 0x3d, 0x7c, 0xb8, 0x7c, 0xa1,
    0xab, 0x83, 0xf8, 0x74, 0x23, 0x70, 0xdb, 0x1c, 0x8c, 0x0c, 0x0c, 0x22,
    0xf1, 0x41, 0xff, 0x4d, 0xe3, 0x3a, 0xe8, 0xbd, 0xb1, 0x4f, 0xee, 0x7e,
    0x6c, 0x06, 0x98, 0x19, 0x32, 0x06, 0x29, 0xc6, 0x6d, 0x94, 0xc7, 0xc9,
    0x7f, 0xf5, 0x29, 0x30, 0xa3, 0xc1, 0xdc, 0xd5, 0x01, 0xb6, 0x0f, 0x0f,
    0x84, 0xbd, 0xa4, 0x72, 0x0e, 0xe1, 0x87, 0xae, 0x85, 0x8a, 0x6e, 0x06,
    0x83, 0x26, 0xed, 0xa5, 0x80, 0x97, 0x16, 0xe3, 0x66, 0xd1, 0xb6, 0x08,
    0xc6, 0x1b, 0x01, 0x00,
};
static const struct drbg_kat_no_reseed kat480_nor_t = {
    14, kat480_nor_entropyin, kat480_nor_nonce, kat480_nor_persstr,
    kat480_nor_addin0, kat480_nor_addin1, kat480_nor_retbytes
};
static const struct drbg_kat kat480_nor = {
    NO_RESEED, NA, NID_sha224, 24, 12, 24, 24, 112, &kat480_nor_t
};

static const unsigned char kat720_nor_entropyin[] = {
    0x70, 0x8f, 0x5c, 0x7e, 0x10, 0xd5, 0x6c, 0x30, 0x45, 0x6e, 0x9e, 0x3f,
    0x58, 0x69, 0xa9, 0x5a, 0x05, 0x1c, 0xf7, 0xcf, 0x10, 0x77, 0x08, 0xc4,
    0x0c, 0xcc, 0xb8, 0x01, 0xa3, 0xa7, 0x63, 0x6d,
};
static const unsigned char kat720_nor_nonce[] = {
    0xba, 0xc7, 0x55, 0xe7, 0x01, 0xb1, 0xfd, 0xd1, 0x70, 0xdc, 0x92, 0xeb,
    0x0f, 0x12, 0xbd, 0x4b,
};
static const unsigned char kat720_nor_persstr[] = {
    0xf1, 0xdb, 0x3a, 0xcd, 0x73, 0x91, 0x4a, 0x62, 0xbc, 0xa9, 0xa5, 0xce,
    0x34, 0x97, 0xb1, 0xb4, 0xda, 0x12, 0xb2, 0x9c, 0xee, 0x7d, 0xc4, 0x78,
    0x1d, 0x7c, 0xad, 0xdf, 0x67, 0xe4, 0x5f, 0x2c,
};
static const unsigned char kat720_nor_addin0[] = {
    0x22, 0x85, 0x22, 0xe5, 0x8e, 0x65, 0xd5, 0x0d, 0xfd, 0x17, 0x6e, 0x8f,
    0xf1, 0x74, 0x9f, 0xaa, 0x70, 0xfc, 0x2c, 0x82, 0xed, 0xa2, 0x5b, 0x07,
    0x48, 0xdd, 0xc5, 0xd4, 0x1f, 0xd7, 0x1b, 0xe5,
};
static const unsigned char kat720_nor_addin1[] = {
    0x7a, 0xf6, 0x0c, 0x47, 0xb4, 0xcd, 0x14, 0x6a, 0x39, 0x88, 0x7c, 0x9b,
    0x81, 0x2a, 0x1d, 0xd8, 0x14, 0xd7, 0x4c, 0x39, 0x86, 0x09, 0xbb, 0xbf,
    0xb5, 0x7e, 0x73, 0xda, 0x9c, 0xaf, 0xf5, 0x7a,
};
static const unsigned char kat720_nor_retbytes[] = {
    0x95, 0x28, 0xc8, 0x8f, 0x0a, 0xea, 0x3f, 0xc0, 0x3b, 0xb8, 0xa9, 0x06,
    0x1e, 0x15, 0x9a, 0x06, 0xd7, 0x8a, 0x2a, 0x65, 0x44, 0x08, 0x80, 0x8a,
    0xa4, 0xd0, 0xe7, 0x3a, 0xb1, 0xa5, 0x1e, 0x5a, 0xa8, 0x5e, 0x8b, 0xca,
    0xe7, 0x2d, 0x34, 0x78, 0x4f, 0xf6, 0xf5, 0x13, 0x19, 0x3e, 0x18, 0x3d,
    0x55, 0x6d, 0xda, 0xc5, 0x67, 0x53, 0x14, 0xf2, 0xb5, 0xcf, 0xe3, 0x92,
    0xd1, 0x52, 0x60, 0x56, 0xaf, 0xe3, 0x2d, 0x7c, 0x03, 0xe0, 0x9b, 0xa2,
    0xbd, 0xf3, 0xb1, 0x0e, 0x22, 0x8b, 0x0f, 0x60, 0x0a, 0x61, 0xcc, 0xcd,
    0x9e, 0x7b, 0xf1, 0x4d, 0xcc, 0xf1, 0x3b, 0x16, 0xa8, 0x38, 0xe6, 0x09,
    0x09, 0x78, 0x53, 0x07, 0xe6, 0x90, 0x5d, 0x51, 0x0d, 0x98, 0x88, 0xea,
    0xab, 0x16, 0x9f, 0xa6, 0x01, 0x55, 0x8f, 0xc9, 0x52, 0xaa, 0x85, 0x59,
    0xd2, 0x70, 0xec, 0xd3, 0x86, 0xd7, 0xfb, 0xd7,
};
static const struct drbg_kat_no_reseed kat720_nor_t = {
    14, kat720_nor_entropyin, kat720_nor_nonce, kat720_nor_persstr,
    kat720_nor_addin0, kat720_nor_addin1, kat720_nor_retbytes
};
static const struct drbg_kat kat720_nor = {
    NO_RESEED, NA, NID_sha256, 32, 16, 32, 32, 128, &kat720_nor_t
};

static const unsigned char kat960_nor_entropyin[] = {
    0x16, 0x1b, 0xef, 0x11, 0x9a, 0xf7, 0x36, 0xfd, 0x23, 0xe1, 0x88, 0xa2,
    0x27, 0xaa, 0xe7, 0x6e, 0xdd, 0x0f, 0xe6, 0x54, 0xf9, 0xf5, 0x83, 0xd7,
    0x5d, 0x77, 0xa3, 0x76, 0x4b, 0xb8, 0x47, 0x9f,
};
static const unsigned char kat960_nor_nonce[] = {
    0xc6, 0xe1, 0x4c, 0x68, 0x30, 0x9c, 0xa8, 0x47, 0x88, 0xad, 0x9f, 0x15,
    0xd5, 0xf0, 0xa9, 0x0a,
};
static const unsigned char kat960_nor_persstr[] = {
    0x8b, 0x20, 0x39, 0xa6, 0xd4, 0xb9, 0x09, 0x90, 0x6a, 0x87, 0x94, 0xdf,
    0x59, 0x95, 0xfe, 0x9e, 0xa9, 0x77, 0xc5, 0xa1, 0xa3, 0x09, 0x82, 0x01,
    0x5a, 0xad, 0xf9, 0x37, 0xc8, 0x63, 0xcf, 0xac,
};
static const unsigned char kat960_nor_addin0[] = {
    0x23, 0xe4, 0xe6, 0xb0, 0xe0, 0xc1, 0xb2, 0x8a, 0x6f, 0x97, 0x31, 0xf8,
    0xb0, 0x99, 0x60, 0xce, 0x7a, 0xda, 0xc1, 0x75, 0x27, 0xb3, 0xbb, 0xac,
    0xa7, 0xc8, 0x11, 0xda, 0xea, 0x4b, 0x1c, 0xf5,
};
static const unsigned char kat960_nor_addin1[] = {
    0xdc, 0x7f, 0xac, 0x6a, 0xed, 0xed, 0x9e, 0x17, 0xb5, 0xbb, 0x5e, 0x2b,
    0xca, 0xd9, 0x42, 0x4d, 0x42, 0xdc, 0x07, 0xe8, 0x09, 0xda, 0x59, 0xd5,
    0x2c, 0xae, 0xcb, 0xa6, 0xe7, 0x5c, 0xa4, 0x57,
};
static const unsigned char kat960_nor_retbytes[] = {
    0x5a, 0x42, 0xb3, 0x5c, 0xf1, 0xb7, 0x2d, 0x25, 0x20, 0xd9, 0x27, 0x19,
    0xa9, 0x4e, 0xf1, 0xa7, 0xca, 0x5b, 0x6d, 0x6c, 0x7e, 0xef, 0x2d, 0xe2,
    0x5c, 0x8e, 0xa4, 0x4c, 0x1f, 0xc3, 0xa9, 0xa5, 0xff, 0x21, 0x28, 0xf4,
    0x7b, 0xbe, 0x58, 0x08, 0x4a, 0x0c, 0x7a, 0x3f, 0xc7, 0x90, 0x62, 0x6e,
    0xff, 0x56, 0x66, 0xb4, 0xc1, 0xe6, 0x8f, 0xb2, 0xf5, 0x3d, 0xe3, 0x37,
    0x0b, 0x29, 0xc3, 0x98, 0xd5, 0x06, 0x7b, 0x25, 0x5f, 0x5f, 0x7f, 0x29,
    0xfd, 0xb0, 0xf8, 0xbc, 0x25, 0x6e, 0xe3, 0xaf, 0xbe, 0x78, 0xa3, 0x39,
    0x81, 0x62, 0x68, 0x37, 0xc5, 0x5f, 0x98, 0x1e, 0x56, 0xeb, 0x2e, 0x1b,
    0xdd, 0x89, 0xca, 0x08, 0x1e, 0x48, 0xf6, 0xda, 0x7c, 0xe6, 0x57, 0x6f,
    0xbd, 0x37, 0xdb, 0xd5, 0x7a, 0x3f, 0x41, 0xcf, 0x41, 0x0c, 0xb3, 0x75,
    0x61, 0x4a, 0xf2, 0x39, 0xf2, 0xe1, 0x02, 0x18, 0xe7, 0x77, 0xfb, 0x97,
    0xa5, 0x5d, 0x9c, 0xc7, 0x32, 0x43, 0x88, 0x2b, 0x8d, 0x8d, 0x2a, 0x2c,
    0x81, 0x2f, 0xbd, 0xea, 0xae, 0xd9, 0x0b, 0x5b, 0xd7, 0x1a, 0x27, 0x4b,
    0x4b, 0x17, 0x1c, 0xd7, 0xe6, 0x61, 0x91, 0x2c, 0x9b, 0x3d, 0xe1, 0x71,
    0x4a, 0x3f, 0xe4, 0x93, 0x1d, 0x8f, 0xc7, 0xcb, 0x1c, 0x9f, 0x64, 0xf4,
    0xe3, 0x7d, 0x4e, 0x5d, 0xbc, 0x31, 0x60, 0x2d, 0x2f, 0x86, 0x99, 0xe0,
};
static const struct drbg_kat_no_reseed kat960_nor_t = {
    14, kat960_nor_entropyin, kat960_nor_nonce, kat960_nor_persstr,
    kat960_nor_addin0, kat960_nor_addin1, kat960_nor_retbytes
};
static const struct drbg_kat kat960_nor = {
    NO_RESEED, NA, NID_sha384, 32, 16, 32, 32, 192, &kat960_nor_t
};

static const unsigned char kat1200_nor_entropyin[] = {
    0xb1, 0x48, 0xb3, 0x17, 0xa2, 0x68, 0x62, 0x8f, 0x04, 0xc9, 0xa8, 0x7a,
    0x0a, 0xc5, 0xf9, 0x14, 0x98, 0xb8, 0x02, 0x0f, 0x4e, 0x48, 0x35, 0x72,
    0x0c, 0x1a, 0x3c, 0xc0, 0x7b, 0x05, 0x0f, 0xa2,
};
static const unsigned char kat1200_nor_nonce[] = {
    0x2f, 0x35, 0xe5, 0xb6, 0x22, 0xae, 0xef, 0xe7, 0x56, 0x05, 0xc2, 0x27,
    0x4e, 0xc8, 0xc6, 0x96,
};
static const unsigned char kat1200_nor_persstr[] = {
    0xfc, 0x52, 0x7a, 0x2f, 0x16, 0xb5, 0x3c, 0x51, 0x3f, 0x94, 0x85, 0x5b,
    0x35, 0xce, 0xa6, 0x09, 0x0c, 0x30, 0x3d, 0xcc, 0x64, 0x2e, 0x98, 0xed,
    0x5f, 0x32, 0x3a, 0xba, 0x0f, 0x35, 0xfa, 0x27,
};
static const unsigned char kat1200_nor_addin0[] = {
    0x2c, 0xc9, 0xf1, 0x37, 0xfc, 0xd8, 0xc2, 0xd5, 0x26, 0xd7, 0x00, 0x93,
    0xfe, 0x11, 0xf9, 0x0a, 0x0a, 0x36, 0xbc, 0x97, 0x64, 0xa4, 0xc5, 0x60,
    0x90, 0x72, 0xe1, 0x81, 0xa2, 0x49, 0x45, 0x16,
};
static const unsigned char kat1200_nor_addin1[] = {
    0xe4, 0x03, 0x61, 0x24, 0x5b, 0x91, 0x88, 0x0e, 0x30, 0x8f, 0xb7, 0x77,
    0xc2, 0x8b, 0xbf, 0xae, 0xa5, 0x98, 0x2e, 0x45, 0xfe, 0xcb, 0x77, 0x57,
    0xbb, 0x1c, 0x9d, 0xe2, 0xdf, 0x9d, 0xc6, 0x12,
};
static const unsigned char kat1200_nor_retbytes[] = {
    0x66, 0xad, 0x04, 0x8b, 0x4d, 0x2d, 0x00, 0x32, 0x23, 0xc6, 0x4d, 0xd9,
    0x82, 0x7c, 0xc2, 0x2e, 0xd3, 0xec, 0x8f, 0xcb, 0x61, 0x20, 0x9d, 0x19,
    0x96, 0x19, 0x17, 0x75, 0x92, 0xe9, 0xb8, 0x92, 0x26, 0xbe, 0x30, 0xb1,
    0x93, 0x0b, 0xdd, 0x74, 0x9f, 0x30, 0xed, 0x09, 0xda, 0x52, 0xab, 0xaa,
    0x2e, 0x59, 0x9a, 0xfa, 0xf9, 0x19, 0x03, 0xe7, 0xa2, 0xb5, 0x9f, 0xfb,
    0x8f, 0xd4, 0x70, 0xe6, 0x60, 0x44, 0x85, 0xa2, 0x7c, 0x20, 0x0d, 0x37,
    0x5f, 0xef, 0xf6, 0x21, 0x11, 0x85, 0x95, 0xa7, 0xa3, 0x05, 0x7b, 0x7e,
    0x31, 0xea, 0xdc, 0x06, 0x87, 0xb1, 0x00, 0x8c, 0x3c, 0xb2, 0xc7, 0x43,
    0x5a, 0x57, 0x04, 0xb1, 0xa1, 0xa6, 0xa3, 0x48, 0x7d, 0x60, 0xfd, 0x14,
    0x79, 0x3c, 0x31, 0x48, 0x6a, 0xf7, 0x65, 0xce, 0x2c, 0xe1, 0x82, 0xde,
    0x88, 0x11, 0x24, 0x45, 0xdd, 0x5f, 0xf1, 0x1b, 0x25, 0x6c, 0xfd, 0xa0,
    0x70, 0x18, 0xb9, 0x5f, 0x97, 0xed, 0xba, 0xb4, 0xe4, 0xc3, 0x9c, 0xa0,
    0x97, 0xc4, 0x2f, 0x9d, 0xce, 0x80, 0xcd, 0x3f, 0x32, 0x67, 0x7f, 0x3c,
    0x22, 0x4a, 0x86, 0xb3, 0x15, 0xd0, 0x2e, 0x37, 0x7d, 0xca, 0x8f, 0x37,
    0x85, 0xe9, 0x74, 0x8f, 0xfd, 0xbe, 0x3f, 0xca, 0xa3, 0xb0, 0xc6, 0xbf,
    0x00, 0x1b, 0x63, 0xb5, 0x74, 0x26, 0x83, 0x63, 0x58, 0xe9, 0xb3, 0x15,
    0xc6, 0x71, 0x8e, 0x0b, 0x74, 0xfb, 0x82, 0xb9, 0xbf, 0x3d, 0xf7, 0x00,
    0xa6, 0x41, 0xab, 0x94, 0x11, 0xd1, 0xb9, 0xfb, 0xa4, 0x23, 0x09, 0xa8,
    0x4b, 0xef, 0x67, 0xa1, 0x42, 0x04, 0xf3, 0x16, 0x0e, 0xd1, 0x6a, 0x54,
    0x97, 0xfe, 0x21, 0x1a, 0xa1, 0xf5, 0xd3, 0xae, 0x4b, 0x85, 0x8b, 0x6d,
    0x44, 0x5f, 0x1d, 0x09, 0x45, 0x43, 0xd0, 0x10, 0x7c, 0xe0, 0x4e, 0xf1,
    0xd1, 0xba, 0x33, 0xab,
};
static const struct drbg_kat_no_reseed kat1200_nor_t = {
    14, kat1200_nor_entropyin, kat1200_nor_nonce, kat1200_nor_persstr,
    kat1200_nor_addin0, kat1200_nor_addin1, kat1200_nor_retbytes
};
static const struct drbg_kat kat1200_nor = {
    NO_RESEED, NA, NID_sha512, 32, 16, 32, 32, 256, &kat1200_nor_t
};

static const unsigned char kat1440_nor_entropyin[] = {
    0x55, 0x86, 0xa1, 0x84, 0x3e, 0x05, 0x8e, 0x09, 0x64, 0xb7, 0x64, 0x45,
    0xf6, 0x1a, 0xe4, 0xbe, 0x69, 0x71, 0xcf, 0xfa, 0xcc, 0x05, 0xed, 0xfa,
};
static const unsigned char kat1440_nor_nonce[] = {
    0x31, 0x43, 0x32, 0x4d, 0x2f, 0x3c, 0x14, 0xf6, 0x06, 0x9d, 0xb0, 0xdd,
};
static const unsigned char kat1440_nor_persstr[] = {
    0x57, 0xd8, 0xbf, 0x22, 0xe4, 0x1c, 0x7b, 0xa6, 0xe7, 0xbc, 0xbb, 0xef,
    0xf5, 0x77, 0x45, 0x11, 0x44, 0x5f, 0x07, 0x0d, 0xa4, 0x08, 0x3b, 0x14,
};
static const unsigned char kat1440_nor_addin0[] = {
    0x1b, 0x87, 0x25, 0x44, 0x7e, 0xc5, 0x39, 0xea, 0x4a, 0x13, 0xc4, 0x7b,
    0x32, 0x3f, 0x1d, 0x6f, 0x43, 0x5b, 0xa7, 0xe6, 0x24, 0xdc, 0xf5, 0xaf,
};
static const unsigned char kat1440_nor_addin1[] = {
    0x86, 0xd3, 0x0a, 0xf4, 0x0a, 0x7a, 0x39, 0x57, 0x64, 0xb8, 0xb6, 0x9f,
    0x26, 0x56, 0x95, 0x4c, 0x7c, 0x3f, 0x1c, 0x30, 0xb2, 0xb7, 0x03, 0xb0,
};
static const unsigned char kat1440_nor_retbytes[] = {
    0x2f, 0xb2, 0xf2, 0x4b, 0x2c, 0x38, 0xf2, 0x17, 0x23, 0x2d, 0xc2, 0x2e,
    0xcc, 0x73, 0x80, 0xb8, 0x24, 0x0b, 0x05, 0xd2, 0xc7, 0xbc, 0x0e, 0x3d,
    0xfd, 0xad, 0x26, 0x8c, 0x8c, 0x10, 0x91, 0x2a, 0x92, 0x59, 0x5d, 0x70,
    0xdd, 0x98, 0xe7, 0xec, 0xdb, 0xdc, 0x6d, 0x7b, 0xce, 0x6c, 0x72, 0xcd,
    0xeb, 0xd7, 0xe1, 0x21, 0xd7, 0x5d, 0xe8, 0xb6, 0x79, 0x5b, 0x66, 0x0b,
    0xe9, 0x09, 0x6a, 0x1f, 0x24, 0xa9, 0x7e, 0x9c, 0x53, 0x44, 0xc3, 0x5f,
    0x04, 0x45, 0x1d, 0xbd, 0x8d, 0x98, 0x08, 0xc7, 0xa8, 0x4c, 0x6f, 0xba,
    0xfa, 0xb6, 0xd0, 0x60, 0x02, 0x64, 0x90, 0xd4, 0x92, 0x06, 0x0f, 0x05,
    0x2f, 0xbf, 0x21, 0xa3, 0xbf, 0xa2, 0xa8, 0xe4, 0xa4, 0x0d, 0xb5, 0x86,
    0x72, 0xca, 0x52, 0xce,
};
static const struct drbg_kat_no_reseed kat1440_nor_t = {
    14, kat1440_nor_entropyin, kat1440_nor_nonce, kat1440_nor_persstr,
    kat1440_nor_addin0, kat1440_nor_addin1, kat1440_nor_retbytes
};
static const struct drbg_kat kat1440_nor = {
    NO_RESEED, NA, NID_sha512_224, 24, 12, 24, 24, 112, &kat1440_nor_t
};

static const unsigned char kat1680_nor_entropyin[] = {
    0x1b, 0xeb, 0xff, 0x7d, 0x14, 0x50, 0xb3, 0xbb, 0xa8, 0xeb, 0x89, 0xee,
    0x19, 0x8f, 0x53, 0x83, 0xa5, 0xe9, 0xd9, 0xfc, 0x7b, 0x35, 0xc0, 0x82,
    0x13, 0x7f, 0x44, 0xb1, 0x52, 0x3f, 0xda, 0xde,
};
static const unsigned char kat1680_nor_nonce[] = {
    0x96, 0xa4, 0x10, 0x18, 0x57, 0xa3, 0x90, 0x4d, 0xd6, 0xeb, 0xd1, 0x01,
    0xe7, 0x3c, 0x9d, 0x5f,
};
static const unsigned char kat1680_nor_persstr[] = {
    0x32, 0x78, 0xe5, 0xa6, 0xfd, 0x03, 0xd3, 0xf7, 0xbb, 0x31, 0x51, 0x16,
    0xb7, 0xd1, 0x72, 0x01, 0x06, 0xe7, 0x7d, 0xef, 0xa3, 0xa8, 0x2a, 0x2c,
    0x68, 0x1d, 0x6d, 0x91, 0xa6, 0xeb, 0x17, 0x2c,
};
static const unsigned char kat1680_nor_addin0[] = {
    0x95, 0xf4, 0xb7, 0x87, 0x1a, 0x64, 0x1c, 0x5c, 0xc4, 0xf8, 0xdd, 0xb8,
    0xcb, 0xd9, 0x87, 0xd3, 0xb0, 0x93, 0x5c, 0xb2, 0xf5, 0x5c, 0x77, 0xce,
    0x34, 0x03, 0x56, 0x33, 0xc8, 0x5c, 0x85, 0xcc,
};
static const unsigned char kat1680_nor_addin1[] = {
    0xb4, 0x94, 0xe2, 0xaf, 0x5b, 0x70, 0x8a, 0xc3, 0x06, 0x72, 0xb8, 0xce,
    0x57, 0xe7, 0x20, 0x40, 0xe8, 0x23, 0x3c, 0x53, 0x80, 0x01, 0xa3, 0x48,
    0xe2, 0xcb, 0xa9, 0x8a, 0x8c, 0xd1, 0x2a, 0xcb,
};
static const unsigned char kat1680_nor_retbytes[] = {
    0xc7, 0x31, 0xcc, 0x7b, 0x21, 0xc4, 0x27, 0x30, 0xbd, 0x3c, 0xca, 0x61,
    0xfc, 0x52, 0x50, 0xb5, 0x07, 0xad, 0x08, 0xb2, 0x4a, 0xc4, 0x71, 0xd5,
    0x26, 0xf2, 0x21, 0x7f, 0x15, 0xdc, 0x4d, 0x1f, 0xea, 0x85, 0xb5, 0x7e,
    0xa9, 0xcf, 0x6b, 0x3d, 0xe9, 0xf6, 0x50, 0xdf, 0x82, 0x26, 0x32, 0x6f,
    0x8d, 0xa0, 0xa7, 0x66, 0xc6, 0xd3, 0xd9, 0xab, 0x77, 0x13, 0x93, 0x8d,
    0x29, 0xd9, 0xd3, 0x3a, 0xe2, 0xff, 0xad, 0x65, 0x82, 0x10, 0x75, 0x12,
    0x86, 0x89, 0x65, 0x27, 0x50, 0x4e, 0xa2, 0x3d, 0x6e, 0x4d, 0xaf, 0xa1,
    0x0f, 0x6e, 0x62, 0x6a, 0x8b, 0x1e, 0x4b, 0x38, 0x6d, 0x2a, 0x27, 0xae,
    0x12, 0xcd, 0x8c, 0xeb, 0x1f, 0x50, 0x24, 0x41, 0x8b, 0x70, 0xc0, 0x05,
    0xb3, 0x5f, 0x2c, 0x89, 0x8a, 0x0f, 0xae, 0xf7, 0x4e, 0xb5, 0x60, 0x44,
    0x40, 0x79, 0x30, 0xdc, 0x1c, 0xf7, 0x3f, 0x70,
};
static const struct drbg_kat_no_reseed kat1680_nor_t = {
    14, kat1680_nor_entropyin, kat1680_nor_nonce, kat1680_nor_persstr,
    kat1680_nor_addin0, kat1680_nor_addin1, kat1680_nor_retbytes
};
static const struct drbg_kat kat1680_nor = {
    NO_RESEED, NA, NID_sha512_256, 32, 16, 32, 32, 128, &kat1680_nor_t
};

/* ------------------------------------------------------------------------ */

static const unsigned char kat240_prt_entropyin[] = {
    0x25, 0xa6, 0xd2, 0x6c, 0x8a, 0x9d, 0x73, 0xfb, 0x1a, 0x4c, 0xba, 0x5e,
    0x39, 0x48, 0xda, 0xef,
};
static const unsigned char kat240_prt_nonce[] = {
    0xe7, 0xda, 0xfc, 0xd8, 0x1a, 0xc1, 0xae, 0xba,
};
static const unsigned char kat240_prt_persstr[] = {
    0xa8, 0xb8, 0x2f, 0xc8, 0xe8, 0x87, 0x46, 0x15, 0x8f, 0xd4, 0x01, 0x0a,
    0x1d, 0x3d, 0x97, 0xe0,
};
static const unsigned char kat240_prt_entropyinpr0[] = {
    0x57, 0xa5, 0x71, 0x72, 0xab, 0x86, 0xfc, 0x29, 0x00, 0x4b, 0x9d, 0x41,
    0x2a, 0x97, 0x01, 0xf1,
};
static const unsigned char kat240_prt_entropyinpr1[] = {
    0x7b, 0x55, 0x8b, 0x48, 0xf3, 0xc8, 0x91, 0xa7, 0x7f, 0xed, 0x29, 0x38,
    0x81, 0x77, 0x51, 0x18,
};
static const unsigned char kat240_prt_addin0[] = {
    0x6e, 0xcf, 0x52, 0xf9, 0x5b, 0x44, 0x43, 0xec, 0x8a, 0xc5, 0x2f, 0xae,
    0x74, 0xe1, 0x93, 0x29,
};
static const unsigned char kat240_prt_addin1[] = {
    0xb0, 0x71, 0x98, 0xa4, 0x9b, 0xc8, 0x54, 0xcf, 0xc9, 0xd6, 0xd7, 0x46,
    0x6f, 0xe2, 0x49, 0x48,
};
static const unsigned char kat240_prt_retbytes[] = {
    0x87, 0x8d, 0x26, 0xfb, 0x57, 0x58, 0x9d, 0x42, 0x49, 0x7b, 0x86, 0x95,
    0x64, 0xa1, 0xda, 0xc5, 0xad, 0xf1, 0xb8, 0x36, 0x15, 0xf9, 0xab, 0x9f,
    0xc3, 0x0b, 0x51, 0x40, 0xf7, 0x9e, 0x3b, 0x7f, 0x52, 0x5f, 0x1e, 0xff,
    0x2e, 0x68, 0x00, 0x28, 0x01, 0x93, 0x9a, 0xa0, 0x72, 0x84, 0x32, 0xef,
    0xad, 0x82, 0x9b, 0x5b, 0x12, 0x49, 0x14, 0x04, 0xfb, 0x50, 0xf2, 0x58,
    0x4a, 0x3b, 0xde, 0xa8, 0x78, 0x5e, 0x79, 0x39, 0x05, 0x01, 0x97, 0x87,
    0x04, 0xa6, 0x67, 0xec, 0x5d, 0x04, 0xda, 0x56,
};
static const struct drbg_kat_pr_true kat240_prt_t = {
    14, kat240_prt_entropyin, kat240_prt_nonce, kat240_prt_persstr,
    kat240_prt_entropyinpr0, kat240_prt_addin0, kat240_prt_entropyinpr1,
    kat240_prt_addin1, kat240_prt_retbytes
};
static const struct drbg_kat kat240_prt = {
    PR_TRUE, NA, NID_sha1, 16, 8, 16, 16, 80, &kat240_prt_t
};

static const unsigned char kat480_prt_entropyin[] = {
    0xdb, 0x97, 0x7c, 0x5b, 0x3a, 0x1e, 0x79, 0x61, 0xe6, 0x4d, 0x99, 0x66,
    0x65, 0x16, 0xfe, 0x56, 0x12, 0xb1, 0x51, 0x95, 0x6e, 0xc5, 0x1a, 0xdd,
};
static const unsigned char kat480_prt_nonce[] = {
    0x05, 0xfa, 0x09, 0x36, 0xe2, 0x83, 0x6c, 0xd2, 0xe8, 0xe4, 0xcc, 0xe9,
};
static const unsigned char kat480_prt_persstr[] = {
    0x4e, 0x8c, 0x5f, 0xfc, 0x09, 0x98, 0xb2, 0x69, 0x64, 0xab, 0x2b, 0x42,
    0xcf, 0x21, 0x3d, 0x1c, 0x43, 0x17, 0x94, 0xbc, 0x40, 0x74, 0x76, 0x17,
};
static const unsigned char kat480_prt_entropyinpr0[] = {
    0x16, 0xda, 0x81, 0x89, 0x74, 0xeb, 0x93, 0x58, 0x0d, 0xb3, 0x0f, 0x40,
    0x0c, 0xff, 0x3e, 0xe1, 0xa3, 0xaf, 0x01, 0xe1, 0x21, 0xe3, 0xcd, 0xb4,
};
static const unsigned char kat480_prt_entropyinpr1[] = {
    0x72, 0xf6, 0x4b, 0xe9, 0x11, 0xcb, 0xa5, 0x05, 0x6b, 0xc9, 0x1f, 0x26,
    0xbd, 0x33, 0x5d, 0x59, 0xc6, 0x30, 0x14, 0x22, 0x79, 0x59, 0xb0, 0xb8,
};
static const unsigned char kat480_prt_addin0[] = {
    0xe4, 0xeb, 0xa0, 0x0e, 0x64, 0xd2, 0x78, 0x54, 0xc0, 0x6d, 0xcd, 0x0a,
    0xa2, 0xcd, 0x45, 0xb4, 0x7b, 0x7f, 0xa2, 0xcd, 0x77, 0xa7, 0xa1, 0x46,
};
static const unsigned char kat480_prt_addin1[] = {
    0x11, 0x11, 0x6d, 0x24, 0x80, 0x29, 0x8a, 0x90, 0x91, 0x43, 0x7a, 0x49,
    0xc6, 0x0b, 0x9d, 0x0a, 0x9d, 0x4c, 0xda, 0x4f, 0xa8, 0x5f, 0x09, 0x5e,
};
static const unsigned char kat480_prt_retbytes[] = {
    0x39, 0x70, 0x77, 0x5a, 0xeb, 0x3d, 0xe4, 0x1b, 0x7b, 0xf1, 0x37, 0xd5,
    0xf5, 0xff, 0xb9, 0xb7, 0x36, 0x76, 0x01, 0x21, 0x3b, 0xb5, 0xf1, 0xfe,
    0x07, 0xca, 0x60, 0x9b, 0x6d, 0x3b, 0xed, 0x56, 0xd2, 0x67, 0xa9, 0x14,
    0xc2, 0xd9, 0x79, 0x3e, 0xf9, 0x90, 0x7b, 0xc1, 0x7f, 0x1d, 0x0c, 0x5d,
    0x01, 0x1d, 0x9b, 0xe4, 0x56, 0x5e, 0x01, 0xdd, 0xc2, 0x7e, 0xb2, 0xe9,
    0xa0, 0x3e, 0x37, 0xf0, 0x82, 0x4a, 0x5d, 0x32, 0xd3, 0x4b, 0x94, 0xeb,
    0x64, 0xf7, 0x11, 0x4a, 0x5c, 0xce, 0x46, 0x3d, 0xad, 0x0c, 0xdf, 0x5e,
    0x89, 0xe5, 0xf3, 0x37, 0xec, 0xea, 0x10, 0xe3, 0x39, 0x3e, 0x2b, 0x6c,
    0xa6, 0x1e, 0x5d, 0x33, 0xc2, 0x94, 0xbf, 0x90, 0x8b, 0xf7, 0x98, 0x7c,
    0xc9, 0xe8, 0x08, 0xa0,
};
static const struct drbg_kat_pr_true kat480_prt_t = {
    14, kat480_prt_entropyin, kat480_prt_nonce, kat480_prt_persstr,
    kat480_prt_entropyinpr0, kat480_prt_addin0, kat480_prt_entropyinpr1,
    kat480_prt_addin1, kat480_prt_retbytes
};
static const struct drbg_kat kat480_prt = {
    PR_TRUE, NA, NID_sha224, 24, 12, 24, 24, 112, &kat480_prt_t
};

static const unsigned char kat720_prt_entropyin[] = {
    0x06, 0x6d, 0xc8, 0xce, 0x75, 0xb2, 0x89, 0x66, 0xa6, 0x85, 0x16, 0x3f,
    0xe2, 0xa4, 0xd4, 0x27, 0xfb, 0xdb, 0x61, 0x66, 0x50, 0x61, 0x6b, 0xa2,
    0x82, 0xfc, 0x33, 0x2b, 0x4e, 0x6f, 0x12, 0x20,
};
static const unsigned char kat720_prt_nonce[] = {
    0x55, 0x9f, 0x7c, 0x64, 0x89, 0x70, 0x83, 0xec, 0x2d, 0x73, 0x70, 0xd9,
    0xf0, 0xe5, 0x07, 0x1f,
};
static const unsigned char kat720_prt_persstr[] = {
    0x88, 0x6f, 0x54, 0x9a, 0xad, 0x1a, 0xc6, 0x3d, 0x18, 0xcb, 0xcc, 0x66,
    0x85, 0xda, 0xa2, 0xc2, 0xf7, 0x9e, 0xb0, 0x89, 0x4c, 0xb4, 0xae, 0xf1,
    0xac, 0x54, 0x4f, 0xce, 0x57, 0xf1, 0x5e, 0x11,
};
static const unsigned char kat720_prt_entropyinpr0[] = {
    0xff, 0x80, 0xb7, 0xd2, 0x6a, 0x05, 0xbc, 0x8a, 0x7a, 0xbe, 0x53, 0x28,
    0x6b, 0x0e, 0xeb, 0x73, 0x3b, 0x71, 0x5a, 0x20, 0x5b, 0xfa, 0x4f, 0xf6,
    0x37, 0x03, 0xde, 0xad, 0xb6, 0xea, 0x0e, 0xf4,
};
static const unsigned char kat720_prt_entropyinpr1[] = {
    0xc7, 0x38, 0x32, 0x53, 0x46, 0x81, 0xed, 0xe3, 0x7e, 0x03, 0x84, 0x6d,
    0x3c, 0x84, 0x17, 0x67, 0x29, 0x7d, 0x24, 0x6c, 0x68, 0x92, 0x41, 0xd2,
    0xe7, 0x75, 0xbe, 0x7e, 0xc9, 0x96, 0x29, 0x3d,
};
static const unsigned char kat720_prt_addin0[] = {
    0xb7, 0x21, 0x5f, 0x14, 0xac, 0x7b, 0xaf, 0xd0, 0xa9, 0x17, 0x72, 0xba,
    0x22, 0xf7, 0x19, 0xaf, 0xbd, 0x20, 0xb3, 0x11, 0x63, 0x6c, 0x2b, 0x1e,
    0x83, 0xe4, 0xa8, 0x23, 0x35, 0x3f, 0xc6, 0xea,
};
static const unsigned char kat720_prt_addin1[] = {
    0xce, 0xd3, 0x1f, 0x7e, 0x0d, 0xae, 0x5b, 0xb5, 0xc0, 0x43, 0xe2, 0x46,
    0xb2, 0x94, 0x73, 0xe2, 0xfd, 0x39, 0x51, 0x2e, 0xad, 0x45, 0x69, 0xee,
    0xe3, 0xe3, 0x80, 0x33, 0x14, 0xab, 0xa7, 0xa3,
};
static const unsigned char kat720_prt_retbytes[] = {
    0x60, 0xc2, 0x34, 0xcf, 0xaf, 0xb4, 0x68, 0x03, 0x3b, 0xf1, 0x95, 0xe5,
    0x78, 0xce, 0x26, 0x6e, 0x14, 0x65, 0x32, 0x6a, 0x96, 0xa9, 0xe0, 0x3f,
    0x8b, 0x89, 0x36, 0x70, 0xef, 0x62, 0x75, 0x4d, 0x5e, 0x80, 0xd5, 0x53,
    0xa1, 0xf8, 0x49, 0x50, 0x20, 0x8b, 0x93, 0x43, 0x07, 0x9f, 0x2e, 0xf8,
    0x56, 0xe9, 0xc5, 0x70, 0x61, 0x85, 0x97, 0xb5, 0xdc, 0x82, 0xa2, 0xda,
    0xea, 0xa3, 0xfd, 0x9b, 0x2f, 0xd2, 0xa0, 0xd7, 0x1b, 0xc6, 0x29, 0x35,
    0xcc, 0xb8, 0x3d, 0xa0, 0x67, 0x98, 0x05, 0xa0, 0xe3, 0x1e, 0xfe, 0xe4,
    0xf0, 0xe5, 0x13, 0xb0, 0x83, 0x17, 0xfa, 0xca, 0x93, 0x5e, 0x38, 0x29,
    0x48, 0xd2, 0x72, 0xdb, 0x76, 0x3e, 0x6d, 0xf3, 0x25, 0x10, 0xff, 0x1b,
    0x99, 0xff, 0xf8, 0xc6, 0x0e, 0xb0, 0xdd, 0x29, 0x2e, 0xbc, 0xbb, 0xc8,
    0x0a, 0x01, 0x6e, 0xd3, 0xb0, 0x0e, 0x4e, 0xab,
};
static const struct drbg_kat_pr_true kat720_prt_t = {
    14, kat720_prt_entropyin, kat720_prt_nonce, kat720_prt_persstr,
    kat720_prt_entropyinpr0, kat720_prt_addin0, kat720_prt_entropyinpr1,
    kat720_prt_addin1, kat720_prt_retbytes
};
static const struct drbg_kat kat720_prt = {
    PR_TRUE, NA, NID_sha256, 32, 16, 32, 32, 128, &kat720_prt_t
};


static const unsigned char kat960_prt_entropyin[] = {
    0x37, 0xea, 0x89, 0xa5, 0xe1, 0x21, 0xd1, 0x26, 0x4b, 0xf5, 0xce, 0x89,
    0x23, 0x3b, 0x09, 0x02, 0x0d, 0xda, 0xeb, 0xf9, 0xb2, 0x7e, 0xc3, 0xec,
    0x8a, 0xfd, 0xd9, 0x72, 0x14, 0x9a, 0x48, 0x72,
};
static const unsigned char kat960_prt_nonce[] = {
    0x17, 0x22, 0x32, 0x8f, 0xa7, 0xe1, 0x34, 0x56, 0x5d, 0xdf, 0x01, 0x45,
    0x02, 0x3d, 0x6a, 0xaa,
};
static const unsigned char kat960_prt_persstr[] = {
    0xe2, 0x19, 0x12, 0xd5, 0x2c, 0xfb, 0x63, 0x14, 0x2b, 0xaa, 0xf0, 0x04,
    0x6a, 0x7f, 0x85, 0xf1, 0x9e, 0x61, 0xfa, 0xd7, 0x9d, 0x65, 0x16, 0xcd,
    0xcc, 0xf6, 0xb0, 0x59, 0x9c, 0x9b, 0xf4, 0x9f,
};
static const unsigned char kat960_prt_entropyinpr0[] = {
    0x91, 0xe3, 0x42, 0xca, 0x93, 0x54, 0x42, 0x6c, 0x25, 0xe8, 0xe6, 0x56,
    0x2d, 0xfd, 0x3c, 0xbf, 0x52, 0xa9, 0xc4, 0x02, 0x8c, 0xa6, 0x1a, 0xae,
    0x26, 0x3f, 0x1d, 0x09, 0x15, 0xe9, 0x16, 0xfa,
};
static const unsigned char kat960_prt_entropyinpr1[] = {
    0x42, 0x85, 0x0d, 0x17, 0x28, 0xf7, 0x14, 0xd9, 0x44, 0xbc, 0x67, 0x85,
    0x0f, 0xa6, 0xdb, 0x4a, 0x7c, 0xc2, 0xe0, 0xf7, 0x3f, 0x5c, 0xa0, 0x44,
    0xd6, 0x34, 0xd7, 0xc7, 0x15, 0x73, 0x71, 0xd2,
};
static const unsigned char kat960_prt_addin0[] = {
    0x0d, 0xc5, 0x4c, 0xe0, 0x8f, 0xf3, 0x47, 0x11, 0xab, 0x0b, 0x78, 0xfb,
    0x70, 0xcd, 0xb4, 0x11, 0x8a, 0x4f, 0x3f, 0x58, 0xa8, 0xf7, 0xdd, 0x83,
    0xdc, 0x4f, 0xc7, 0x74, 0x4e, 0xe1, 0x81, 0xb6,
};
static const unsigned char kat960_prt_addin1[] = {
    0xaf, 0xc8, 0xc5, 0x30, 0xd9, 0x10, 0xa2, 0x25, 0x73, 0xe0, 0xee, 0xc2,
    0xc1, 0xa6, 0x75, 0x9a, 0x94, 0xac, 0x38, 0xd0, 0x7b, 0x52, 0xd4, 0x69,
    0x38, 0xb5, 0x9c, 0xb2, 0x40, 0xeb, 0xb0, 0x03,
};
static const unsigned char kat960_prt_retbytes[] = {
    0x1f, 0x8c, 0x8a, 0xd0, 0xa0, 0xad, 0x62, 0x55, 0x1d, 0xca, 0x84, 0x75,
    0x31, 0xf0, 0x9b, 0x86, 0x5d, 0xbe, 0x97, 0xb2, 0x7a, 0xe1, 0x5d, 0x94,
    0xef, 0x1c, 0xc6, 0xa9, 0x28, 0x07, 0x76, 0x66, 0x20, 0x4c, 0x1e, 0x90,
    0x0a, 0x46, 0x88, 0xe9, 0xb3, 0x13, 0x96, 0x93, 0x18, 0x5b, 0xa6, 0xf8,
    0x63, 0x66, 0x5d, 0x80, 0x74, 0x1f, 0xa6, 0x25, 0x14, 0x40, 0x96, 0x3b,
    0x89, 0xcc, 0x35, 0x47, 0xb3, 0x78, 0x1c, 0x71, 0xa5, 0xec, 0xcf, 0x48,
    0x60, 0xc2, 0x89, 0x88, 0x2c, 0x42, 0xd2, 0x23, 0xdd, 0xa3, 0xb8, 0xc3,
    0x0d, 0x63, 0x88, 0xf8, 0xd2, 0xab, 0x9e, 0x19, 0x47, 0xe0, 0x33, 0x0d,
    0xe4, 0x28, 0xce, 0xdd, 0x06, 0xde, 0x65, 0x06, 0xaa, 0xd1, 0x97, 0xa1,
    0x07, 0x00, 0xc6, 0xeb, 0x4d, 0x34, 0x18, 0x5a, 0x27, 0x75, 0x1c, 0x83,
    0xb4, 0x64, 0x79, 0xc6, 0x5b, 0x4f, 0x34, 0xe7, 0xf0, 0x90, 0x78, 0x97,
    0xc0, 0xad, 0xcf, 0xe0, 0xaf, 0xd6, 0x7c, 0x89, 0x4a, 0xa1, 0x49, 0x1a,
    0xe4, 0xda, 0x29, 0xac, 0x1d, 0x47, 0xe9, 0x46, 0x08, 0x94, 0x4a, 0x67,
    0xc1, 0xa4, 0xe4, 0x75, 0x4e, 0xa1, 0x69, 0x2b, 0x79, 0xd6, 0x78, 0x70,
    0xef, 0x78, 0xb6, 0xda, 0xae, 0xe4, 0xcd, 0x46, 0x00, 0xbd, 0x1e, 0xea,
    0xc1, 0xd6, 0x7f, 0x81, 0xee, 0xd7, 0x3e, 0x8a, 0x54, 0x8c, 0x20, 0x67,
};
static const struct drbg_kat_pr_true kat960_prt_t = {
    14, kat960_prt_entropyin, kat960_prt_nonce, kat960_prt_persstr,
    kat960_prt_entropyinpr0, kat960_prt_addin0, kat960_prt_entropyinpr1,
    kat960_prt_addin1, kat960_prt_retbytes
};
static const struct drbg_kat kat960_prt = {
    PR_TRUE, NA, NID_sha384, 32, 16, 32, 32, 192, &kat960_prt_t
};

static const unsigned char kat1200_prt_entropyin[] = {
    0x19, 0xff, 0x49, 0x7e, 0x90, 0x6d, 0x08, 0x11, 0xa0, 0xb7, 0xf7, 0x75,
    0x86, 0x18, 0xaf, 0xab, 0x44, 0xfc, 0xe8, 0xed, 0x35, 0x38, 0xab, 0xfd,
    0x10, 0x1c, 0x58, 0x6a, 0x9c, 0xea, 0xf6, 0x2d,
};
static const unsigned char kat1200_prt_nonce[] = {
    0x49, 0x80, 0xcb, 0x99, 0x83, 0x87, 0x0a, 0x35, 0xd8, 0x00, 0xff, 0xf9,
    0x32, 0x78, 0x40, 0xc5,
};
static const unsigned char kat1200_prt_persstr[] = {
    0x2e, 0x34, 0xbe, 0xa4, 0xc1, 0x28, 0x43, 0x4a, 0xa2, 0x85, 0x9c, 0xc3,
    0x39, 0x13, 0x71, 0x6b, 0xc8, 0x05, 0xa1, 0x83, 0xbb, 0x17, 0x62, 0xfc,
    0x3a, 0xb5, 0x44, 0x89, 0xd9, 0xbd, 0x1c, 0x56,
};
static const unsigned char kat1200_prt_entropyinpr0[] = {
    0xbe, 0x45, 0xca, 0x98, 0x78, 0x24, 0x7b, 0x00, 0x8e, 0x5d, 0xe8, 0xc5,
    0x88, 0xb6, 0xcb, 0x84, 0x04, 0x8c, 0x0d, 0x90, 0x35, 0x7d, 0xbd, 0xfc,
    0x98, 0xe9, 0xe2, 0x87, 0x91, 0x2f, 0xdb, 0x33,
};
static const unsigned char kat1200_prt_entropyinpr1[] = {
    0xb8, 0x71, 0x61, 0x1f, 0x8f, 0xcb, 0x8c, 0x86, 0x0a, 0x72, 0xc4, 0xfd,
    0x40, 0x6d, 0x49, 0x39, 0x33, 0x5a, 0x03, 0x1e, 0x0d, 0xe9, 0xf2, 0xd4,
    0x36, 0xd4, 0x73, 0x6b, 0x6b, 0x06, 0x0c, 0x2d,
};
static const unsigned char kat1200_prt_addin0[] = {
    0x47, 0xdd, 0x5c, 0x72, 0xea, 0x44, 0xdb, 0xf1, 0xd2, 0x35, 0x04, 0x0b,
    0x18, 0xa8, 0x21, 0xc2, 0x0f, 0xb4, 0x0b, 0x88, 0xc0, 0x39, 0x1a, 0x5c,
    0x16, 0x3f, 0x9c, 0x90, 0x2f, 0x6c, 0x6e, 0xb7,
};
static const unsigned char kat1200_prt_addin1[] = {
    0xcf, 0x20, 0x40, 0xe9, 0x04, 0x6a, 0x69, 0xdd, 0x96, 0x38, 0xde, 0x94,
    0x1f, 0x00, 0x90, 0xb7, 0x53, 0x5c, 0x51, 0xcf, 0xa9, 0xf1, 0xc7, 0xbb,
    0x2a, 0x56, 0xa3, 0x32, 0x32, 0x69, 0x43, 0xd3,
};
static const unsigned char kat1200_prt_retbytes[] = {
    0x2d, 0x99, 0x0f, 0x0d, 0xe4, 0x3d, 0x3a, 0x4b, 0x29, 0x30, 0x54, 0x2c,
    0x27, 0xad, 0x27, 0x45, 0x8e, 0x88, 0x65, 0xca, 0x6b, 0x8f, 0x27, 0xfd,
    0x7a, 0x96, 0x9c, 0xf4, 0xe2, 0xa0, 0x32, 0x3e, 0x38, 0xfe, 0x6f, 0x50,
    0x5a, 0x2d, 0xba, 0x48, 0x8e, 0xa6, 0xb0, 0x43, 0x65, 0x20, 0x9c, 0x6d,
    0xb7, 0x86, 0xcb, 0xbf, 0x0a, 0x7c, 0x73, 0xb4, 0xfd, 0x56, 0xd2, 0x49,
    0x87, 0x71, 0x9d, 0xb0, 0xfd, 0xba, 0x1a, 0x3f, 0x07, 0x14, 0x95, 0x21,
    0xdc, 0xf5, 0xb7, 0x75, 0x9c, 0x61, 0x0d, 0xa2, 0x2d, 0x15, 0x10, 0x57,
    0xac, 0xef, 0xe7, 0x0d, 0xf1, 0xcc, 0xae, 0xb6, 0x7a, 0x97, 0x51, 0x59,
    0xb8, 0x99, 0x6a, 0xca, 0x93, 0xd7, 0xa4, 0x80, 0x96, 0x92, 0x6d, 0xb4,
    0x38, 0x1b, 0xbc, 0xe4, 0x81, 0x27, 0x7d, 0x7a, 0xb2, 0x7c, 0xbc, 0x03,
    0x88, 0xf0, 0xb7, 0xce, 0xdb, 0xbf, 0xb8, 0x42, 0x1c, 0xb1, 0xdc, 0x5f,
    0x2a, 0x9c, 0x67, 0x7f, 0x62, 0xac, 0xf9, 0x6a, 0xb2, 0x5e, 0x7e, 0x40,
    0x6c, 0xe8, 0x2f, 0x5b, 0x96, 0xbc, 0xb4, 0x71, 0xaf, 0xbd, 0xf4, 0xb3,
    0xf5, 0xa6, 0xfb, 0xcb, 0x8d, 0xa4, 0x5d, 0x22, 0x58, 0xe3, 0x50, 0xe7,
    0x7d, 0x46, 0x33, 0xb0, 0xc1, 0xda, 0x69, 0x16, 0x62, 0xdd, 0x86, 0x99,
    0x09, 0xdc, 0xfd, 0x7c, 0x8e, 0xd0, 0xf5, 0x4b, 0xa7, 0xaf, 0x0f, 0x9c,
    0x03, 0x8e, 0xb3, 0x2d, 0x32, 0xb7, 0x05, 0xe5, 0x1b, 0x35, 0xbb, 0x3c,
    0x2e, 0xef, 0xf0, 0x10, 0xbb, 0x47, 0xee, 0x32, 0x6c, 0x23, 0x18, 0xb5,
    0xbc, 0xda, 0x96, 0x3c, 0x2d, 0xad, 0x41, 0x9c, 0x59, 0x23, 0xe3, 0x68,
    0xd9, 0xb2, 0x8f, 0x25, 0xb0, 0x48, 0xa8, 0x7b, 0xdb, 0xa0, 0xa9, 0x0d,
    0x98, 0xc2, 0x4c, 0x81, 0xb6, 0xdb, 0xde, 0x0f, 0x58, 0x05, 0x4a, 0x41,
    0xa8, 0x29, 0x3a, 0x65,
};
static const struct drbg_kat_pr_true kat1200_prt_t = {
    14, kat1200_prt_entropyin, kat1200_prt_nonce, kat1200_prt_persstr,
    kat1200_prt_entropyinpr0, kat1200_prt_addin0, kat1200_prt_entropyinpr1,
    kat1200_prt_addin1, kat1200_prt_retbytes
};
static const struct drbg_kat kat1200_prt = {
    PR_TRUE, NA, NID_sha512, 32, 16, 32, 32, 256, &kat1200_prt_t
};

static const unsigned char kat1440_prt_entropyin[] = {
    0xb7, 0xb8, 0x4c, 0x20, 0x4a, 0x15, 0x2f, 0x6a, 0x01, 0x4d, 0x10, 0xd8,
    0x1e, 0x70, 0x86, 0x9a, 0xd2, 0xeb, 0x4f, 0x7a, 0x7e, 0x7e, 0xdc, 0x18,
};
static const unsigned char kat1440_prt_nonce[] = {
    0x6b, 0x44, 0xc0, 0x33, 0xfb, 0x82, 0xd3, 0x60, 0x59, 0x8d, 0xee, 0x68,
};
static const unsigned char kat1440_prt_persstr[] = {
    0x75, 0x32, 0xcb, 0x6b, 0xfb, 0xbf, 0xbe, 0xbd, 0x4d, 0x4c, 0x7a, 0xf9,
    0xad, 0x89, 0x0a, 0x50, 0xcb, 0xe8, 0x20, 0x2c, 0x38, 0x80, 0xf8, 0x9c,
};
static const unsigned char kat1440_prt_entropyinpr0[] = {
    0xc0, 0x77, 0xb4, 0xc2, 0xce, 0xff, 0xa7, 0x88, 0x7a, 0xf9, 0x09, 0xa6,
    0xf1, 0xea, 0x46, 0xed, 0xc7, 0x3e, 0x02, 0x05, 0xdf, 0x68, 0x53, 0x7c,
};
static const unsigned char kat1440_prt_entropyinpr1[] = {
    0xaa, 0x9b, 0x3d, 0xba, 0x73, 0x76, 0xb0, 0xa2, 0x1d, 0x34, 0xee, 0x6a,
    0xc8, 0x93, 0x9a, 0x62, 0x5d, 0xbf, 0xec, 0x17, 0x2a, 0x10, 0x8c, 0x4c,
};
static const unsigned char kat1440_prt_addin0[] = {
    0x5f, 0xc8, 0x2b, 0x7c, 0x36, 0x8e, 0x0e, 0xe3, 0x45, 0xb6, 0x03, 0xc3,
    0x6a, 0xb1, 0xbb, 0x86, 0x3a, 0xd1, 0x44, 0x6e, 0x61, 0xfe, 0x6a, 0xf2,
};
static const unsigned char kat1440_prt_addin1[] = {
    0xa0, 0x0a, 0xa1, 0x2c, 0x4a, 0x26, 0x03, 0x0b, 0x79, 0x89, 0x7e, 0x04,
    0xd0, 0x17, 0x1b, 0xbc, 0xe1, 0xcd, 0x72, 0x57, 0xe0, 0xcc, 0xe3, 0x79,
};
static const unsigned char kat1440_prt_retbytes[] = {
    0x54, 0xfb, 0x77, 0x8f, 0xcf, 0xc5, 0x54, 0x9e, 0x19, 0x02, 0x71, 0xdc,
    0x12, 0x38, 0x9f, 0x42, 0xea, 0x81, 0x28, 0xdf, 0x55, 0xe6, 0x19, 0x3e,
    0x03, 0x07, 0x38, 0x88, 0xb4, 0xbe, 0x31, 0xe2, 0xd7, 0xa7, 0x88, 0x45,
    0xc4, 0x73, 0x62, 0xc4, 0xe9, 0x6b, 0x41, 0xfc, 0xe5, 0x03, 0xfb, 0x97,
    0x0f, 0x91, 0x76, 0xbd, 0xb9, 0xb5, 0xd6, 0x64, 0xc3, 0x86, 0x89, 0x8a,
    0x0e, 0x44, 0xff, 0xe1, 0x2f, 0x94, 0x80, 0x69, 0x9b, 0x7d, 0x56, 0x6d,
    0x69, 0x7a, 0x4f, 0x52, 0x02, 0x68, 0xf6, 0x2e, 0x46, 0x03, 0x59, 0xa3,
    0x9d, 0x09, 0x1f, 0x4c, 0x37, 0x2a, 0xd3, 0x3e, 0xf0, 0xee, 0xf5, 0x86,
    0x22, 0xf4, 0x88, 0xc9, 0x34, 0x8a, 0xb5, 0xfd, 0x69, 0x3d, 0x4e, 0xde,
    0xce, 0x79, 0x4b, 0x12,
};
static const struct drbg_kat_pr_true kat1440_prt_t = {
    14, kat1440_prt_entropyin, kat1440_prt_nonce, kat1440_prt_persstr,
    kat1440_prt_entropyinpr0, kat1440_prt_addin0, kat1440_prt_entropyinpr1,
    kat1440_prt_addin1, kat1440_prt_retbytes
};
static const struct drbg_kat kat1440_prt = {
    PR_TRUE, NA, NID_sha512_224, 24, 12, 24, 24, 112, &kat1440_prt_t
};

static const unsigned char kat1680_prt_entropyin[] = {
    0x29, 0xe3, 0xfa, 0xcd, 0x8f, 0x0d, 0xb5, 0x25, 0xdd, 0x97, 0x17, 0x9f,
    0xba, 0x20, 0x04, 0x78, 0x2c, 0xc7, 0x45, 0xc8, 0xdc, 0xc8, 0x60, 0xc3,
    0x45, 0x8d, 0x7b, 0xf7, 0x70, 0xe1, 0x80, 0x28,
};
static const unsigned char kat1680_prt_nonce[] = {
    0xa9, 0x6e, 0x20, 0x37, 0x82, 0xea, 0x56, 0xf0, 0xb9, 0xdb, 0x55, 0x88,
    0xb3, 0xea, 0x70, 0x2b,
};
static const unsigned char kat1680_prt_persstr[] = {
    0xe6, 0xb5, 0xc4, 0x0e, 0x32, 0x0a, 0x97, 0x07, 0x18, 0x5a, 0x05, 0xee,
    0x91, 0xae, 0x73, 0x85, 0xe7, 0xca, 0x2e, 0x81, 0xf7, 0xbf, 0x65, 0xee,
    0xd6, 0xf9, 0x31, 0x3a, 0x95, 0x9e, 0x5d, 0xd7,
};
static const unsigned char kat1680_prt_entropyinpr0[] = {
    0xff, 0x68, 0xae, 0x1f, 0xfa, 0xac, 0xe1, 0xf0, 0xe8, 0x29, 0x98, 0x31,
    0x84, 0x6e, 0x1e, 0x83, 0x79, 0xf9, 0xf9, 0x69, 0x6a, 0xa3, 0x2c, 0x76,
    0x2c, 0x56, 0xaa, 0xd2, 0xdb, 0x58, 0x60, 0x25,
};
static const unsigned char kat1680_prt_entropyinpr1[] = {
    0xf6, 0xf2, 0x29, 0x6c, 0x12, 0x20, 0xee, 0xc3, 0xb6, 0x81, 0xff, 0xe7,
    0xdb, 0x9b, 0xf8, 0x83, 0xbf, 0x2a, 0x87, 0x13, 0x23, 0x87, 0x05, 0xb9,
    0x48, 0x14, 0x7f, 0xd7, 0xfc, 0x5e, 0x52, 0x06,
};
static const unsigned char kat1680_prt_addin0[] = {
    0xe7, 0x8d, 0xcf, 0x5e, 0x29, 0xa8, 0x39, 0x3e, 0x83, 0xe8, 0x6d, 0xae,
    0xf0, 0x1d, 0x2a, 0xd7, 0x4b, 0x22, 0xee, 0xe0, 0x93, 0xd9, 0x46, 0x8b,
    0xfa, 0xa0, 0xe8, 0xa1, 0xb8, 0x36, 0x44, 0x95,
};
static const unsigned char kat1680_prt_addin1[] = {
    0x9d, 0x06, 0x7e, 0x72, 0xf6, 0x31, 0x0a, 0x0c, 0xb7, 0xe9, 0xeb, 0x2e,
    0xf7, 0xb0, 0x2e, 0x1b, 0x8b, 0x9b, 0xf8, 0x0c, 0x51, 0xa7, 0x93, 0x3a,
    0x5a, 0xa6, 0x9f, 0x13, 0x52, 0x9f, 0xf8, 0x49,
};
static const unsigned char kat1680_prt_retbytes[] = {
    0x6a, 0xf6, 0x89, 0xce, 0xc6, 0x2a, 0x63, 0x34, 0x92, 0xf6, 0xe2, 0x4b,
    0x75, 0x4d, 0x38, 0xdd, 0x6a, 0xb0, 0xb5, 0x56, 0xe9, 0x18, 0x02, 0xd7,
    0x2f, 0x14, 0xdc, 0x8c, 0x0e, 0x9f, 0xf5, 0x0d, 0xf7, 0x28, 0xb4, 0xe7,
    0x99, 0xa0, 0x36, 0xc3, 0xc4, 0x86, 0x30, 0xbe, 0x11, 0x6c, 0x8d, 0x6f,
    0x19, 0x1f, 0x30, 0x82, 0x1b, 0x4b, 0x8c, 0x74, 0xd7, 0x16, 0xe8, 0xa8,
    0x24, 0xd6, 0xc4, 0x1b, 0x81, 0xcc, 0xc2, 0x0c, 0xd0, 0x21, 0x57, 0x12,
    0xaa, 0xab, 0x7e, 0x1b, 0x92, 0x66, 0xc1, 0x41, 0xac, 0x66, 0xa5, 0xbb,
    0xb2, 0xe5, 0x00, 0x45, 0x5a, 0xbc, 0xbe, 0xa3, 0x91, 0x8f, 0x9f, 0xcb,
    0x4c, 0x22, 0x0b, 0xb5, 0xa9, 0xbf, 0x78, 0x15, 0xcb, 0xd2, 0x43, 0x5b,
    0x67, 0xcc, 0xf6, 0x3f, 0x89, 0x89, 0xcb, 0xc8, 0x63, 0x6e, 0xe8, 0x14,
    0x09, 0xdb, 0xc2, 0x3a, 0xf9, 0x1f, 0x16, 0xd2,
};
static const struct drbg_kat_pr_true kat1680_prt_t = {
    14, kat1680_prt_entropyin, kat1680_prt_nonce, kat1680_prt_persstr,
    kat1680_prt_entropyinpr0, kat1680_prt_addin0, kat1680_prt_entropyinpr1,
    kat1680_prt_addin1, kat1680_prt_retbytes
};
static const struct drbg_kat kat1680_prt = {
    PR_TRUE, NA, NID_sha512_256, 32, 16, 32, 32, 128, &kat1680_prt_t
};


/* -------------------------------------------------------------------------- */

static const unsigned char kat1_prf_entropyin[] = {
    0x16, 0x10, 0xb8, 0x28, 0xcc, 0xd2, 0x7d, 0xe0, 0x8c, 0xee, 0xa0, 0x32,
    0xa2, 0x0e, 0x92, 0x08,
};
static const unsigned char kat1_prf_nonce[] = {
    0x49, 0x2c, 0xf1, 0x70, 0x92, 0x42, 0xf6, 0xb5,
};
static const unsigned char kat1_prf_persstr[] = {0};
static const unsigned char kat1_prf_entropyin_reseed[] = {
    0x72, 0xd2, 0x8c, 0x90, 0x8e, 0xda, 0xf9, 0xa4, 0xd1, 0xe5, 0x26, 0xd8,
    0xf2, 0xde, 0xd5, 0x44,
};
static const unsigned char kat1_prf_addin_reseed[] = {0};
static const unsigned char kat1_prf_addin0[] = {0};
static const unsigned char kat1_prf_addin1[] = {0};
static const unsigned char kat1_prf_retbytes[] = {
    0x56, 0xf3, 0x3d, 0x4f, 0xdb, 0xb9, 0xa5, 0xb6, 0x4d, 0x26, 0x23, 0x44,
    0x97, 0xe9, 0xdc, 0xb8, 0x77, 0x98, 0xc6, 0x8d, 0x08, 0xf7, 0xc4, 0x11,
    0x99, 0xd4, 0xbd, 0xdf, 0x97, 0xeb, 0xbf, 0x6c, 0xb5, 0x55, 0x0e, 0x5d,
    0x14, 0x9f, 0xf4, 0xd5, 0xbd, 0x0f, 0x05, 0xf2, 0x5a, 0x69, 0x88, 0xc1,
    0x74, 0x36, 0x39, 0x62, 0x27, 0x18, 0x4a, 0xf8, 0x4a, 0x56, 0x43, 0x35,
    0x65, 0x8e, 0x2f, 0x85, 0x72, 0xbe, 0xa3, 0x33, 0xee, 0xe2, 0xab, 0xff,
    0x22, 0xff, 0xa6, 0xde, 0x3e, 0x22, 0xac, 0xa2,
};
static const struct drbg_kat_pr_false kat1_prf_t = {
    0, kat1_prf_entropyin, kat1_prf_nonce, kat1_prf_persstr,
    kat1_prf_entropyin_reseed, kat1_prf_addin_reseed,
    kat1_prf_addin0, kat1_prf_addin1, kat1_prf_retbytes
};
static const struct drbg_kat kat1_prf = {
    PR_FALSE, NA, NID_sha1, 16, 8, 0, 0, 80, &kat1_prf_t
};

static const unsigned char kat1680_prf_entropyin[] = {
    0x5c, 0x15, 0x82, 0xc1, 0x31, 0x34, 0xbd, 0x98, 0x4b, 0xa6, 0xc2, 0x7d,
    0x32, 0xbd, 0xf3, 0xfa, 0x18, 0x47, 0xc6, 0xf9, 0xc3, 0x7f, 0x9a, 0x5c,
    0xb3, 0x15, 0xca, 0xba, 0xc9, 0xc8, 0xf2, 0xa6,
};
static const unsigned char kat1680_prf_nonce[] = {
    0x5e, 0x3e, 0xba, 0xe1, 0xc1, 0xb1, 0x15, 0x07, 0xe0, 0xce, 0x8c, 0xe6,
    0x80, 0x65, 0x75, 0x18,
};
static const unsigned char kat1680_prf_persstr[] = {
    0x26, 0x1f, 0x0f, 0xa2, 0xfc, 0x41, 0xd2, 0x03, 0x63, 0xb0, 0x97, 0x5c,
    0x58, 0xbd, 0x79, 0x54, 0x8c, 0x13, 0x3a, 0x66, 0xe8, 0xed, 0xd7, 0xc4,
    0x40, 0xb7, 0xf6, 0x9d, 0x37, 0xb0, 0x32, 0x32,
};
static const unsigned char kat1680_prf_entropyin_reseed[] = {
    0xda, 0x39, 0x0d, 0xbb, 0x19, 0x77, 0xa8, 0x98, 0x30, 0xcf, 0xd0, 0x28,
    0x27, 0xec, 0xf2, 0x49, 0xaf, 0x17, 0xba, 0xf6, 0x81, 0x29, 0x61, 0xf7,
    0x15, 0x79, 0xb0, 0xbb, 0x07, 0x79, 0x53, 0x7c,
};
static const unsigned char kat1680_prf_addin_reseed[] = {
    0xf3, 0xe9, 0x21, 0x0b, 0x23, 0x57, 0x96, 0x85, 0x8e, 0x81, 0x88, 0x20,
    0x2c, 0x66, 0xe7, 0xe0, 0x15, 0xc6, 0x97, 0x6d, 0x4f, 0x68, 0x72, 0xb0,
    0xc5, 0x6b, 0x80, 0xf3, 0x4b, 0x1e, 0xc4, 0x4a,
};
static const unsigned char kat1680_prf_addin0[] = {
    0x9a, 0xb2, 0x99, 0x96, 0x3b, 0xc9, 0x6b, 0x27, 0xd5, 0xe8, 0x06, 0xf1,
    0x05, 0xb9, 0x30, 0xf4, 0x9d, 0x4a, 0x27, 0x0e, 0xc9, 0xb9, 0x8d, 0x13,
    0xff, 0x43, 0x8e, 0xa9, 0xc5, 0x8f, 0x5b, 0x3d,
};
static const unsigned char kat1680_prf_addin1[] = {
    0x9e, 0x08, 0xae, 0xcc, 0xed, 0xd8, 0xae, 0x05, 0xea, 0xfb, 0xbf, 0xaf,
    0xd8, 0x79, 0x0a, 0x3e, 0x39, 0x81, 0xd8, 0x5c, 0x74, 0x3f, 0x5e, 0x6d,
    0xa1, 0x8a, 0xc4, 0xf0, 0x41, 0xb9, 0xe6, 0x27,
};
static const unsigned char kat1680_prf_retbytes[] = {
    0x51, 0x8c, 0xa2, 0x7a, 0xfa, 0xb1, 0xb5, 0x8f, 0xfa, 0x9e, 0x9d, 0x04,
    0xdd, 0x7b, 0xb7, 0xbf, 0xc4, 0x11, 0x9a, 0x29, 0x9e, 0x08, 0x72, 0x7d,
    0x8c, 0xf8, 0x99, 0x9d, 0xc4, 0x40, 0x04, 0x0f, 0x62, 0x14, 0x24, 0x6c,
    0x56, 0xbe, 0xfb, 0x2f, 0xbb, 0xc4, 0x78, 0x4c, 0x96, 0x79, 0x94, 0x59,
    0xcf, 0xac, 0x77, 0x88, 0x3a, 0x7b, 0x1f, 0xb2, 0x7b, 0xb8, 0xd9, 0xc9,
    0x09, 0xc7, 0xe6, 0x2f, 0x01, 0x1c, 0xbb, 0x5c, 0x53, 0x32, 0xd7, 0x80,
    0xfc, 0xc5, 0x00, 0x59, 0x3c, 0x2a, 0x70, 0x67, 0x41, 0x1f, 0x24, 0x67,
    0x72, 0xd8, 0x22, 0xb4, 0xd6, 0xda, 0xc3, 0x93, 0xc6, 0xb4, 0x82, 0x0b,
    0xcb, 0x6c, 0xdc, 0x26, 0x59, 0xbe, 0x54, 0x34, 0xcf, 0xfb, 0xd5, 0x35,
    0xd5, 0x13, 0x05, 0x75, 0x70, 0x7b, 0x94, 0x35, 0x8d, 0x15, 0xb0, 0x88,
    0xbc, 0x46, 0x37, 0xa8, 0x3a, 0x97, 0xf7, 0x8f,
};
static const struct drbg_kat_pr_false kat1680_prf_t = {
    14, kat1680_prf_entropyin, kat1680_prf_nonce, kat1680_prf_persstr,
    kat1680_prf_entropyin_reseed, kat1680_prf_addin_reseed,
    kat1680_prf_addin0, kat1680_prf_addin1, kat1680_prf_retbytes
};
static const struct drbg_kat kat1680_prf = {
    PR_FALSE, NA, NID_sha512_256, 32, 16, 32, 32, 128, &kat1680_prf_t
};

/* -------------------------------------------------------------------------- */

const struct drbg_kat *drbg_hash_test[] = {
    &kat1_nor, &kat2_nor, &kat3_nor, &kat4_nor, &kat5_nor,
    &kat6_nor, &kat7_nor, &kat8_nor, &kat9_nor, &kat10_nor,
    &kat11_nor, &kat12_nor, &kat13_nor, &kat14_nor, &kat15_nor,
    &kat16_nor, &kat17_nor, &kat18_nor, &kat19_nor, &kat20_nor,
    &kat21_nor, &kat22_nor, &kat23_nor, &kat24_nor, &kat25_nor,
    &kat26_nor, &kat27_nor, &kat28_nor, &kat29_nor, &kat30_nor,
    &kat31_nor, &kat32_nor, &kat33_nor, &kat34_nor, &kat35_nor,
    &kat36_nor, &kat37_nor, &kat38_nor, &kat39_nor, &kat40_nor,
    &kat41_nor, &kat42_nor, &kat43_nor, &kat44_nor, &kat45_nor,
    &kat46_nor, &kat47_nor, &kat48_nor, &kat49_nor, &kat50_nor,
    &kat51_nor, &kat52_nor, &kat53_nor, &kat54_nor, &kat55_nor,
    &kat56_nor, &kat57_nor, &kat58_nor, &kat59_nor, &kat60_nor,
    &kat61_nor, &kat62_nor, &kat63_nor, &kat64_nor, &kat65_nor,
    &kat66_nor, &kat67_nor, &kat68_nor, &kat69_nor, &kat70_nor,
    &kat71_nor, &kat72_nor, &kat73_nor, &kat74_nor, &kat75_nor,
    &kat76_nor, &kat77_nor, &kat78_nor, &kat79_nor, &kat80_nor,
    &kat81_nor, &kat82_nor, &kat83_nor, &kat84_nor, &kat85_nor,
    &kat86_nor, &kat87_nor, &kat88_nor, &kat89_nor, &kat90_nor,
    &kat91_nor, &kat92_nor, &kat93_nor, &kat94_nor, &kat95_nor,
    &kat96_nor, &kat97_nor, &kat98_nor, &kat99_nor, &kat100_nor,
    &kat101_nor, &kat102_nor, &kat103_nor, &kat104_nor, &kat105_nor,
    &kat106_nor, &kat107_nor, &kat108_nor, &kat109_nor, &kat110_nor,
    &kat111_nor, &kat112_nor, &kat113_nor, &kat114_nor, &kat115_nor,
    &kat116_nor, &kat117_nor, &kat118_nor, &kat119_nor, &kat120_nor,
    &kat121_nor, &kat122_nor, &kat123_nor, &kat124_nor, &kat125_nor,
    &kat126_nor, &kat127_nor, &kat128_nor, &kat129_nor, &kat130_nor,
    &kat131_nor, &kat132_nor, &kat133_nor, &kat134_nor, &kat135_nor,
    &kat136_nor, &kat137_nor, &kat138_nor, &kat139_nor, &kat140_nor,
    &kat141_nor, &kat142_nor, &kat143_nor, &kat144_nor, &kat145_nor,
    &kat146_nor, &kat147_nor, &kat148_nor, &kat149_nor, &kat150_nor,
    &kat151_nor, &kat152_nor, &kat153_nor, &kat154_nor, &kat155_nor,
    &kat156_nor, &kat157_nor, &kat158_nor, &kat159_nor, &kat160_nor,
    &kat161_nor, &kat162_nor, &kat163_nor, &kat164_nor, &kat165_nor,
    &kat166_nor, &kat167_nor, &kat168_nor, &kat169_nor, &kat170_nor,
    &kat171_nor, &kat172_nor, &kat173_nor, &kat174_nor, &kat175_nor,
    &kat176_nor, &kat177_nor, &kat178_nor, &kat179_nor, &kat180_nor,
    &kat181_nor, &kat182_nor, &kat183_nor, &kat184_nor, &kat185_nor,
    &kat186_nor, &kat187_nor, &kat188_nor, &kat189_nor, &kat190_nor,
    &kat191_nor, &kat192_nor, &kat193_nor, &kat194_nor, &kat195_nor,
    &kat196_nor, &kat197_nor, &kat198_nor, &kat199_nor, &kat200_nor,
    &kat201_nor, &kat202_nor, &kat203_nor, &kat204_nor, &kat205_nor,
    &kat206_nor, &kat207_nor, &kat208_nor, &kat209_nor, &kat210_nor,
    &kat211_nor, &kat212_nor, &kat213_nor, &kat214_nor, &kat215_nor,
    &kat216_nor, &kat217_nor, &kat218_nor, &kat219_nor, &kat220_nor,
    &kat221_nor, &kat222_nor, &kat223_nor, &kat224_nor, &kat225_nor,
    &kat226_nor, &kat227_nor, &kat228_nor, &kat229_nor, &kat230_nor,
    &kat231_nor, &kat232_nor, &kat233_nor, &kat234_nor, &kat235_nor,
    &kat236_nor, &kat237_nor, &kat238_nor, &kat239_nor, &kat240_nor,
    &kat480_nor, &kat720_nor, &kat960_nor, &kat1200_nor, &kat1440_nor,
    &kat1680_nor,
    &kat240_prt, &kat480_prt, &kat720_prt, &kat960_prt, &kat1200_prt,
    &kat1440_prt, &kat1680_prt, &kat1_prf, &kat1680_prf
};
const size_t drbg_hash_nelem = OSSL_NELEM(drbg_hash_test);
