/* cache.c -- pam_smb cache management code 
   
   This code is (c) Dave Airlie 1998 
   and is released under the GNU GPL
   as provided with this software
   */

#include "config.h"

#include <stdlib.h>

#ifdef USE_DB2
#include <db_185.h>
#else
#include <db.h>
#endif

#include <fcntl.h>
#include <syslog.h>
#include <errno.h>

#include "cache.h"

extern int debug;
extern DB *cache_db;
extern int cur_cache_size;

/****************************************************************************
 Initalise the cache database
****************************************************************************/
int serv_initcachedb()
{
  int type, flags, mode;

  type=DB_HASH;
  flags=O_RDWR;
  mode=0;

  cache_db = dbopen(NULL, flags, mode, type, NULL);

  if (cache_db==NULL)
    {
      perror("db_open failed");
      exit(-1);
    }
}

/****************************************************************************
 Add an entry to the cache
****************************************************************************/
int add_entry_to_cache(usercache_data *entry)
{
  int res;
  DBT key;
  DBT value;
  time_t curtime;

  key.data = entry->ux_user;
  key.size = strlen(entry->ux_user)+1;

  curtime=time(NULL);
  entry->usertimeout=curtime;

  value.data = entry;
  value.size = sizeof(usercache_data);

  res=(cache_db->put)(cache_db, &key, &value, 0);
  if (res==0)
    {
      if (debug)   syslog(LOG_AUTHPRIV | LOG_DEBUG, "db_put: inserted entry\n");
      return 0;
    }
  cur_cache_size++;
  syslog(LOG_AUTHPRIV | LOG_ERR, "error on db_put exiting\n");
  perror("Error on db_put \n");
  sigterm_handler(0);
}

    
/****************************************************************************
 Check if an entry is in the cache
****************************************************************************/
int check_entry_in_cache(usercache_data *entry)
{
  int res;
  DBT key;
  DBT value;
  usercache_data *check;

  key.data = entry->ux_user;
  key.size = strlen(entry->ux_user)+1;

  res=(cache_db->get)(cache_db, &key, &value, 0);
  if (res==0)
    {
      /* Check if password is correct */
      check=value.data;
      if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "db_get: found entry checking passwords");
   
      if (!(strcmp(entry->password, check->password)))
	{
	  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "db_get: found entry return OK\n");
	  return CACHE_OK;
	}
      else
	{
	  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "db_get: password wrong return ENOENT\n");
	  return CACHE_ENOENT;
	}
    }
  else if (res==1)
    {
      if (debug)   syslog(LOG_AUTHPRIV | LOG_DEBUG, "db_get: found no entry return ENOENT\n");
      return CACHE_ENOENT;
    }
  syslog(LOG_AUTHPRIV | LOG_ERR, "error on db_get return ENOENT\n");

  return CACHE_ENOENT;  
}


/****************************************************************************
 Handler for SIGALRM signal, does cache cleanout
****************************************************************************/
void sigalrm_handler(int signum)
{

  /* do cache management */
  int reset=0, res, finished=0;
  DBT key;
  DBT value;
  usercache_data *cacheent;
  time_t curtime;

  syslog(LOG_AUTHPRIV | LOG_NOTICE, "Got SIGALRM: going cleaning cache");


  curtime=time(NULL);
  
  while (!finished)
    {
      res=(cache_db->seq)(cache_db, &key, &value, ((reset==0) ? R_FIRST : R_NEXT));
      reset=1;
      if (res==0) 
	{
	  cacheent=value.data;

	  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache: checking entry %s, %s %d %d\n", key.data, cacheent->ux_user, curtime, cacheent->usertimeout);

	  if (curtime-cacheent->usertimeout > CACHE_TIMEOUT)
	    {
	      res=(cache_db->del)(cache_db, &key, R_CURSOR);
	      if (res==0)
		{
		  if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache: deleted entry %s\n", key.data);
		  cur_cache_size--;
		}
	      else
		if (debug) syslog(LOG_AUTHPRIV | LOG_DEBUG, "cache: error deleting entry or entry did not exist  %d, %d.. Wierd!!!\n", res, errno);
	    }
	 
	}
      else
	finished=1;
    }
  
  return;
}
