/**
 *  Copyright 2009 Opera Software ASA. All rights reserved.
 *
 * This file contains Original Code and/or Contributions to the Original 
 * Code as defined in the Opera Web Applications License (the License). 
 * You may not use this file except in compliance with the License. Please 
 * obtain a copy of the License at http://www.opera.com/license/owal/
 * and read it before using this file. 
 *
 * The Original Code and all Contributions to the Original Code distributed 
 * under the License are distributed on an AS IS basis, WITHOUT WARRANTY 
 * OF ANY KIND, EITHER EXPRESS OR IMPLIED, AND OPERA HEREBY DISCLAIMS ALL 
 * SUCH WARRANTIES, INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT. 
 *
 * Please see the License for the specific language governing rights and 
 * limitations under the License.
 */
 
/**
  * A Dialog library
  * 
  * 
  * Dialog.showMessage(message, options)
  * Dialog.showMessageYesNo(message, options)
  * Dialog.showMessageOkCancel(message, options)
  *
  * message: innerHTML or HTMLElement
  * options: {
  **  title: Title of the Dialog. "Alert", "Error", etc.
  **  callback: The callback function, incase no buttons are provided
  *   
  *-  yes: callback
  *-  no: callback
  *
  *-  ok: callback
  *-  cancel: callback
  * }
  */
var Dialog = {
    defaultOptions:
    {
        buttons: [{ 
            label: 'OK', 
            callback: function(response){ 
                
            }
        }]
    },
    applyDefaults: function(destination, source)
    {
        if ( source == null || typeof(source) != 'object')
        {
            return destination || source;
        }
        destination = destination || ((source instanceof Array) ? [] : {});
        for ( var key in source )
        {
            destination[key] = this.applyDefaults(destination[key], source[key]);
        }
        return destination;
    },
    greyOut: function(turnOff)
    {
        var greyOverlay = document.getElementById("Dialog_greyOverlay");
        if(!greyOverlay)
        {
            greyOverlay = document.createElement('div');
            greyOverlay.id = "Dialog_greyOverlay";
            greyOverlay.style = "position: absolute; top: 0; bottom: 0; left: 0; right: 0; background: black; opacity: 0.5; z-index: 999;";
            document.body.appendChild(greyOverlay);
        }
        greyOverlay.style.display = (turnOff===false)?"none":"block";
    },
    XHR: function(eventData)
    {
        options = {}
        if(eventData.error === true){
          options.title = eventData.title;
          options.buttons = [];
          options.buttons[0] = {};
          options.buttons[0].label = "Cancel".toLocale();
          options.buttons[0].callback = eventData.cancel;
          options.buttons[1] = {};
          options.buttons[1].label = "Retry".toLocale();
          options.buttons[1].callback = eventData.retry;
        } else {
          options.title = eventData.title;
          options.buttons = [];
          options.buttons[0] = {};
          options.buttons[0].label = "Cancel".toLocale();
          options.buttons[0].callback = eventData.cancel;
        }
        message = eventData.message || "<p class='XHR_busy' style='text-align: center; margin: 4px 0;'><img src='data:image/gif;base64,R0lGODlhvgAOAOYAAP///8nJyQEaS5OTk4qOknNycyB05zWU6CqC6DiW5g1RwgQtgQg9ihho3hxu0ESe+GOo5Fqx+HDK+G7D+HDK9TOM7ojN74+vzgszbGrF9DOO6AtIqWe99zSO6FOs9xtZkS+I2tTU1BlqzoXL7kGa9hRIc2ar5KWkpLy7vFel66bD1QUoYRdo3il53MTExI/N+2K67Clxs5GbpDuJtgc7ikaZ6onV+JuirRlgqzCJ7YmKiz6X9mfD9i6G65vX9gg7lhJdyKje90RRYpOSkrbj+MrJyZubm4qOkZWUlAEZRoqPkgEUPMnIyAIeVI2RlWFhYQEaSkN9oZKVl5STk8zLy83NzQEWRFVgbQIbTbKysmbC7wMeSbCytD2T6jOB6juO6n3N+E6s4DON7jeV53Cw53eUv0qP5lWZ7a2trkuZ3h92xRNb0TFlmHDC6z+Z5l+l4mWs7YKjy4nA7QQxhwg5fnu9+Vy26UKf0n7A+mqx8AAAAAAAAAAAAAAAAAAAAAAAACH/C05FVFNDQVBFMi4wAwEAAAAh+QQEAAAAACwAAAAAvgAOAAAH/4AAACFGQ0MDiImKi4yNjo+QkZKTlJWWl5iNSEZoVYIAOk8FOkcEpqeoqaqrrK2ur7CxsrO0tbarSk5SUygAJ08XQT4WxMXGFiYQysvMECYjx8fJzc3P0dHT1MvW18bZ2s7Q3cXf2tzjyODb4ujl1NYjbWFcIQUyREFg+vv7GRkJAAMKDCihoMGDEgYqBEihocOHDhcqhEixocSBFSleFJgR4saAHR9+ZOhQy50oKAqoCGIDTMZ/I3l0hPlRZkiaG23OHJlA50uePivivBiU4lCJPrXM+JBypQ2EBjMcmEq1KlUeUKNa3ToVa1atXK16/SpBatiqY7+aPduVbMG1bP/TZoV7Vi6MGSWa+gAzoa/fCRw6dNBAeHDhwR04/F0cGDHhx4cVL2ZsWLBgDZcJS57sNzDkw5gHb+YM2PBn0xpGc/aMOvRl1ZM9Y4aMGvbdvCpfcIjAu7cHBMCDCw/uobdx3r+HK0dQ/Pjx5MuFN3fuO/rw6dQjQLfOPHt17sCxU99uXTzvFDFwq8Dj4YF79yQMyJ9Pf/6O9/jh198v/37+/PHxV59//70XoID2FYjfgQgaQKCCDCL4YIERCjjhe13EgEFKF+RRgxgVhJhDAyQ2wEKJKDbQQwUgghiiiCmeGGMPYrjIYog1VjDiiTKmSCKNNr5444glsiBjjyri+GLujizu6COKLAB545JNPomkijm62CKMVqK44pZCtvDBCijoEEcNLTigpggKtOnmm26KoOacdLIJ550KyEnnng7YieebevI5p59/thmooH0WCiiigyoaJ6NrOmoooyLgQIcVLmRxAxAbMMAADQuEKuqootLg6amogkrqqguYiuqrn7K6qquwnqqqrKXWmiquo9KqKwO85vqrp8GG6iusdGBghRAhAIDCDRg0gYUA1FZr7bXYZqvtttx26+234IYr7rjkZgtFEktc4cInIbjgAhMBxCvvvPTWa++9+Oar77789uvvvwAHbG8RRVDRLACBAAAh+QQEAAAAACwBAAIAvAAKAAAH/4BPF3JkJhAQJiMWi4yNi4aHkZKIio6NkJOSiZaWmJmHm5yXn5qVoo+kkaGnFp6fq6eukW9pIGgycim6uhS9vr++CcLDxMPAx8XJwsfMysnMyM7E0MDS09TB1svYvdpjalEXeSluwjzcGdrm1Onq59jt2u/s6gkZ6PXz0PHW+sz8ysZomPFBXI0xB3hIWMiwoYQMByJKnChRoUOGEClqTHjRYcaNEy12XPgRZESRI0uaRNlRJUiWF11utDOjxAU4NTpwmMCzp88JHDpo6EBUg9GjQnf+7BnUqFCiT6EqXcq0qFOhQ7NOpQoU69WrSblW9XrUqdGtVINCzYr0rFieTf+9WpUKo+aFFF88RNjLt28EDwgCCx4sWK9fvoAJK0Zg+DDixYobO/4LmbBkx4krB758OLNmzn49V9abIkaJOGdSPFjNuvUDEgZiy54te4fr1rBp6zZg+zbu3bp7+16dG3jt4ayLG+eNnPjy482VG+/dJQaGODjFiKlQQTt37TkasGhAXnx58ix6bOfe/Xv48/DRq2/vvfv29+fHw+/Bnn5//PqVFyB/69n3n4AInsdfewba916A+ZFHIIP9ifFghAqy18IHK8RRQwsOhCiiiCIoYOKJKJ4owogslpjiiwqsyGKLML4o44wk1pjijTg64KKOJvKI449ACjkjkTreKAJCDnRscQMQGzAg5ZRS0rDAlVhmiSUNVHZppZZgLsBll2SGCeaYZFJpppZopsnAl2te2WaacMY5J5l1rtkmHRhsIUQgACH5BAQAAAAALAEAAgC8AAoAAAf/gE8qcmQmEIeIiCYjFo2Oj42GiZMQi5CQkpSKjJeYmomWnY+Zn6GikZ+bp46kmqanhm9pIGgycikpCbq7vBS+v8C/vMO9wcHEyMbKyMTKxszDzsfQu9LA1NXWwrpjamwXeSlu0DzaGdi65c7n6OrS7Nju6+gJ8srw1PbG+OTmvGMacJQAV2PMgYMID/KQwLChQwkZEkpMuPBhw4gTJ1a0eDGjRo4OMXpEuBGkyJEHSnI8OVKlRZYHQODAcAFOjQ4dNGjImZPDhJ9Ag07ggBPnzpw7dXbwKRQo0aRGdR5d2lToU6RTjVKt6hSrV6RbuQ4FK7WoUaZiryY9ChUtV6I8/4/KpJniC4K7eBF4iMC3r98IHvIKzrv3b9/AgwcXNnw4sWLGfhE7xrsYsuTJeiE3xny3MuPLeVvMjHPGi4HTqHc8WM269QMSqGPLNqDaNWvYs2fXtn07t27erXH7Tg2893DixV8fR15c+OwGG1bEMdOigXXrPcRU0F6h+/btOa6LH9+gh3ftYtJXCD+eRQP37st3T89dvRj24lnEfy+/vvr57O1HXnncnefdevxdF99+5m2HnoH4kQdff/VB2J517gHxQxNltLCGAiAqIIIDJJZoogMihKjiiiKeaGKKLLI4oosvxigjjTXaqOKMOKKo4449kgjjjy0GOaSN0W1xwz0GP8yxwAI0MCDllFQyQMOTWGaZZZRVTnmllmBC2WWVX4a55ZhVmqkll2haqeaZbbr55pNsolmmmitsIUQgACH5BAQAAAAALAEAAgC8AAoAAAf/gE8qQT4WhhYmEIqLjBAmI4eRkomNjY+Sk5WWkJiZmouXnZGUn46cooakn6GoiKWKrJIjbWFcN0QUubkJvL2+vbrBwhS/xbzDwsbFyMPKv8zJzsDQutLT1MTWCdhad2yDNmAUGdoJPNjk2ufM6dbr0O3S7+zl88jxzvbD+Mr6wjA4SoCTkOGAwYMID/KQwLChQ4IJIxpc+LBhQYkJKVa0iDHjRocXO078yFHkSJIQTR7Q+DEDCBwYVPgAw6GDhps2b+K0yWGCz59AJ9TMSdSm0Z5Bfw7FqVNDh6cdkCZV+tSp0ao5pU4VmtNqUZ5bqRJ1uvOm1qlDr1bVGTWsz5ov/2O+4ICgrt27dj1E2Mu3bwQPeAPX1euXL2DBeAkXNow48eK+hxsPfsxY8mTKfy1fpny4BUwVdbwYGE269OgdD1KrXv2AhOnXp1mvdg3bNGrZs2vbxp1bN+nbvFv7/h08Ne3hwHkfb7BhxQUzLRpIny6dhfQeFcRUyK69e4Uc1BtYp24d+3bu28WAtz4+fHnt6c+nBx9++ngW2MXozy7/e/X61OXXnX7wqQdgfT3Ax9+C/rnX3nX87VeggdMB8UMTZbSwhgIcduihCA6EKOKIDojg4YkfkjiiiSiiCKKKK7boIowxypgijSGyaCOHL+Ko44490vijAs1ZccMGP8yxwDeSTC5JAwNQRiklAzQ0aWWTT04ZZZVXXpmlllt26SWYU4pp5Zdkcmmmk2SGuSabbVJ55QpWCBEIADs=' style='width: 80%;'></p>";
        this.showMessage(message, options);
    },
    closeMessage: function(){
        Dialog.greyOut(false);
        var panel = document.getElementById("Dialog_panel");
        if(panel) panel.parentNode.removeChild(panel);
    }
    ,
    promptMessage: function(message, options)
    {
        options = options || {};
        options.input = options.input || {};
        options.input.type = options.input.type || 'text';
        
        this.showMessage(message, options);
    },
    getInput: function(options)
    {
        var field;
        var callback;

        switch ( options.type )
        {
            case 'text':
                field = document.createElement('input');
                field.type = 'text';
                field.value = options['default'] || '';
                callback = function()
                {
                    return field.value;
                }
                break;
            case 'textarea':
                field = document.createElement('textarea');
                field.value = options['default'] || '';
                callback = function()
                {
                    return field.value;
                }
                break;
        }

        if ( options.size && options.size.width ) field.style.width = options.size.width+'px';
        if ( options.size && options.size.height ) field.style.height = options.size.height+'px';

        if ( options.position )
        {
            field.style.position = 'absolute';
            field.style.top = options.position.y+'px';
            field.style.left = options.position.x+'px';
        }
        
        field.className = options.className || '';
        this.applyDefaults(field.style, options.style);

        var container = document.createElement('div');
        container.style.width = '100%';
        container.style.paddingLeft = '5px';
        container.style.paddingRight = '5px';
        //container.style.border = '1px solid red';
        container.appendChild(field);

        return {component: container, valueFunction: callback, field: field};
    },
    showMessage: function(message, options)
    {
        var self = this;
        options = this.applyDefaults(options || {}, this.defaultOptions);
        Dialog.closeMessage();
        Dialog.greyOut();
        var panel = document.createElement('div');
        panel.id = "Dialog_panel";
        panel.style = "position: absolute; z-index: 1000; background: transparent url(data:image/png;base64,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) 0 0 repeat-y; padding: 0 5px; -o-background-size: 100%; display: block; width: 90%; margin-top: 10px;"
        if(document.styleSheets)
        {
          document.styleSheets[0].insertRule("#Dialog_panel:after{ background: transparent url(data:image/png;base64,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) 0 0 no-repeat; -o-background-size: 100%; display: block; content: ' '; width: 100%; padding: 0 5px; height: 6px; margin: 0 -5px -3px -5px;}", 0);
        }
        // Title
        options.title = (!options.title)?"Alert".toLocale(): options.title;
        var title_pane = document.createElement('h3');
        title_pane.id = "Dialog_title";
        title_pane.style = "background: transparent url(data:image/png;base64,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) 0 0 no-repeat; -o-background-size: 100%; display: block; width: 100%; padding: 2px 5px; margin: 0 -5px; margin-top: -10px;";
        title_pane.innerHTML = options.title;
        panel.appendChild(title_pane);

        // Message
        var message_pane = document.createElement('div');
        if (message instanceof Element)
        {
            message_pane.appendChild(message);
        }
        else
        {
            message_pane.innerHTML = message;
        }
        message_pane.id = "Dialog_message_pane";
        message_pane.style = "max-height: 150px; overflow: auto;";
        panel.appendChild(message_pane);

        // Input
        var input;
        if ( options.input )
        {
            input = this.getInput(options.input);
            panel.appendChild(input.component);
        }

        // Buttons
        buttons_pane = document.createElement('div');
        buttons_pane.id = "Dialog_buttons_pane";
        buttons_pane.style = "margin-top: 5px; min-height: 40px;"
        if(options.buttons.length == 2)
        {
          buttons_pane.className = "double";
          if(document.styleSheets)
          {
            document.styleSheets[0].insertRule("#Dialog_buttons_pane.double input[type='button']:first-child{ float:left;}", 0);
            document.styleSheets[0].insertRule("#Dialog_buttons_pane.double input[type='button']:last-child{ float:right;}", 0);
            document.styleSheets[0].insertRule("#Dialog_buttons_pane.double:after{ clear: both; display: block; content: ' ';}", 0);
          }
        } else {
          buttons_pane.className = "single";
          buttons_pane.style = "text-align: center;"
        }
        for ( var i = 0; i < options.buttons.length; i++ ) { (function(i)
        {
            var button = options.buttons[i];
            var button_frame = document.createElement('input');
            button_frame.type = "button"
            button_frame.value = button.label;
            //button_frame.innerHTML = button.label;
            button_frame.className = button.className;
            //button_frame.style = "width: 70px; background: transparent url(data:image/png;base64,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) 0 0 no-repeat;"
            button_frame.onclick = function()
            {
                Dialog.closeMessage();
                if ( input )
                {
                    if ( button.callback ) button.callback(i, button.label, input.valueFunction());
                }
                else
                {
                    if ( button.callback ) button.callback(i, button.label);
                }
                if(panel.parentNode) panel.parentNode.removeChild(panel);
            }
            buttons_pane.appendChild(button_frame);
            buttons_pane.appendChild(document.createTextNode(' '));
        }).call(this, i)};
        panel.appendChild(buttons_pane);

        panel.display = 'none';
        document.body.appendChild(panel);
        panel.display = '';
        var greyOverlay = document.getElementById("Dialog_greyOverlay");
        panel.style.top = (greyOverlay.offsetHeight - panel.offsetHeight) / 2;
        panel.style.left = (greyOverlay.offsetWidth - panel.offsetWidth) / 2;
        
        if ( options.input && options.input.focus ) { input.field.focus() }
        if ( options.input && options.input.select ) { input.field.select() }
    },
    showMessageYesNo: function(message, options)
    {
        this.showMessage(message,
        {
            title: options.title,
            buttons: [
                {
                    label: 'Yes'.toLocale(),
                    callback: options.yes
                },
                {
                    label: 'No'.toLocale(),
                    callback: options.no
                }
            ]
        });
    },
    showMessageOkCancel: function(message, options)
    {
        this.showMessage(message,
        {
            title: options.title,
            buttons: [
                {
                    label: 'OK'.toLocale(),
                    callback: options.ok
                },
                {
                    label: 'Cancel'.toLocale(),
                    callback: options.cancel
                }
            ]
        });
    }
}
/**
  *
  *
  */
document.addEventListener("XHR_fetching", function(evt){
    Dialog.XHR( evt.data, function() {
        
    });
}, false);
document.addEventListener("XHR_receiving", function(evt){
    Dialog.XHR( evt.data, function() {
        
    });
}, false);
document.addEventListener("XHR_error", function(evt){
    Dialog.XHR( evt.data, function() {
        
    });
}, false);
document.addEventListener("XHR_complete", function(evt){
    Dialog.closeMessage();
}, false);


